/* install.c  allows user to pick the type of monitor and adaptor in use */
/* and the color of the foreground, background and emphasized letters. */
/* writes values to a file specified on command line, default install.dat */
/*      link with screenf.lib */

/* `MIDI Sequencing In C', Jim Conger, M&T Books, 1989 */

/* #define TURBOC 1   Define if using Turbo C, leave out for Microsoft */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>

#include "standard.h"
#include "screenf.h"
#include "install.h"

/* function declarations */

void init_colors( void );
void put_to_file( void *addr, int size, FILE *stream );
void get_from_file( void *addr, int size, FILE *stream );
void set_text_attrib( int mode );
void set_graph_attrib( int mode );
int load_video_data( char *filename );

/* globals - these variables will normally be globals in a program */

int g_text_mode, g_graph_mode, g_dots_h, g_dots_v, g_let_dots_v, g_let_dots_h,
    g_norm_attrib, g_emph_attrib, g_cursor_attrib, g_text_char_h,g_text_char_v,
    g_graph_char_h,g_graph_char_v,g_text_colors,   g_graph_colors;

void
main(int argc, char *argv[])
{
   int ans, pick, color_let, let_bak, color_emph, emph_bak,
                                                color_cursor, cursor_bak;
   char filename[15];
   FILE *outfile;
   struct strchain *chain;

   g_text_mode = 3;                     /* default values (CGA assumed) */
   g_graph_mode = 6;
   set_text_attrib( g_text_mode );      /* initialize variables */
   set_graph_attrib(g_graph_mode);
   g_norm_attrib = 7;
   g_emph_attrib = 0x0F;
   g_cursor_attrib = 0x70;
   clearscreen(BWC);
   chain = inpchain("install.scr", SCRNWIDE + 1 );
   if (chain == NULL) {
      fputs("\nCould not open file install.scr - not on default disk?",stdout);
      exit(1);
   }

   if (argc > 1)        /* get output file from command line or use default */
      strcpy( filename, argv[1] );
   else
      strcpy( filename, "install.dat");

   ans = load_video_data( filename );
   if (!ans)
      writerr("First time installation - no install file.", 15, BWC, BWC );

   color_let    =    g_norm_attrib & 0x0F;
   let_bak      =   (g_norm_attrib & 0xF0) >> 4;
   color_cursor =  g_cursor_attrib & 0x0F;
   cursor_bak   = (g_cursor_attrib & 0xF0) >> 4;
   color_emph   =    g_emph_attrib & 0x0F;
   emph_bak     =   (g_emph_attrib & 0xF0) >> 4;

   pick = 0;
   while (1) {
      g_norm_attrib   =    (let_bak << 4) + color_let;
      g_cursor_attrib = (cursor_bak << 4) + color_cursor;
      g_emph_attrib =     (emph_bak << 4) + color_emph;
      clearscreen( g_norm_attrib );
      dispchain( chain, 1, g_norm_attrib );
      init_colors();
                                        /* display example lines for attrib */

      writeword("Normal letters will look like:",    21, 20, BWC );
      writeword("These letters.",                    57, 20, g_norm_attrib );
      writeword("Cursor letters will look like:",    21, 21, BWC );
      writeword("These letters.",                    57, 21, g_cursor_attrib );
      writeword("Emphasized letters will look like:",21, 22, BWC );
      writeword("These letters.",                    57, 22, g_emph_attrib );

      write_int( g_text_mode,   7,  6, BRBWC ); /* put numeric values on */
      write_int( g_graph_mode,  7,  9, BRBWC ); /* screen by selection */
      write_int( color_let,    17, 17, BRBWC ); /* menu item */
      write_int( let_bak,      17, 18, BRBWC );
      write_int( color_cursor, 46, 17, BRBWC );
      write_int( cursor_bak,   46, 18, BRBWC );
      write_int( color_emph,   75, 17, BRBWC );
      write_int( emph_bak,     75, 18, BRBWC );

      /* allow cursor movement to select a menu item, pick is item chosen */

      pick = movescrn( g_text_mode, scrndata, pick, NPARAM - 1,
                                     g_norm_attrib, g_cursor_attrib );
      switch (pick) {
      case 0:
         getint( SCRNTALL - 1, "Enter text mode number ->",
                                        &g_text_mode,  2,  7, BWC, BRBWC );
         set_text_attrib( g_text_mode );
         break;
      case 1:
         getint( SCRNTALL - 1, "Enter graphics mode number ->",
                                        &g_graph_mode, 6, 18, BWC, BRBWC );
         set_graph_attrib( g_graph_mode );
         break;
      case 2:
         getint( SCRNTALL - 1, "Enter normal letter attribute number ->",
                                        &color_let,    0, 15, BWC, BRBWC );
         break;
      case 3:
         getint( SCRNTALL - 1, "Enter cursor letter attribute number ->",
                                        &color_cursor, 2, 15, BWC, BRBWC );
         break;
      case 4:
         getint( SCRNTALL - 1, "Enter emphasized letter attribute number ->",
                                        &color_emph,   0, 15, BWC, BRBWC );
         break;
      case 5:
         getint( SCRNTALL - 1, "Enter normal letter background attribute number ->",
                                        &let_bak,      0, 15, BWC, BRBWC );
         break;
      case 6:
         getint( SCRNTALL - 1, "Enter cursor background attribute number ->",
                                        &cursor_bak,   0, 15, BWC, BRBWC );
         break;
      case 7:
         getint( SCRNTALL - 1, "Enter emphasized background attribute number ->",
                                        &emph_bak,     0, 15, BWC, BRBWC );
         break;
      case 8:
         outfile = fopen( filename, "w");
         if (outfile == NULL) {
            fputs("\nCould not open output file ", stderr );
            fputs( filename, stderr );
	    fputc('\n', stderr );
            clearscreen(BWC);
            exit(-1);
         }
         else {
            put_to_file( &g_text_mode,     1, outfile );
            put_to_file( &g_graph_mode,    1, outfile );
            put_to_file( &g_dots_v,        1, outfile );
            put_to_file( &g_dots_h,        1, outfile );
            put_to_file( &g_norm_attrib,   1, outfile );
            put_to_file( &g_cursor_attrib, 1, outfile );
            put_to_file( &g_emph_attrib,   1, outfile );
            put_to_file( &g_let_dots_v,    1, outfile );
            put_to_file( &g_let_dots_h,    1, outfile );
            put_to_file( &g_text_char_h,   1, outfile );
            put_to_file( &g_text_char_v,   1, outfile );
            put_to_file( &g_graph_char_h,  1, outfile );
            put_to_file( &g_graph_char_v,  1, outfile );
            put_to_file( &g_text_colors,   1, outfile );
            put_to_file( &g_graph_colors,  1, outfile );

            fclose(outfile);
         }
         clearscreen(BWC);
         exit(0);
      case -2:                  /* ESC */
      case 9:                   /* quit */
         pick = 0;
         writeword("Don't forget to save data. Quit ? (Y/N)->",1,SCRNTALL,BWC);
         ans = getche();
         if (toupper(ans) != 'Y')
            break;
         else {
            clearscreen(BWC);
            exit(2);
         }
      default:
         writerr("Use arrow keys to move cursor, ret to select.",
                                                SCRNTALL - 1, BWC, BRBWC );
      } /* endswitch */
   }
}


void
init_colors()                   /* show each attribute as a number */
{
   int i, x;
   char str[3];

   for (i = 0; i < 16; i++) {
      itoa( i, str, 10 );
      x = 24 + 3*i;
      writeword( str, x, 13, BWC );
      writeword( str, x, 14, i );
      writeword( str, x, 15, i << 4 );
   }
}


void
set_text_attrib( int mode )     /* set globals based on text mode selected */
{
   switch (mode) {
   case 2:
   case 3:
      g_text_char_h = 80;
      g_text_char_v = 25;
      g_text_colors = 16;
      break;
   case 7:
      g_text_char_h = 80;
      g_text_char_v = 25;
      g_text_colors = 4;
      break;
   default:
      g_text_mode = 3;
      writerr("Text screen must be mode 2, 3, or 7.", SCRNTALL, BWC, BRBWC );
   } /* endswitch */
}


void
set_graph_attrib( int mode )  /* set globals based on graphics mode selected */
{
   char nbuf[10];

   switch (mode) {
   case 4:
   case 5:
      g_dots_h       = 320;
      g_dots_v       = 200;
      g_let_dots_v   = 8;
      g_let_dots_h   = 8;
      g_graph_char_h = 40;
      g_graph_char_v = 25;
      g_graph_colors = 4;
      break;
   case 6:
      g_dots_h       = 640;
      g_dots_v       = 200;
      g_let_dots_v   = 8;
      g_let_dots_h   = 8;
      g_graph_char_h = 80;
      g_graph_char_v = 25;
      g_graph_colors = 2;
      break;
   case 13:
      g_dots_h       = 320;
      g_dots_v       = 200;
      g_let_dots_v   = 8;
      g_let_dots_h   = 8;
      g_graph_char_h = 40;
      g_graph_char_v = 25;
      g_graph_colors = 16;
      break;
   case 14:
      g_dots_h       = 640;
      g_dots_v       = 200;
      g_let_dots_v   = 8;
      g_let_dots_h   = 8;
      g_graph_char_h = 80;
      g_graph_char_v = 25;
      g_graph_colors = 4;
      break;
   case 15:             /* can be either Hercules or EGA */
      writeword("Hercules mono or EGA ? (H/E) ->", 0, SCRNTALL - 1, BWC );
      gets(nbuf);
      if (toupper(*nbuf) == 'H') {
         g_dots_h       = 720;
         g_dots_v       = 348;
         g_let_dots_v   = 14;
         g_let_dots_h   = 9;
         g_graph_char_h = 80;
         g_graph_char_v = 25;
         g_graph_colors = 2;
      }
      else if (toupper(*nbuf) == 'E') {
         g_dots_h       = 640;
         g_dots_v       = 350;
         g_let_dots_v   = 14;
         g_let_dots_h   = 8;
         g_graph_char_h = 80;
         g_graph_char_v = 25;
         g_graph_colors = 4;
      }
      else
         writerr("Must be either E or H.", SCRNTALL - 1, BWC, BRBWC );
      break;
   case 16:
      g_dots_h       = 640;
      g_dots_v       = 350;
      g_let_dots_v   = 14;
      g_let_dots_h   = 8;
      g_graph_char_h = 80;
      g_graph_char_v = 25;
      g_graph_colors = 16;
      break;
   case 17:                 /* VGA modes */
      g_dots_h       = 640;
      g_dots_v       = 480;
      g_let_dots_v   = 16;
      g_let_dots_h   = 8;
      g_graph_char_h = 80;
      g_graph_char_v = 30;
      g_graph_colors = 2;
      break;
   case 18:
      g_dots_h       = 640;
      g_dots_v       = 480;
      g_let_dots_v   = 16;
      g_let_dots_h   = 8;
      g_graph_char_h = 80;
      g_graph_char_v = 30;
      g_graph_colors = 16;
      break;
   case 19:
      g_dots_h       = 320;
      g_dots_v       = 200;
      g_let_dots_v   = 8;
      g_let_dots_h   = 8;
      g_graph_char_h = 40;
      g_graph_char_v = 25;
      g_graph_colors = 256;
      break;
   default:
      writerr("Graphics screen must have a valid mode number.",
                                                SCRNTALL - 1, BWC, BRBWC );
   } /* endswitch */
}


/* this function will also appear in the body of a program that uses */
/* install.c to load it's video parameter data from a file.  Usually */
/* the file is install.dat */
int
load_video_data( char *filename )       /* load visible part of video data */
{                                       /* to global data defined in video.h */
   FILE *infile;

   infile = fopen( filename, "r");
   if (infile == NULL)
      return 0;

   get_from_file( &g_text_mode,     1, infile );
   get_from_file( &g_graph_mode,    1, infile );
   get_from_file( &g_dots_v,        1, infile );
   get_from_file( &g_dots_h,        1, infile );
   get_from_file( &g_norm_attrib,   1, infile );
   get_from_file( &g_cursor_attrib, 1, infile );
   get_from_file( &g_emph_attrib,   1, infile );
   get_from_file( &g_let_dots_v,    1, infile );
   get_from_file( &g_let_dots_h,    1, infile );
   get_from_file( &g_text_char_h,   1, infile );
   get_from_file( &g_text_char_v,   1, infile );
   get_from_file( &g_graph_char_h,  1, infile );
   get_from_file( &g_graph_char_v,  1, infile );
   get_from_file( &g_text_colors,   1, infile );
   get_from_file( &g_graph_colors,  1, infile );

   fclose(infile);
   return 1;
}


void                                        /* put near int data to stream */
put_to_file( void *addr, int size, FILE *stream )
{
   int i, end;
   char *caddr;

   caddr = (char *)addr;
   end = size * sizeof(int);
   for (i = 0; i < end; i++)
      fputc( *caddr++, stream );
}

                                            /* get int data from stream, */
void                                        /* put into near memory */
get_from_file( void *addr, int size, FILE *stream )
{
   int i, end;
   char *caddr;

   caddr = (char *)addr;
   end = size * sizeof(int);
   for (i = 0; i < end; i++)
      *caddr++ = (char)fgetc( stream );
}
