/*
 File: TIMER.C

 Copyright 1997 by Jake Stine
 
 Description:
   This is a simple timer for DOS, that can operate at any single given
   frequency rate (usually 60Hz).  This timer is Windows95 dos-box friendly.

 Notes:
  - The handler passed to Timer_Init must be prepared to be called from an
    interrupt.  ie, Watcom C users must compile with the /zu option enabled.

  - This timer module should NEVER be used in conjunction with VTIMER.  Both
    are designed to solo in a program, and trying to run both will result in
    strange behavior at best.
   
*/


#include <stdio.h>
#include <conio.h>
#include <dos.h>
#include <stdlib.h>
#include "timer.h"
#include "mmio.h"


volatile int framecount = 0;

void framecounter(void)
{
   framecount++;
}


static BOOL  timer_initialized = 0, timer_active = 0;
static UWORD timer_rate;
static int   fcnt, timeskip;
static void (*timer_func)(void) = NULL;
static void (interrupt far *timer_oldhandler)(void);

#ifdef __WATCOMC__
#pragma aux timer_oldhandler parm nomemory modify nomemory;
#pragma aux timer_func       parm nomemory modify nomemory;
#endif


static void starttimer1(UWORD speed)
{
    outp(0x43,0x34);
    outp(0x40,speed&0xff);
    outp(0x40,speed>>8);
}


static void interrupt timer_handler(void)
{
    static int countdown = 1, timedown = 0, i;

#ifdef __BORLANDC__
    pushad();                       // borland interrupt fix
#endif

   if(timer_active)
   {   countdown--;
       if(countdown == 0)
       {   timer_func();
           countdown = fcnt;
       }
   }

   if(timer_oldhandler!=NULL)
   {   timedown += timeskip;
       for(i=timedown/256; i; i--) timer_oldhandler();
       timedown &= 255;
   }
   
   outp(0x20,0x20);

#ifdef __BORLANDC__
    popad();                       // borland interrupt fix
#endif

}


void Timer_Start(void)
{
    timer_active = 1;
}


void Timer_Stop(void)
{
    timer_active = 0;
}
    

void Timer_Exit(void)
{
    Timer_Stop();
    if(timer_initialized)
    {    disable();
         outp(0x43,0x34);
         outp(0x40,0);
         outp(0x40,0);
         _dos_setvect(0x8,timer_oldhandler);
         enable();
         timer_initialized = 0;
    }
}


void Timer_Init(int rate, void (*handler)(void))
// Initialize the DOS-based hardware timer.  'rate' is measured in 1/100th
// Hz - or 1820 is 18.2Hz.
{
    SLONG tmp;
    int   i;

    Timer_Stop();
    timer_func = handler;

    // set up the replay framerate.  Framerates below 18.18Hz have to be
    // 'fixed' because the interrupt handler cannot go any slower.

    i = rate;  fcnt = 1;
    while(rate < 1818)
    {   rate += i;
        fcnt++;
    }

    tmp = ((119138100L) / rate);
    if(tmp > 0xFFFF) timer_rate = 0; else timer_rate = tmp;

    timeskip = (1820<<8) / rate;

    if(!timer_initialized)
    {   timer_oldhandler = _dos_getvect(0x8);
        _dos_setvect(0x8,timer_handler);
        starttimer1(timer_rate);
        timer_initialized = 1;
    }
}
                       

void Timer_SetSpeed(UWORD rate)
// rate is specificed in 1/100 Hz (ex, 1820 = 18.2 Hz)
{
    SLONG tmp,i;

    // set up the replay framerate.  Framerates below 18.18Hz have to be
    // 'fixed' because the interrupt handler cannot go any slower.

    i = rate;  fcnt = 1;
    while(rate < 1818)
    {   rate += i;
        fcnt++;
    }

    tmp = ((119138100L) / rate);
    if(tmp > 0xFFFF) timer_rate = 0; else timer_rate = tmp;

    timeskip = (1820<<8) / rate;
    starttimer1(timer_rate);    
}

#ifdef DJGPP
void _dos_setvect(unsigned vecno, PVI handler)
{
	_go32_dpmi_seginfo seginfo;
	PVI oldvect;

	_go32_dpmi_get_protected_mode_interrupt_vector(vecno, &seginfo);
    seginfo.pm_offset = (ULONG) handler;
	seginfo.pm_selector = _go32_my_cs();
	_go32_dpmi_allocate_iret_wrapper(&seginfo);
	_go32_dpmi_set_protected_mode_interrupt_vector(vecno, &seginfo);
    return;
}

PVI _dos_getvect(unsigned vecno)
{
	_go32_dpmi_seginfo seginfo;

	_go32_dpmi_get_protected_mode_interrupt_vector(vecno, &seginfo);
    return (PVI) seginfo.pm_offset;
}
#endif
