/*
  --> MikMod 3.0
   -> Release Date 09/20/97
   
  This example demonstrates how to create a simple module player.  If you
  are interested in a more advanced interface (such as a game) which would
  require the use of sound effects, see "MIKSOUND.C"

  This example file uses a 'polling' method of sound update.  This method
  is very portable, and is Windows friendly, but is not usually the best
  way to handle sound updating in many DOS applications.  See MIKINTER.C
  for the interrupt driven example of a simple module player.

  This version of the MikMod 3.0 library is still in the initial stages,
  meaning that it still has some sucky "features" (yes, bugs :), but it's
  pretty functional otherwise.  New releases should be made periodically.

  For more information concerning the use of MikMod 3.0, see the documen-
  tation in the /DOCS directory.

---------

  Keep a look out for updates and new betas!

  mailto:dracoirs@epix.net
  http://www.epix.net/~dracoirs/
 
*/

#include "mikmod.h"
#include "getopt.h"
#include "timer.h"
#include <conio.h>

#define ESCAPE     27

#define F1         59
#define F2         60
#define F3         61
#define F4         62
#define F5         63
#define F6         64
#define F7         65
#define F8         66
#define F9         67
#define F10        68


#define LIST_LOADERS  1
#define LIST_DRIVERS  2


// strings are defined in message.c
extern CHAR banner[], helptext01[], helptext02[], presskey[];


// Globals that would likely be used outside the main module

int    cursong;
UNIMOD *mf;               // module handle / format information.  See
                          // "mikmod.h" for a description of its contents.


// Static Module Variables.
// ------------------------
// The following variables can be modified via command line options.

static BOOL  morehelp     = 0,
             m15_enabled  = 0,     // set = registers 15-inst module loader
             emu_enabled  = 1;     // set = display mod status line

static int   ldlist       = 0;     // List Loaders / Drivers?
static int   cfg_maxchn   = 64;    // max channels that can be allocated by a module

static BOOL  cfg_extspd  = 1,      // Extended Speed enable
             cfg_panning = 1,      // DMP panning enable (8xx effects)
             cfg_loop    = 1;      // auto song-looping enable


// Emumerated list of command-line options for convienience.
// Note that these are enumerated in the SAME order as the options are
// listed in the P_OPTION structure below!

enum
{
    CMD_REVPAN = 0,
    CMD_REVERB,
    CMD_PANSEP,
    CMD_NOEMU,
    CMD_SOFTMIX,
    CMD_DEVICE,
    CMD_MAXCHN,
    CMD_VOLUME,
    CMD_LDEVICES,
    CMD_LLOADERS,
    CMD_REPEAT,
    CMD_STEREO,
    CMD_16BIT,
    CMD_EXTENDED,
    CMD_PANNING,
    CMD_FREQ,
    CMD_15INST,
    CMD_HELP1,
    CMD_HELP2,
    CMD_TOTAL
};


// -----------------------------------------------------------------------
// Special GetOpt information structure.  The first part is the option, and
// the second part is the parameter type.
//   P_BOOLEAN  = checks for a '+' or '-' after the option, if none found,
//                '+' is assumed.
//
//   P_NUMVALUE = Checks for a numberafter the option, whitespace between
//                is ignored (ie, /d5 and /d 5 are both acceptable)
//
//   P_STRING  =  Checks for a string (like a filename following an option)
//                ie, "/sng roberto.it"  would return "roberto.it"
//
// NOTE: Options should ALWAYS be listed in order from longest to shortest!
//       Otherwise, the parser will mistake the first letter or letters of
//       a longer option as being a shorter one it checks for first!

P_OPTION mikoption[CMD_TOTAL] =
{
   { "revpan", P_BOOLEAN  },
   { "reverb", P_NUMVALUE },
   { "pansep", P_NUMVALUE },
   { "noemu", P_BOOLEAN },
   { "soft", P_BOOLEAN },
   { "d",  P_NUMVALUE },
   { "c",  P_NUMVALUE },
   { "v",  P_NUMVALUE },
   { "ld", P_BOOLEAN  },
   { "ll", P_BOOLEAN  },
   { "r",  P_BOOLEAN  },
   { "m",  P_BOOLEAN  },
   { "8",  P_BOOLEAN  },
   { "x",  P_BOOLEAN  },
   { "p",  P_BOOLEAN  },
   { "f",  P_NUMVALUE },
   { "15i",P_BOOLEAN  },
   { "h",  P_BOOLEAN  },
   { "?",  P_BOOLEAN  }
};

P_PARSE mikparse = { CMD_TOTAL, mikoption };

void mikcommand(int index, P_VALUE *unival)
{
    SLONG value = unival->number;

    switch(index)
    {
        case CMD_REVERB:                       // 'reverb' - set reverb
            if(value > 16) value = 16;
            md_reverb = value;
        break;

        case CMD_NOEMU:
            if(value == -1) emu_enabled = 0; else emu_enabled = value;
        break;

        case CMD_SOFTMIX:                      // 'soft' - enable / disable software mixer
            if(value)
                md_mode |= DMODE_SOFT_MUSIC;
            else
                md_mode &= ~DMODE_SOFT_MUSIC;
        break;

        case CMD_DEVICE:                       // 'd' - select device
            md_device = value;
        break;

        case CMD_MAXCHN:                       // 'c' - set max number of channels
            cfg_maxchn = value;
            if(cfg_maxchn > 255) cfg_maxchn = 255;
            if(cfg_maxchn < 2) cfg_maxchn = 2;
        break;

        case CMD_VOLUME:                       // 'v' - set volume
            if(value > 100) value = 100;
            md_volume = (value*128) / 100;
        break;

        case CMD_LDEVICES:                     // 'ld' - list device drivers avail
            ldlist = LIST_DRIVERS;
        break;

        case CMD_LLOADERS:                     // 'll' - list song loaders avail
            ldlist = LIST_LOADERS;
        break;

        case CMD_REPEAT:                       // 'r' - enable / disable song looping
            if(value==-1) cfg_loop=1; else cfg_loop=value;                 
        break;

        case CMD_STEREO:                       // 'm' - enable / disable stereo
            if((value==-1) || (value==0))
                md_mode &= ~DMODE_STEREO;
            else
                md_mode |= DMODE_STEREO;
        break;

        case CMD_16BIT:                        // '8' - enable / disable 16 bit sound
            if(value)
                md_mode &= ~DMODE_16BITS;
            else
                md_mode |= DMODE_16BITS;
        break;

        case CMD_EXTENDED:                     // 'x' - extended speed
            cfg_extspd = 0;
        break;

        case CMD_PANSEP:                       // 'pansep' - panning separation percentage
            md_pansep = value;
        break;

        case CMD_PANNING:                      // 'p' - panning
            if(value==-1) cfg_panning = 1; else cfg_panning = value;
        break;

        case CMD_REVPAN:                      // 'revpan' - reverse stereo
            if((value==-1) || (value==1))
                md_mode |= DMODE_REVERSE;
            else
                md_mode &= ~DMODE_REVERSE;
        break;

        case CMD_FREQ:                         // 'f' - frequency set
            md_mixfreq = value;
        break;

        case CMD_15INST:                        // enable / disable 15inst loader
            if(value==-1) m15_enabled = 1;  else m15_enabled = value;
        break;
       
        case CMD_HELP1:                         // 'h' - help and stuff
        case CMD_HELP2:                         // '?' - help and stuff
            morehelp = 1;
        break;
    }
}


void errorhandler(void)

// When called, the following global variables will have been set:
//  _mm_errno     - INTEGER; the error that occured [listed in mmio.h]
//  _mm_critical  - BOOLEAN; set to TRUE if an error occurs within
//                  MikMod_Init() or when trying to start playing of a
//                  module - in which case MikMod automatically shuts
//                  down and impliments the NoSound driver.

{
    printf("MikMod Error: %s\n\n",_mm_errmsg[_mm_errno]);

    if(_mm_critical)
    {   exit(_mm_errno);     // Mod player not much use without sound, so
                             // just exit the program.
    }
}


int main(int argc, CHAR *argv[])
{
   FILESTACK *fstack;  // linked list of the module playlist
   BOOL  quit = 0,     // set true when time to exit modplayer (escape pressed)
         next;         // set true when time to play next song (space pressed)
   int   nfiles;


   // Register the MikMod error handler.  If any errors occur within the
   // MikMod libraries (including the ngetopt and all _mm_ functions),
   // the given error handler will be called and _mm_errno set. [see
   // errorhandler(void) above]

   MikMod_RegisterErrorHandler(errorhandler);


   // Initialize soundcard parameters.
   // --------------------------------
   // you _have_ to do this before calling MikMod_Init().  Changing them
   // afterward will result in unpredictable behavior, so don't do it!  You
   // can change these values if you call MikMod_Exit(); and then
   // MikMod_Init() again.
   // Note that the values below are the default assignments as found in
   // MDRIVER.C, and are unneccessary here if you plan to use the defaults
   // only.

   md_mixfreq      = 44100;            // standard mixing freq
   md_dmabufsize   = 4000;             // DMA buffer size of 4 seconds (default = 50 millseconds)
   md_device       = 0;                // standard device: autodetect
   md_volume       = 96;               // driver volume (max 128)
   md_musicvolume  = 128;              // music volume (max 128)
   md_sndfxvolume  = 128;              // sound effects volume (max 128)
   md_pansep       = 128;              // panning separation (0 = mono, 128 = full stereo)
   md_reverb       = 6;                // Reverb (max 15)


   // Device Mixing Mode
   // ------------------
   // Available supported options (more in the next beta):
   //
   //   DMODE_STEREO       - Enables stereo output
   //   DMODE_16BITS       - Enables 16 bits output (see VIRTCH.C for enabling
   //                        16 bit software mixing)
   //   DMODE_REVERSE      - Turns on Reverse Stereo (Left<->Right)
   //   DMODE_SOFT_SNDFX   - Force software mixing of sound effects
   //   DMODE_SOFT_MUSIC   - Force software mixing of music
   
   // md_mode = DMODE_16BITS | DMODE_STEREO | DMODE_SURROUND;

   puts(banner);

   // The new GetOpt System
   // ---------------------
   // ex_init(CHAR *defpath, CHAR *filemask, int sort);
   //
   //  Returns TRUE on error.
   //
   //  defpath   -  The default path to check if no files were found on the command
   //               line.  If ANY files are specified, this option is ignored.
   //  filemask  -  The default filemask to use if none given.  ie, 'c:\songs\' needs
   //               a filemask attached to mean anything.
   //  sort      -  File sorting method.  Availible Options:
   //                 EX_FULLSORT  - Sort list into directories, then by name.
   //                 EX_FILESORT  - Sort list by filenames only - ignore directories.
   
   ex_init(NULL, "*.*", EX_FULLSORT);

   // grab options off the command-line and return the number of files after
   // wildcard expansion.

   nfiles = ngetopt("/",&mikparse, argc, argv, mikcommand);
   ex_exit();            // all done parsing options, so clean up.
     

   // ==========================================
   //    MikMod Loader / Driver Registration!
   // ==========================================

   MikMod_RegisterAllLoaders();
   if(m15_enabled) MikMod_RegisterLoader(load_m15);    // if you use m15load, register it as last!

   MikMod_RegisterAllDrivers();
   MikMod_RegisterDriver(drv_wav);     // raw driver - dumps output to a raw file
   MikMod_RegisterDriver(drv_raw);     // raw driver - dumps output to a raw file


   // =============================================
   // Check for a bad commandline or help request.
   // Done after registering the loaders / drivers, otherwise the /ll
   // and /ld commands would not work.

   if((nfiles==0) || morehelp || ldlist)
   {   // there was an error in the commandline, or there were no true
       // arguments, so display a usage message

       puts("Usage: MIKMOD [switches] <clokwork.mod> <children.mod> ... \n"
            "       MIKMOD [switches] <*.*> <*.s3m> <children.mod> ...\n");

       if(ldlist)
       {   switch(ldlist)
           {   case LIST_LOADERS:
                   puts("\nAvailible module loaders:\n");
                   ML_InfoLoader();
               break;
               
               case LIST_DRIVERS:
                   puts("\nAvailible soundcard drivers:\n");
                   MD_InfoDriver();
               break;
           }
       } else if(morehelp)
       {   puts(helptext01);
           puts(presskey);
           if(getch() == ESCAPE) exit(1);
           puts(helptext02);
       } else
       {   puts("Type MIKMOD /h for more help.\n");
       }
       exit(1);
    }


    // ===================================================================
    // Initialize MikMod (initializes soundcard and associated mixers, etc)
    // NOTE: Errors here are now handled by the error handler above.  How-
    //       ever, MikMod_Init() still returns the boolean value TRUE on
    //       error.

    MikMod_Init();

    printf("Using %s for %d bit %s %s sound at %u Hz\n\n",
           md_driver->Name,
           (md_mode&DMODE_16BITS) ? 16:8,
           (md_mode&DMODE_INTERP) ? "interpolated" : "normal",
           (md_mode&DMODE_STEREO) ? "stereo" : "mono",
           md_mixfreq);

#ifdef __OS2__
    DosSetPriority( PRTYS_THREAD, PRTYC_TIMECRITICAL, 5L, 0UL );
#endif

    fstack = filestack;
    for(cursong = 0; (cursong < nfiles) && (fstack != NULL) && !quit; cursong++, fstack=fstack->next)
    {   printf("File    : %s\n",fstack->path);

        // To load a module, pass the module name, and the MAXIMUM number
        // of channels that is allowed to be allocated.  MikMod_LoadSong will
        // allocate the number of channels needed (dependant on module type,
        // as non-IT formats do not require more than pf->numchn channels).

        // Note that if an error occurs, it is handled by the error handler.
        // Any critical errors will exit the player, other errors can simply
        // skip the song.

        if((mf=MikMod_LoadSong(fstack->path,cfg_maxchn)) == NULL) continue;

        // Set up the module playback information.
        // These values can be changed at any time!
        // See MIKMOD.TXT or MIKMOD.H for a description of the various
        // elements in the UNIMOD structure.

        mf->extspd  = cfg_extspd;
        mf->panflag = cfg_panning;
        mf->loop    = cfg_loop;

        printf("Songname: %s\n"
               "Modtype : %s\n"
               "Periods : %s,%s\n"
               "Channels: %d (song), %d (mixed)\n",
               mf->songname,
               mf->modtype,
               (mf->flags & UF_XMPERIODS) ? "XM type" : "mod type",
               (mf->flags & UF_LINEAR) ? "Linear" : "Log",
               mf->numchn,md_numchn);


        Player_Start(mf);      // start playing the module
        next = 0;

        while(Player_Active() && !next)
        {   CHAR c;

            if(kbhit())
            {   c = getch();
                switch(c)
                {   case 0 :     // if first getch==0, get extended keycode
                        c = getch();
                        switch(c)
                        {   case  F1  :    // F1 thru F10:
                            case  F2  :    // Set song volume by increments
                            case  F3  :    // of 10.  F1 = 10%, F2 = 20%, etc.
                            case  F4  :    // Maximum volume is 128.
                            case  F5  :
                            case  F6  :
                            case  F7  :
                            case  F8  :
                            case  F9  :
                            case  F10 :
                            {   // note volume scale ...
                                //   md_volume = (percentage * 128) / 100

                                int t = (c == F1) ? 13 : (((c%10)*1280) / 100) + 26;
                                if(t > 128)
                                   md_volume = 128;
                                else
                                   md_volume = t;
                            }
                            break;
                        }
                    break;


                    case '1':
                    case '2':
                    case '3':
                    case '4':
                    case '5':
                    case '6':
                    case '7':
                    case '8':
                    case '9':
                        MP_ToggleMute(mf,MUTE_INCLUSIVE,1,c-50);
                        MP_ToggleMute(mf,MUTE_INCLUSIVE,c-48,mf->numchn);
                    break;
        
    
                    case '+' :  Player_NextPosition();  break;
                    case '-' :  Player_PrevPosition();  break;
                    case 'P' :
                    case 'p' :  Player_TogglePause();   break;

                    case ESCAPE :  quit = 1;
                    case ' ' :     next = 1;          break;
                }
            }

            MikMod_Update();

            // wait a bit, because calling MD_Update too rapidly can cause problems
            // on some soundcards.

            delay(10);

            if(emu_enabled)
            {   printf("\rsngpos:%d patpos:%d sngspd %d bpm %d   ",
                            mf->sngpos,mf->patpos,mf->sngspd,mf->bpm);
                #ifdef DJGPP
                fflush(stdout);
                #endif
            }
        }

        Player_Stop();      // stop playing.
        MikMod_FreeSong(mf);    // and free the module
        puts("\n");
    }

    MikMod_Exit();       // deinitialize MikMod and sound card hardware

    return 0;
}

