/*

 File: MIKINTER.C

 Description:
  Demonstrates a simple module player that is interrupt based, rather than
  polling based [see MIKMOD.C for polling based].

  Notice in particular lines # 96 - 100

 Notes:
  - Interrupt-based playback does NOT work in the background of Windows95.

  - drv_raw (RAW file dump driver) will NOT work as interrupt driven!

  - Remember that if you call your own procedure, and noth MikMod_Update(),
    you have to use the /zu copiler option on any modules containing code
    that would be run from the interrupt (watcom).

*/

#include <conio.h>
#include "mikmod.h"
#include "timer.h"
#include "getopt.h"


#define ESCAPE     27

#define LIST_LOADERS  1
#define LIST_DRIVERS  2



UNIMOD  *mf;

void errorhandler(void)

// When called, the following global variables will have been set:
//  _mm_errno     - INTEGER; the error that occured [listed in mmio.h]
//  _mm_critical  - BOOLEAN; if true, then the error is basically non-re-
//                  coverable and mikmod automatically defaults back to
//                  the nosound driver.

{
    printf("\nMikMod Error : %s\n",_mm_errmsg[_mm_errno]);

    if(_mm_critical)
    {   exit(_mm_errno);     // Mod player not much use without sound, so
                             // just exit the program.
    }
}


void main(int argc, char *argv[])
{
    FILESTACK *fstack;  // linked list of the module playlist
    BOOL  quit = 0,     // set true when time to exit modplayer (escape pressed)
          next;         // set true when time to play next song (space pressed)


    puts("\n"
         "MikMod 3.0 - Interrupt Driven Module Player\n"
         "===========================================");

    // Register the MikMod error handler.  If any errors occur within the
    // MikMod libraries (including the ngetopt and all _mm_ functions),
    // the given error handler will be called and _mm_errno set. [see
    // errorhandler(void) above]

    MikMod_RegisterErrorHandler(errorhandler);


   // grab options off the command-line.

    ex_init(NULL, "*.*", EX_FULLSORT);
    ngetopt(NULL,NULL, argc, argv, NULL);
    ex_exit();


    // =======================================
    // Register all std loaders and drivers for use.
    //  - Remember that drv_raw CANNOT be used if mikmod is
    //    run by interrupt!

    MikMod_RegisterAllLoaders();
    MikMod_RegisterAllDrivers();


    if(argc==1)
    {   // there was an error in the commandline, or there were no true
        // arguments, so display a usage message

        puts("Usage: MIKINTER [switches] <clokwork.mod> <children.mod> ... \n"
             "       MIKINTER [switches] <*.*> <*.s3m> <children.mod> ...\n");

        exit(1);
    }


    // Tie in MikMod_Update() to the System Timer (interrupt 8).
    // I use the timer routines I package with MikMod to do the task here.
   
    Timer_Init(6000,MikMod_Update);    // Init the timer to run at 60Hz
    Timer_Start();


    // ===================================================================
    // Initialize MikMod (initializes soundcard and associated mixers, etc)
    // NOTE: Errors here are now handled by the error handler above.  How-
    //       ever, MikMod_Init() still returns the boolean value TRUE on
    //       error.

    MikMod_Init();

    printf("Using %s for %d bit %s %s sound at %u Hz\n\n",
           md_driver->Name,
           (md_mode&DMODE_16BITS) ? 16:8,
           (md_mode&DMODE_INTERP) ? "interpolated" : "normal",
           (md_mode&DMODE_STEREO) ? "stereo" : "mono",
           md_mixfreq);

#ifdef __OS2__
    DosSetPriority( PRTYS_THREAD, PRTYC_TIMECRITICAL, 5L, 0UL );
#endif

    fstack = filestack;
    for(; (fstack!=NULL) && !quit; fstack=fstack->next)
    {   printf("File    : %s\n",fstack->path);

        // MikMod_LoadSong - Default to a maximum of 64 channels (all music,
        // none sound effects).  To allocate room for sound effects, see
        // "MIKSOUND.C" or MIKMOD.TXT reference on "ML_LoadFP()"

        // Note that if an error occurs, it is handled by the error handler.
        // Any critical errors will exit the player, other errors can simply
        // skip the song.

        if((mf=MikMod_LoadSong(fstack->path,64)) == NULL) continue;

        printf("Songname: %s\n"
               "Modtype : %s\n"
               "Periods : %s,%s\n"
               "Channels: %d (song), %d (mixed)\n",
               mf->songname,
               mf->modtype,
               (mf->flags&UF_XMPERIODS) ? "XM type" : "mod type",
               (mf->flags&UF_LINEAR) ? "Linear" : "Log",
               mf->numchn,md_numchn);

        // start playing the module:

        Player_Start(mf);

        next = 0;

        while(Player_Active() && !next)
        {   CHAR c;

            if(kbhit())
            {   c = getch();
                switch(c)
                {   case '+' :  Player_NextPosition();  break;
                    case '-' :  Player_PrevPosition();  break;
                    case 'P' :
                    case 'p' :  Player_TogglePause();   break;

                    case ESCAPE :  quit = 1;
                    case ' ' :     next = 1;          break;
                }
            }

            printf("\rsngpos:%d patpos:%d sngspd %d bpm %d   ",
                        mf->sngpos,mf->patpos,mf->sngspd,mf->bpm);
            #ifdef DJGPP
            fflush(stdout);
            #endif
        }

        Player_Stop();          // stop playing.
        MikMod_FreeSong(mf);    // and free the module
        puts("\n");
    }

    Timer_Exit();        // Stop the timer before exit (very bad not to!)
    MikMod_Exit();       // deinitialize MikMod and sound card hardware
}

