/*

 --> Mikmod UniFormat Conversion Utility
  -> Version 3.0 -- Released August 20th, 1997.

FileName: MIKCVT.C

Description:
Program to convert any supported module into a UniMOD [.UNI] module.
If compiled and run under DOS/Windows, this MIKCVT.C does not sup-
port wildcards.  This feature is intentionally left out for now to
maintain easy portability to the Unix platform.

DOS users can add wildcard support themselves by following the ngetopt
example in MIKMOD.C.


Portability:
All systems - all compilers

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "mikmod.h"


//  Declare external loaders:


static FILE   *fpi, *fpo;
static UBYTE  unitag[]  = "UN06\x1A";


/***************************************************************************
***************************************************************************/

BOOL TrkCmp(UBYTE * t1, UBYTE * t2)
{
    UWORD l1, l2;

    if((t1 == NULL) || (t2 == NULL))
    return 0;

    l1 = TrkLen(t1);
    l2 = TrkLen(t2);

    if (l1 != l2)
    return 0;

    return (MyCmp(t1, t2, l1));
}



void ReplaceTrack(UNIMOD * mf, int t1, int t2)
{
    int t;

    for(t=0; t<mf->numpat*mf->numchn; t++)
    {   if(mf->patterns[t] == t1)
             mf->patterns[t] = t2;
    }
}



void OptimizeTracks(UNIMOD * mf)

//  Optimizes the number of tracks in a modfile by removing tracks with
//  identical contents.

{
    int   t, u, same, newcnt = 0;
    UBYTE *ta;
    UBYTE **newtrk;

    if (!(newtrk = (UBYTE **)malloc(mf->numtrk * sizeof(UBYTE *)))) return;

    for(t=0; t<mf->numtrk; t++)
    {   // ta is track to examine

        ta = mf->tracks[t];

        // does ta look familiar ?

        for(same = u = 0; u<newcnt; u++)
        {   if(TrkCmp(ta, newtrk[u]))
            {   same = 1;
                break;
            }
        }

        if(same)
        {   ReplaceTrack(mf, t, u);
        } else
        {   ReplaceTrack(mf, t, newcnt);
            newtrk[newcnt++] = ta;
        }

        printf("\rOptimizing: %d\%", (t * 100L) / mf->numtrk);
    }

    printf("\rOptimized : %d tracks -> %d tracks\n", mf->numtrk, newcnt);

    free(mf->tracks);
    mf->tracks = newtrk;
    mf->numtrk = newcnt;
}


/***************************************************************************
***************************************************************************/

CHAR *stripname(CHAR *path, CHAR *ext)

// Strips the filename from a path, and replaces or adds
// a new extension to it.

{
        CHAR *n, *m;
        static CHAR newname[256];

        // extract the filename from the path

#ifdef unix
        n = ((n = strrchr(path, '/')) == NULL) ? path : n + 1;
#else
        n = ((n = strrchr(path, '\\')) == NULL) ? path : n + 1;
        if(m = strrchr(n, ':')) n=m+1;
#endif

        // copy the filename into 'newname'
        strncpy(newname,n,255);
        newname[255] = 0;

        // remove the extension
        if (n = strrchr(newname, '.'))
        *n = 0;

        // and tack on the new extension
        return strcat(newname, ext);
}


/***************************************************************************
***************************************************************************/

int main(int argc, char *argv[])
{
    UNIMOD     *mf;
    SAMPLE     *s;
    INSTRUMENT *i;
    int        t, v, w, strip = 0, numsmp, numins;
    CHAR       *outname;

    

    puts("\n"
         "MIKCVT v3.0 - MikMod UniFormat Conversion Utility\n"
         "=================================================\n");
         

    // Register the NoSound driver only - the modules will not be played.

    MikMod_RegisterDriver(drv_nos);
    MikMod_RegisterAllLoaders();

    MD_Init();        // nosound driver HAS to initialize!


    if(argc < 2)
    {   // display a usage message

        puts("Usage: MIKCVT [-strip] <fletch.mod> ...\n\n"
             "-strip: Leave all text out from the UNI file.\n\n");

        exit(-1);
    }

    for (t=1; t<argc; t++)
    {   if((t == 1) && (!strcmp(argv[1], "-strip")))
        {
            strip = 1;
            puts("Stripping all text.\n");

            continue;
        }

        // Open the first file (source module)

        printf("In file : %s\n", argv[t]);
        if((fpi = fopen(argv[t], "rb")) == NULL)
        {   printf("MikCvt Error: Error opening input file\n");
            break;
        }

        // Open the desination module

        outname = stripname(argv[t], ".uni");
        printf("Out file: %s\n", outname);
        if ((fpo = fopen(outname, "wb")) == NULL)
        {   printf("MikCvt Error: Error opening output file\n");
            break;
        }

        // load the module .. maxchan of 0, because we won't be playing
        // anything.

        mf = ML_LoadFP(fpi,0);

        // didn't work -> exit with error

        if (mf == NULL)
        {   printf("MikCvt Error: %s\n", _mm_errmsg[_mm_errno]);
            fclose(fpi);
            break;
        }

        printf("Songname: %s\n"
               "Modtype : %s\n",
               mf->songname,
               mf->modtype);

        numsmp = mf->numsmp;
        numins = mf->numins;
        OptimizeTracks(mf);


        // Write UNI header
        // UNI format version 3.00 (#6)

        _mm_write_UBYTES(unitag, 5, fpo);

        _mm_write_M_UWORD(mf->flags, fpo);
        _mm_write_UBYTE(mf->numchn, fpo);
        _mm_write_UBYTE(mf->numvoices, fpo);
        _mm_write_M_UWORD(mf->numpos, fpo);
        _mm_write_M_UWORD(mf->numpat, fpo);
        _mm_write_M_UWORD(mf->numtrk, fpo);
        _mm_write_M_UWORD(numins, fpo);
        _mm_write_M_UWORD(numsmp, fpo);
        _mm_write_M_UWORD(mf->reppos, fpo);
        _mm_write_UBYTE(mf->initspeed, fpo);
        _mm_write_UBYTE(mf->inittempo, fpo);
        _mm_write_UBYTE(mf->initvolume, fpo);

        StringWrite(strip ? NULL : mf->songname, fpo);
        StringWrite(strip ? NULL : mf->composer, fpo);
        StringWrite(strip ? NULL : mf->comment, fpo);

        _mm_write_UBYTES(mf->positions, mf->numpos, fpo);
        _mm_write_M_UWORDS(mf->panning, mf->numchn, fpo);
        _mm_write_UBYTES(mf->chanvol, mf->numchn, fpo);

        // Write sample information

        puts("Writing sample header information.. ");

        s = mf->samples;
        for(v=0; v<numsmp; v++, s++)
        {   _mm_write_M_UWORD(s->flags, fpo);
            _mm_write_M_ULONG(s->speed, fpo);
            _mm_write_UBYTE(s->volume, fpo);
            _mm_write_M_UWORD(s->panning, fpo);
            _mm_write_M_ULONG(s->length, fpo);
            _mm_write_M_ULONG(s->loopstart, fpo);
            _mm_write_M_ULONG(s->loopend, fpo);
            _mm_write_M_ULONG(s->susbegin, fpo);
            _mm_write_M_ULONG(s->susend, fpo);

            _mm_write_UBYTE(s->globvol, fpo);
            _mm_write_UBYTE(s->vibflags, fpo);
            _mm_write_UBYTE(s->vibtype, fpo);
            _mm_write_UBYTE(s->vibsweep, fpo);
            _mm_write_UBYTE(s->vibdepth, fpo);
            _mm_write_UBYTE(s->vibrate, fpo);

            StringWrite(strip ? NULL : s->samplename, fpo);
        }


        // Write instruments

        if(mf->flags & UF_INST)
        {   // This module uses instrument information, so save it out

            puts("Writing instrument header information.. ");

            i = mf->instruments;
            for(v=0; v<numins; v++, i++)
            {   _mm_write_UBYTE(i->flags, fpo);
                _mm_write_UBYTE(i->nnatype, fpo);
                _mm_write_UBYTE(i->dca, fpo);
                _mm_write_UBYTE(i->dct, fpo);
                _mm_write_UBYTE(i->globvol, fpo);
                _mm_write_M_UWORD(i->panning, fpo);

                _mm_write_UBYTE(i->pitpansep, fpo);
                _mm_write_UBYTE(i->pitpancenter, fpo);
                _mm_write_UBYTE(i->rvolvar, fpo);
                _mm_write_UBYTE(i->rpanvar, fpo);

                _mm_write_M_UWORD(i->volfade, fpo);
    
                // Dump out the volume envelope

                _mm_write_UBYTE(i->volflg, fpo);
                _mm_write_UBYTE(i->volpts, fpo);
                _mm_write_UBYTE(i->volsusbeg, fpo);
                _mm_write_UBYTE(i->volsusend, fpo);
                _mm_write_UBYTE(i->volbeg, fpo);
                _mm_write_UBYTE(i->volend, fpo);

                for(w=0; w<i->volpts; w++)
                {   _mm_write_M_SWORD(i->volenv[w].pos, fpo);
                    _mm_write_M_SWORD(i->volenv[w].val, fpo);
                }

                // Dump out the panning envelope

                _mm_write_UBYTE(i->panflg, fpo);
                _mm_write_UBYTE(i->panpts, fpo);
                _mm_write_UBYTE(i->pansusbeg, fpo);
                _mm_write_UBYTE(i->pansusend, fpo);
                _mm_write_UBYTE(i->panbeg, fpo);
                _mm_write_UBYTE(i->panend, fpo);

                for(w=0; w<i->panpts; w++)
                {   _mm_write_M_SWORD(i->panenv[w].pos, fpo);
                    _mm_write_M_SWORD(i->panenv[w].val, fpo);
                }

                // Dump out the pitch envelope

                _mm_write_UBYTE(i->pitflg, fpo);
                _mm_write_UBYTE(i->pitpts, fpo);
                _mm_write_UBYTE(i->pitsusbeg, fpo);
                _mm_write_UBYTE(i->pitsusend, fpo);
                _mm_write_UBYTE(i->pitbeg, fpo);
                _mm_write_UBYTE(i->pitend, fpo);

                for(w=0; w<i->pitpts; w++)
                {   _mm_write_M_SWORD(i->pitenv[w].pos, fpo);
                    _mm_write_M_SWORD(i->pitenv[w].val, fpo);
                }

                _mm_write_UBYTES(i->samplenumber, 120, fpo);
                _mm_write_UBYTES(i->samplenote, 120, fpo);

                StringWrite(strip ? NULL : i->insname, fpo);
            }
        }

        // Write pattern information

        _mm_write_M_UWORDS(mf->pattrows, mf->numpat, fpo);
        _mm_write_M_UWORDS(mf->patterns, mf->numpat * mf->numchn, fpo);

        // Write track information

        for(v=0; v<mf->numtrk; v++)
        {   _mm_write_M_UWORD(TrkLen(mf->tracks[v]), fpo);
            _mm_write_UBYTES(mf->tracks[v], TrkLen(mf->tracks[v]), fpo);
        }


        // Now dump out the sampled data
        
        s = mf->samples;
        for(v=0; v<numsmp; v++, s++)
        {   if(s->seekpos) _mm_fseek(fpi, s->seekpos, SEEK_SET);
            _mm_copyfile(fpi, fpo, s->length * ((s->flags & SF_16BITS) ? 2 : 1));
        }

        puts("Done.");

        // and clean up

        fclose(fpo);
        fclose(fpi);
        MikMod_FreeSong(mf);
    }
    return 0;
}

