/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.rhythm;
import herman.elements.*;
import herman.misc.*;
import herman.rhythm.rules.*;

public class Foot
{
  
  private Foot(){}


  public final static int RULE_START=0;
  public final static int RULE_MIDDLE=1;
  public final static int RULE_END=2;

  public final static int AMBIBRACH =0;
  public final static int ANAPEST =1;
  public final static int DACTYL =2;
  public final static int IAMB =3;
  public final static int TROCHEE =4;


  private static Object[] rhythmRules = new Object[0];

  public static int lastFootUsed = -1;

  public static void init()
  {
    ItemList rules = new ItemList();


    rules.add(new AmbibrachRuleStart());
    rules.add(new AnapestRuleStart());
    rules.add(new IambRuleStart());
    rules.add(new TrocheeRuleStart());
    rules.add(new DactylRuleStart());

    rules.add(new AmbibrachRuleMiddle());
    rules.add(new AnapestRuleMiddle());
    rules.add(new IambRuleMiddle());
    rules.add(new TrocheeRuleMiddle());
    rules.add(new DactylRuleMiddle());
    rules.add(new ReverseFootMiddle());


    rules.add(new AmbibrachRuleEnd());
    rules.add(new DactylRuleEnd());
    rules.add(new ReverseFootEnd());
    rules.add(new TrocheeRuleEnd());

    rules.add(new AddSilenceEnd());
    //    rules.add(new AddSilenceMiddle());

    
    rhythmRules = rules.getArray();
  }




  public static Object[] getPattern(RhythmStuff params)
  {
    ItemList feet = new ItemList();
    int ruleType = RULE_START;
    double beatsUsed = 0;

    lastFootUsed = -1;

    while(true)
    {
      // to hold list of valid rules
      ItemList rules = new ItemList();
      
      // go through the rules, looking for those in range
      for(int i=0; i< rhythmRules.length; i++)
	if ((((RhythmRule) rhythmRules[i]).rangeStart <= params.tensionLevel) &&
	    (((RhythmRule) rhythmRules[i]).rangeEnd >= params.tensionLevel) &&
	    (((RhythmRule) rhythmRules[i]).ruleType == ruleType))
	{
	  int rangeSize = ((RhythmRule) rhythmRules[i]).rangeEnd - 
	    (((RhythmRule) rhythmRules[i]).rangeStart);
	  int numCopies = (100 / rangeSize) / 20;
	  numCopies *= ((RhythmRule) rhythmRules[i]).likelihood;
	  if (numCopies == 0) numCopies = 1;
	  
	  for(int j=0; j < numCopies; j++)
	    rules.add(rhythmRules[i]);
	}

      // check we have SOME rules
      if (rules.length() == 0)
	throw new FatalError("No Appropriate Rhythm rules for tension= " + 
			     params.tensionLevel + " rule type = " +
			     ruleType);
    

      // execute a rule, taking note of how many new beats were added
      beatsUsed += executeRule(feet, params, ruleType, beatsUsed, rules);
     
      // if the last rule executed was an END, then we can't
      // add in any more beats, so we exit
      if (ruleType == RULE_END) break;
 
      // if the rule we just executed was a START rule, then we can't	
      // be at the start anymore... duh! :)
      if (ruleType == RULE_START) ruleType = RULE_MIDDLE;

      // If we've got less than 2 beats left to add in, then it's
      // time for an END rule
      if ((((float) params.motifLengthBeats) - beatsUsed) < 2)
      	ruleType = RULE_END;
    }
    
    // return the stuff
    Object[] retval = {(Object) feet.getArray(), 
		       (Object) new Double(beatsUsed)};
    return(retval);
  }
  


  private static double executeRule(ItemList feet, RhythmStuff params,
				    int ruleType, double nextTimeBeats,
				    ItemList rules)
  {
    // choose rule (randomly)
    int ruleNum = (int)
      Math.round(Rhythm.randGenerator.nextDouble() * (rules.length()-1));
    
    // execute it
    return(((RhythmRule) rules.getElement(ruleNum)).apply(feet,params,
							  nextTimeBeats));
  }


  public static double addAmbibrach(ItemList feet, RhythmStuff params, 
				    double nextTimeBeats)
  {
    double beatCounter=0;

    beatCounter += addUnaccented(feet, params, nextTimeBeats + beatCounter);
    beatCounter += addAccented(feet, params, nextTimeBeats + beatCounter);
    beatCounter += addUnaccented(feet, params, nextTimeBeats + beatCounter);

    return(beatCounter);
  }



  public static double addAnapest(ItemList feet, RhythmStuff params, 
				  double nextTimeBeats)
  {
    double beatCounter=0;

    beatCounter += addUnaccented(feet, params, nextTimeBeats + beatCounter);
    beatCounter += addUnaccented(feet, params, nextTimeBeats + beatCounter);
    beatCounter += addAccented(feet, params, nextTimeBeats + beatCounter);
 
    return(beatCounter);
  }



  public static double addDactyl(ItemList feet, RhythmStuff params, 
				 double nextTimeBeats)
  {
    double beatCounter=0;

    beatCounter += addAccented(feet, params, nextTimeBeats + beatCounter);
    beatCounter += addUnaccented(feet, params, nextTimeBeats + beatCounter);
    beatCounter += addUnaccented(feet, params, nextTimeBeats + beatCounter);

    return(beatCounter);
  }



  public static double addIamb(ItemList feet, RhythmStuff params, 
				 double nextTimeBeats)
  {
    double beatCounter=0;

    beatCounter += addUnaccented(feet, params, nextTimeBeats + beatCounter);
    beatCounter += addAccented(feet, params, nextTimeBeats + beatCounter);

    return(beatCounter);
  }



  public static double addTrochee(ItemList feet, RhythmStuff params, 
				  double nextTimeBeats)
  {
    double beatCounter=0;

    beatCounter += addAccented(feet, params, nextTimeBeats + beatCounter);
    beatCounter += addUnaccented(feet, params, nextTimeBeats + beatCounter);

    return(beatCounter);
  }


  public static double addUnaccented(ItemList feet, RhythmStuff params,
				     double nextTimeBeats)
  {
    Event e = new Event();
    e.startTime =
      (long) Math.round(nextTimeBeats * params.beatLengthMillis);
    e.duration =
      (long) Math.round(params.unAccentedNoteLengthBeats * 
			params.beatLengthMillis);
    e.velocity = params.unAccentedNoteVelocity;
    feet.add(e);
    
    return(params.unAccentedNoteLengthBeats);
  }



  public static double addAccented(ItemList feet, RhythmStuff params,
				   double nextTimeBeats)
  {
    double duration=0;
    
    Event e = new Event();
    e.startTime =
      (long) Math.round(nextTimeBeats * params.beatLengthMillis);
    switch(params.accentingType)
    {
    case RhythmStuff.ACCENT_NONE:
      e.duration =
	(long) Math.round(params.unAccentedNoteLengthBeats *
			  params.beatLengthMillis);
      e.velocity = params.unAccentedNoteVelocity;
      duration = params.unAccentedNoteLengthBeats;
      break;
      
    case RhythmStuff.ACCENT_VELOCITY:
      e.duration =
	(long) Math.round(params.unAccentedNoteLengthBeats *
			  params.beatLengthMillis);
      e.velocity = params.accentedNoteVelocity;
      duration = params.unAccentedNoteLengthBeats;
      break;
      
    case RhythmStuff.ACCENT_LENGTH:
      e.duration =
	(long) Math.round(params.accentedNoteLengthBeats *
			  params.beatLengthMillis);
      e.velocity = params.unAccentedNoteVelocity;
      duration = params.accentedNoteLengthBeats;
      break;
      
    case RhythmStuff.ACCENT_VELOCITYLENGTH:
      e.duration =
	(long) Math.round(params.accentedNoteLengthBeats *
			  params.beatLengthMillis);      
      e.velocity = params.accentedNoteVelocity;
      duration = params.accentedNoteLengthBeats;
      break;
    }
    feet.add(e);
    
    return(duration);
  }
}
