/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.properties;
/** 
 * The Pitch class represents pitch data by storing a relative pitch and a 
 *  register, from which it can calculate absolute pitch.
 *  
 *  <p>Relative pitch: an integer representing one of the 12 tones, with
 *    'c' being 0.  Alternatively a String description can be used:
 *    <ul>
 *    <li>"c"  == C
 *    <li>"cB" == C-double-flat
 *    <li>"cb" == C-flat
 *    <li>"c#" == C-sharp
 *    <li>"cx" == C-double-sharp
 *  </ul>
 *  Case is not important for the note character
 *  <p>
 *  Register: an integer representing the 12-tone register which sets
 *    the basis for the relative pitch.  A zero register means that the 
 *    relative pitch of 0 corresponds to an absolute pitch of "Middle C".
 *  <p>
 *  The absolute pitch is calculated as:
 *     (Register * 12) + [int value for Middle C] + Relative pitch
 **/

public class Pitch implements Cloneable {
  // The absolute pitch is the relative pitch...
  private int relativePitch = -1000;           // 0 = c, 1 = d, -1 = b, etc. 

  // set in the appropriate register...
  private int register = 0;                // 0 = register of "Middle" C

  // relative to "Middle" C.
  private static final int MiddleC = 60;   // MIDI value for "Middle" C
   
  // Convert a#,bx,cb,... etc. to integers for relative pitch
  private static int stringToPitch( String s ) {
    int p = 0;
    char first = s.charAt(0);
    if ((first == 'c')||(first == 'C')) p = 0;
    else if ((first == 'd')||(first == 'D')) p = 2;
    else if ((first == 'e')||(first == 'E')) p = 4;
    else if ((first == 'f')||(first == 'F')) p = 5;
    else if ((first == 'g')||(first == 'G')) p = 7;
    else if ((first == 'a')||(first == 'A')) p = 9;
    else if ((first == 'b')||(first == 'B')) p = 11;

    if (s.length() > 1) {
      if (s.charAt(1) == 'B') p = p - 2;
      else if (s.charAt(1) == 'b') p--;
      else if (s.charAt(1) == '#') p++;
      else if (s.charAt(1) == 'x') p = p+ 2;
    }
    return p;
  }
  
  // Convert integers for relative pitch to Strings
  private static String pitchToString(int p ) {
    StringBuffer s = new StringBuffer(2);
    if (p != -1000) {p = p % 12;}
    if (p < 0) p = p + 12;
    if (p == 0) s.insert(0,'C');
    else if (p == 1) s.insert(0,"C#");
    else if (p == 2) s.insert(0,'D');
    else if (p == 3) s.insert(0,"Eb");
    else if (p == 4) s.insert(0,'E');
    else if (p == 5) s.insert(0,'F');
    else if (p == 6) s.insert(0,"F#");
    else if (p == 7) s.insert(0,'G');
    else if (p == 8) s.insert(0,"Ab");
    else if (p == 9) s.insert(0,'A');
    else if (p == 10) s.insert(0,"Bb");
    else if (p == 11) s.insert(0,'B');
    String str = new String(s);
    str.trim();
    return str;
  }
  
/**
 * Return String describing absolute pitch (for printing).
 */
  public String toString() {
    return "Pitch: " + getAbsPitch() + ", " + 
      pitchToString(relativePitch);
  }

  public Pitch() {/* Nothing */}

/**
* Allocates new Pitch object and sets the relative pitch to the integer 
* value of the argument.
*/

  public Pitch( int p ) {
    setRelPitch( p );
    return;
  }
  
/**
* Allocates new Pitch object and sets the relative pitch based on the  
* String description given in the argument..
*/
  public Pitch( String p ) {
    setRelPitch( stringToPitch(p) );
    return;
  }

/**
* Allocates new Pitch object and sets the absolute pitch. 
* @param p   the integer value of the relative pitch
* @param r   the integer value of the register
*/
  public Pitch( int p, int r ) {
    setRelPitch( p );
    setRegister( r );
    return;
  }

/**
* Allocate new Pitch object and set the absolute pitch.
* @param p    the String description of the relative pitch
* @param r    the integer value of the register
*/
  public Pitch( String p, int r ) {
    setRelPitch( stringToPitch(p) );
    setRegister( r );
    return;
  }

// Access methods

/**
 * Set the relative pitch using an integer value.
 */
  public void setRelPitch( int p ) { relativePitch = p;}

/**
 * Set the relative pitch using a String description.
 */
  public void setRelPitch( String p ) { relativePitch = stringToPitch(p);}

/**
 * Set the register using an integer value.
 */
  public void setRegister( int r ) { register = r;}

/**
 * Returns true if the pitch has not been set yet.
 */
  public boolean isNull() { if (relativePitch==-1000) 
    return true; else return false; }

  /** 
   * Sets the pitch back to null
   */
  public void setToNull() {
    relativePitch = -1000;
    register = 0;
  }

/* ******************* NEED EXCEPTION HANDLING FOR NUL CASE!!! */
/**
 * Get the relative pitch as an integer value.
 */
  public int getRelPitch() { return relativePitch;}

/**
 * Get the relative pitch as a String description.
 */
  public String getStrRelPitch() {return pitchToString(relativePitch);}

/**
 * Get the register as an integer value.
 */
  public int getRegister() { return register;}

/**
 * Get the absolute pitch as an integer value.
 */
  public int getAbsPitch() { return (12*register+MiddleC+relativePitch);}

  // Relative pitch is the default

/**
 * Default is to set the relative pitch.
 */
  public void setPitch( int p ) { relativePitch = p;}

/**
 * Default is to set the relative pitch.
 */
  public void setPitch( String p ) { relativePitch = stringToPitch(p);}

/**
 * Default is to get the relative pitch as an integer.
 */
  public int getPitch() { return relativePitch;}

/**
 *  For debugging.
 */
  public static void main(String[] args) {
    Pitch p1 = new Pitch();
    System.out.println(p1.toString());
    System.out.println(p1.isNull());
    p1.setPitch(1);
    System.out.println(p1.toString());
    p1.setRegister(1);
    System.out.println(p1.toString());
  }

/**
 * Allows cloning
 */
  public Object clone() {
    Object o = null;
    try {
      o = super.clone();
    } catch (CloneNotSupportedException e) {
      System.out.println("Pitch can't clone");
    }
    return o;
  }

}




