/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.properties;
/** 
 * The Key class represents musical key as a base type (e.g. C#) and a mode 
 *  (major or minor).
 *  <p>
 *  The base type can be accessed either as an integer in [0,11] (with 0 being 
 *    "C") or as a String description:
 *    <ul>
 *    <li>"C"  == C
 *    <li>"CB" == C-double-flat
 *    <li>"Cb" == C-flat
 *    <li>"C#" == C-sharp
 *    <li>"Cx" == C-double-sharp
 *  </ul>
 *  Case is not important for the first note character.
 *  <p>
 *  The mode is represented as a String, either "major" or "minor".
 **/

public class Key implements Cloneable {
  // The base type is the relative pitch of a Pitch object.
  private Pitch keyType = new Pitch();            

  // The mode is a StringBuffer, either "major" or "minor"
  private StringBuffer mode = new StringBuffer("NONE");

/**
 * Return String describing key (for printing).
 */
  public String toString() {
    return  keyType.getStrRelPitch() + " " + 
      getMode();
  }

  public Key() { /* Nothing */}
  
/**
 * Allocates a new Key object using an integer for the base type.
 *@param kt     an integer representing the base type
 *@param m      a String representing the mode
 */
  public Key(int kt, String m) {
    setMode(m);
    setKeyType(kt);
    return;
  }

/**
 * Allocates a new Key object using a String for the base type.
 *@param kt     an String representing the base type
 *@param m      a String representing the mode
 */
  public Key( String kt, String m ) {
    setMode(m);
    setKeyType(kt);
    return;
  }


// Access Methods  

/**
 * Returns true if the key has not been set yet.
 */
  public boolean isNull() { if (keyType.isNull()) 
    return true; else return false; }
/**
 * Set the key back to null
 */
  public void setToNull() { 
    keyType = new Pitch();
  }

/**
 * Sets base type using an integer.
 */
  public void setKeyType( int kt ) { 
    keyType.setRelPitch(kt);
  }

/**
 * Sets base type using a String
 */
  public void setKeyType( String kt ) { 
    keyType.setRelPitch(kt);
  }

/**
 * Sets the mode using a String
 */
  public void setMode( String m ) {  
    mode.insert(0,m);
    mode.setLength(m.length());
  }

/**
 * Gets the base type as a String
 */
  public String getStrKeyType() { return keyType.getStrRelPitch();}

/**
 * Gets base type as an integer
 */
  public int getIntKeyType() { return keyType.getRelPitch();}

/**
 * Gets mode as a String
 */
  public String getMode() { return (mode.toString()).trim();}

/**
 *  For debugging.
 */
  public static void main(String[] args) {
    Key p1 = new Key();
    System.out.println(p1.toString());
    System.out.println(p1.isNull());
    p1.setKeyType(5);
    p1.setMode("minor");
    System.out.println(p1.toString());
    p1.setKeyType("Eb");
    p1.setMode("major");
    System.out.println(p1.toString());
  }

/**
 *  Allows cloning
 */
  public Object clone() {
    Key k = null;
    try {
      k = (Key)super.clone();
    } catch (CloneNotSupportedException e) {
      System.out.println("Key can't clone");
    }
    k.keyType = (Pitch)k.keyType.clone();
    k.mode = new StringBuffer(mode.toString());
    return k;
  }

}




