/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.properties;

/** 
 * The ChordType class represents chord types as as degree (I...VII), 
 *  a type (M,m,dim,aug), and any extra tones included beyond the triad 
 *  (like sevenths and ninths).
 *  <p>
 *  The degree can be accessed either as an integer in [1,7] or as the Roman
 *    numerals I...VII
 *  <p>
 *  The type can be accessed either as an integer in [0,3] or as the 
 *  respective String description: "major", "minor", "diminished", and
 *  "augmented".
 *  <p>
 *  The extra tones can be accessed either as an integer in [0,8] or as the
 *  respective String description: "none", "m7", "M7", "m9", "M9", "m7 & m9",
 *    "M7 & m9", "m7 & M9", "M7 & M9".
 *
 **/

public class ChordType implements Cloneable {
  // The degree is 1...7 (corresp. to I...VII)
  private int degree = -1000;          

  // The type is 0...3 (major, minor, diminished, augmented)
  private int type = -1000;             

  // The extra tones can be 0..4 (none, m7, M7, m9, M9 + combinations)
  private int extra = 0;
 
  // Converts integer index to a string description of degree
  private static String intToStringDegree( int index ) {
    switch(index) {
    case 1: String deg = "I"; return deg;
    case 2: String deg2 = "II"; return deg2;
    case 3: String deg3 = "III"; return deg3;
    case 4: String deg4 = "IV"; return deg4;
    case 5: String deg5 = "V"; return deg5; 
    case 6: String deg6 = "VI"; return deg6;
    case 7: String deg7 = "VII"; return deg7;
    } 
    return null;
  }
 
  // Converts integer index to a string description of type
  private static String intToStringType( int index ) {
    switch(index) {
    case 0: String t = "major"; return t;
    case 1: String t2 = "minor"; return t2;
    case 2: String t3 = "diminished"; return t3;
    case 3: String t4 = "augmented"; return t4;
    }
    return null;
  }

  // Converts integer index to a string description of extra tones
  private static String intToStringExtras( int index ) {
    switch(index) {
    case 0: String e = "none"; return e;
    case 1: String e2 = "m7"; return e2;
    case 2: String e3 = "M7"; return e3;
    case 3: String e4 = "m9"; return e4;
    case 4: String e5 = "M9"; return e5;
    case 5: String e6 = "m7 & m9"; return e6;
    case 6: String e7 = "M7 & m9"; return e7;
    case 7: String e8 = "m7 & M9"; return e8;
    case 8: String e9 = "M7 & M9"; return e9;
    }
    return null;
  }

  // Converts string description of degree into int index
  private static int stringToIntDegree( String descrp ) {
    if (descrp.equals("I")) return 1;
    else if (descrp.equals("II")) return 2;
    else if (descrp.equals("III")) return 3;
    else if (descrp.equals("IV")) return 4;
    else if (descrp.equals("V")) return 5;
    else if (descrp.equals("VI")) return 6;
    else if (descrp.equals("VII")) return 7;
    return 0;
  }

  // Converts string description of type into int index
  private static int stringToIntType( String descrp ) {
    if (descrp.equals("major")) return 0;
    else if (descrp.equals("minor")) return 1;
    else if (descrp.equals("diminished")) return 2;
    else if (descrp.equals("augmented")) return 3;
    return 0;
  }

  // Converts string description of extra tones into int index
  private static int stringToIntExtras( String descrp ) {
    if (descrp.equals("none")) return 0;
    else if (descrp.equals("m7")) return 1;
    else if (descrp.equals("M7")) return 2;
    else if (descrp.equals("m9")) return 3;
    else if (descrp.equals("M9")) return 4;
    else if (descrp.equals("m7 & m9")) return 5;
    else if (descrp.equals("M7 & m9")) return 6;
    else if (descrp.equals("m7 & M9")) return 7;
    else if (descrp.equals("M7 & M9")) return 8;
    return 0;
  }

/**
 * Return String describing chord type (for printing).
 */
  public String toString() {
    return intToStringDegree(degree) + " " + 
      intToStringType(type) + " with " + intToStringExtras(extra);
  }

  public ChordType() {/* Nothing */}

/**
 * Allocates a new ChordType object using integers to set the degree, type,
 *  and extra tones fields.
 */
  public ChordType(int d, int t, int e) {
    setDegree(d);
    setType(t);
    setExtras(e);
    return;
  }

/**
 * Allocates a new ChordType object using Strings to set the degree, type,
 *  and extra tones fields.
 */
  public ChordType( String d, String t, String e ) {
    setDegree( d );
    setType(t);
    setExtras(e);
    return;
  }


// Access Methods  
  /* *************** THE ACCESS METHODS NEED Exception HANDLING!!!!! */
/**
 * Returns true if the chord type has not been set yet.
 */
  public boolean isNull() { if (degree==-1000) 
    return true; else return false; }

/**
 * Sets the chord type back to null
 */
  public void setToNull() { 
    degree = -1000;
    type = -1000;
    extra = 0;
  }

/**
 * Set the degree using an integer
 */
  public void setDegree( int d ) { degree = d;}

/**
 * Set the type using an integer
 */
  public void setType( int t ) { type = t;}

/**
 * Set the extra tones using an integer
 */
  public void setExtras( int e ) { extra = e;}

/**
 * Set the degree using a String
 */
  public void setDegree( String d ) { degree = stringToIntDegree(d);}

/**
 * Set the type using a String
 */
  public void setType( String t ) { type = stringToIntType(t);}

/**
 * Set the extra tones using a String
 */
  public void setExtras( String e ) { extra = stringToIntExtras(e);}


/**
 * Get the degree as an integer
 */
  public int getIntDegree() { return degree;}

/**
 * Get the type as an integer
 */
  public int getIntType() { return type;}

/**
 * Get the extra tones as an integer
 */
  public int getIntExtras() { return extra;}

/**
 * Get the degree as a String
 */
  public String getStringDegree() { return intToStringDegree(degree);}

/**
 * Get the type as a String
 */
  public String getStringType() { return intToStringType(type);}

/**
 * Get the extra tones as a String
 */
  public String getStringExtras() { return intToStringExtras(extra);}


/**
 *  For debugging.
 */
  public static void main(String[] args) {
    ChordType p1 = new ChordType();
    System.out.println(p1.toString());
    System.out.println(p1.isNull());
    p1.setDegree(3);
    p1.setType(2);
    p1.setExtras(1);
    System.out.println(p1.toString());
    p1.setDegree(7);
    System.out.println(p1.toString());
  }

/**
 * Allows cloning
 */
  public Object clone() {
    Object o = null;
    try {
      o = super.clone();
    } catch (CloneNotSupportedException e) {
      System.out.println("ChordType can't clone");
    }
    return o;
  }
}




