/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.pitch.melody;
import herman.properties.*;
import herman.pitch.tools.*;
import herman.pitch.harmony.*;
import herman.elements.*;
/** 
 * The MelodyScale class contains the rules that help to limit melodic pitches
 *  to those valid in the given key.
 *  <p>
 *  @see Key
 **/

public class MelodyScale {

  //The "Not Possible Number"
  private static final float no = -1000.0f;
  private static final float Yes = 1.0f;

  //scaleFlag determines whether or not the melody must be in the correct key
  private static boolean scaleFlag = true;
  
  //Scales:
  private static final float[] majorScaleTones =
  {0,2,4,5,7,9,11};

  private static final float[] minorScaleTones =
  {0,2,3,5,7,8,9,10,11};

  private static final float[] majorScaleLimits =
  {Yes,no,Yes,no,Yes,Yes,no,Yes,no,Yes,no,Yes};

  private static final float[] minorScaleLimits =
  {Yes,no,Yes,Yes,no,Yes,no,Yes,Yes,Yes,Yes,Yes};

  public MelodyScale() { /* Nothing */}
  

    

// Access Methods

/** 
 *  Turns the relative pitch of the supplied melody event into a KeyPitch
 *   object.
 */
  public static KeyPitch getKeyPitch(Event mel) {
    KeyPitch keyPitch = new KeyPitch();
    int kp = 0;
    int keyType = mel.key.getIntKeyType() % 12;
    if (keyType < 0) keyType = keyType + 12;

    if (mel.pitch.isNull()) {
      kp = 0;
      keyPitch.setKeyPitch(kp);
      return keyPitch;
    }
    int pitch = mel.pitch.getRelPitch();
    
    kp = pitch - keyType;
    keyPitch.setKeyPitch(kp);
    return keyPitch;
  }

/**
 *  Returns a float[] indicating the scale tones available in the supplied
 *  key.
 */
 public static float[] getScaleTones( Key chordKey) {
    float[] scaleTones = new float[12];
 
    if (scaleFlag) {
      ArrayTools.setArrayValue(scaleTones,no);

      //Set the scaleTones array based on key and mode
      if (chordKey.getMode().equals("major")) {
	scaleTones = majorScaleLimits;
      } else {scaleTones = minorScaleLimits;}
    }
    //Otherwise, if rules don't apply,
    else {ArrayTools.setArrayValue(scaleTones, 0.0f);}

    //Return
    return scaleTones;
 } 

  
/** 
 * Changes an integer tone into the proper scale degree [0,6] for the 
 * supplied key
 */ 
  public static int changeToScaleDegree(int tone, Key key) {
    boolean scaleFlagStatus = true;
    if (!scaleFlag) scaleFlagStatus = false;

    scaleFlag = true;
    float[] scaleTones = getScaleTones(key);
    scaleFlag = scaleFlagStatus;
    
    tone = tone % 12;
    if (tone < 0 ) tone = tone + 12;

    int deg = 0;
    for (int i = 0; i<scaleTones.length; i++) {
      if (tone == i) return deg;
      if (scaleTones[i] > 0.0f) deg = deg + 1;
    }
    
    return 0;
  }
    

/**
 *
 *  Moves the event pitch into the mode of the supplied chord, if it is not 
 *  already
 *   a valid member of that mode.  Also makes sure that the status of the
 *   6th and 7th's in minor is the same for the chord and the event.
 */
  public static void moveToMode(Event mel, Chord chord) {
    KeyPitch keyPitch = getKeyPitch(mel);

    //basePitch is the basic scale tone of keyPitch
    int basePitch = keyPitch.getKeyPitch() % 12;
    if(basePitch < 0) {basePitch = basePitch + 12;}

    boolean isScaleTone = false;
    boolean isScale = false;
    if (mel.key.getMode().equals(chord.key.getMode())) isScaleTone = true;

    if ((chord.key.getMode().equals("major"))&&(!isScale)) {
      mel.key.setMode("major");                          //Set the mel mode
      for (int i = 0; i<majorScaleTones.length; i++) {
	if (basePitch == majorScaleTones[i]) {
	  isScaleTone = true;                    //Tone is in scale
	}
      }

      if (!isScaleTone) {
       mel.pitch.setRelPitch(mel.pitch.getRelPitch()+1);   //bump up into scale
      }
    }

    else if((chord.key.getMode().equals("minor"))&&(!isScale)) { 
      mel.key.setMode("minor");
      for (int i = 0; i<minorScaleTones.length; i++) {
	if (basePitch == minorScaleTones[i]) {
	  isScaleTone = true;                    //Tone is in scale
	}
      }
      if (!isScaleTone) {                        //if tone not in scale
	mel.pitch.setRelPitch(mel.pitch.getRelPitch()-1); //bump dwn into scale
	basePitch = basePitch - 1;   
      }

      if ((PivotRules.hasRaised7(chord))&&(basePitch==10)) {      
	                 //if chord has raised 7th & keyPitch unraised 7th
	mel.pitch.setRelPitch(mel.pitch.getRelPitch()+1);  //raise the 7th
      }
      else if ((PivotRules.hasRaised6(chord))&&(basePitch==8)) {      
	                 //if chord has raised 6th & keyPitch unraised 6th
	mel.pitch.setRelPitch(mel.pitch.getRelPitch()+1);  //raise the 6th
      }
      else if ((PivotRules.hasUnRaised7(chord))&&(basePitch==11)) {      
	                 //if chord has unraised 7th & keyPitch raised 7th
	mel.pitch.setRelPitch(mel.pitch.getRelPitch()-1);  //lower the 7th
      }
      else if ((PivotRules.hasUnRaised6(chord))&&(basePitch==9)) {      
	                 //if chord has unraised 6th & keyPitch raised 6th
	mel.pitch.setRelPitch(mel.pitch.getRelPitch()-1);  //lower the 6th
      }

    }
  }

  /**
 *
 *  Moves the event pitch into the mode of the supplied key, if it is not 
 *  already
 *   a valid member of that mode.
 */
  public static void moveToMode(Event mel, Key key) {
    KeyPitch keyPitch = getKeyPitch(mel);

    //basePitch is the basic scale tone of keyPitch
    int basePitch = keyPitch.getKeyPitch() % 12;
    if(basePitch < 0) {basePitch = basePitch + 12;}

    boolean isScaleTone = false;
    boolean isScale = false;
    if (mel.key.getMode().equals(key.getMode())) isScaleTone = true;

    if ((key.getMode().equals("major"))&&(!isScale)) {
      mel.key.setMode("major");                          //Set the mel mode
      for (int i = 0; i<majorScaleTones.length; i++) {
	if (basePitch == majorScaleTones[i]) {
	  isScaleTone = true;                    //Tone is in scale
	}
      }

      if (!isScaleTone) {
       mel.pitch.setRelPitch(mel.pitch.getRelPitch()+1);   //bump up into scale
      }
    }

    else if((key.getMode().equals("minor"))&&(!isScale)) { 
      mel.key.setMode("minor");
      for (int i = 0; i<minorScaleTones.length; i++) {
	if (basePitch == minorScaleTones[i]) {
	  isScaleTone = true;                    //Tone is in scale
	}
      }
      if (!isScaleTone) {                        //if tone not in scale
	mel.pitch.setRelPitch(mel.pitch.getRelPitch()-1); //bump dwn into scale
	basePitch = basePitch - 1;   
      }
    }
  }

/** 
 *  Returns the status of the scaleFlag.  If "true", the melody must remain in
 *   the key of its first associated chord.
 */
  public static boolean checkFlag() {
    if (scaleFlag) return true;
    return false;
  }


/** 
 *  Set the status of the scaleFlag.  If "true", the melody must remain in the
 *   current key.
 */
  public static void setFlag(boolean flag) {
    scaleFlag = flag;
  }

/**
 *  For debugging.
 */
  public static void main(String[] args) {
  
  }
}
