/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.pitch.melody;
import herman.elements.*;
import herman.pitch.tools.*;
import herman.pitch.harmony.*;
import herman.properties.*;
import herman.pitch.instruments.*;

/** 
*   Generates the melodic progression of the music
*/

public class MelodyProgression {
  

  /**
   * The in queue containing melodic events to be instantiated.
   */
  public static Queue inQueue = new Queue();
  private static float Yes = 1.0f;
  private static float no = -1000.0f;
  private static final int INTERVALnum = 29;
  private static boolean hasChord = true;

  // Parameter arrays
  static float[] newIntervals = new float[INTERVALnum];
  static float[] scaleTones = new float[12];
  static float[] scaleIntervals = new float[INTERVALnum];
  static float[] chordTones = new float[12];
  static float[] newTones = new float[12];
  static float[] tempTones = new float[12];
  static float[] chordIntervals = new float[INTERVALnum];
  static float[] preferredIntervals = new float[INTERVALnum];
  static float[] harshIntervals = new float[INTERVALnum];
  static float[] followIntervals = new float[INTERVALnum];
  static float[] tempIntervals = new float[INTERVALnum];

  // Last melodic event
  static Event lastMelody;

  // Last Interval
  private static Interval lastInterval = new Interval();

 
  //"Special" Blocks:

  // Check to see if at least one event probability > 0 
  private static boolean zeroCheck( float[] prob) {
    int greaterThanZeroFlag = 0;
    for (int i = 0; i < prob.length; i++)
      if (prob[i] >= 0.0f)
	greaterThanZeroFlag++;
    return (greaterThanZeroFlag > 0) ? true : false;
  }

  //Clear the "special" blocks:
  private static void clearBlocks() {
    
  }

  // Send an event to the Instrumentation module
  private static void instantiateMelody (Event melEvent) {
    Instrumentation.melodyInQueue.push(melEvent);
  }


  // Translation functions .  
  private static float[] tonesToIntervals(float[] tones, KeyPitch keyPitch) {
    float[] intervals = new float[INTERVALnum];
    ArrayTools.setArrayValue(intervals,no);

    //basePitch is the basic scale tone of keyPitch
    int basePitch = keyPitch.getKeyPitch() % 12;
    if (basePitch < 0 ) {basePitch = basePitch +12;}

    for (int i = 0; i<tones.length; i++) {
      if (tones[i] > 0.0f) {
	intervals[14 + i - basePitch] = Yes;    //index 14 = interval of 0
	
	//The "mirror" image about the basePitch also = Yes
	if ((i-basePitch) > 0) intervals[(14-12+i-basePitch)] = Yes;
	else if ((i-basePitch) < 0) intervals[(14+12+i-basePitch)] = Yes;
	else {intervals[26] = Yes; intervals[2] = Yes;} //the octaves

	//Cover the ninths:
	if ((i-basePitch)==1) intervals[27] = Yes;
	else if ((i-basePitch)==2) intervals[28] = Yes;
	else if ((i-basePitch)==-1) intervals[1] = Yes;
	else if ((i-basePitch)==-2) intervals[0] = Yes;
      }
    }

    return intervals;
  }

  //Assign lastMelody and lastInterval
  private static void assignLasts(Event melEvent) {
    lastInterval = PitchTools.subtract(melEvent.pitch, lastMelody.pitch);
    lastMelody = melEvent;
  }

  //Assign lastMelody and lastInterval
  private static void assignLasts(Event melEvent, Interval interval) {
    lastMelody = melEvent;
    lastInterval = interval;
  }
  
/**
 *  Handles requests for new instantiated melodic events.  Returns true if
 *    successful.
 */
  public static boolean getNewMelody() {
    makeNewMelody();
    return true;
  }
 
/*
 * Start new melodic event instantiation process
 */
  private static void makeNewMelody() {

    //Get new harmonic event
    Event melEvent = ((Event)inQueue.pop());
    
    // Reset the newInterval array
    ArrayTools.setArrayValue( newIntervals, 0.0f );
    ArrayTools.setArrayValue( newTones, 0.0f );

    Chord firstChord = new Chord();
    //Check to see if first harmonic event instantiated:
    // if not, instantiate it
    if (melEvent.supConChord.size() > 0) {
      firstChord = ((Chord) melEvent.supConChord.getFirst());
      while (firstChord.chordType.isNull()||firstChord.key.isNull()) {
	HarmonyProgression.getNewChord();
      }
      hasChord = true;
    } else {hasChord = false;}
    
    /* if (hasChord) {
     *   System.out.println("We've got a chord!!");
     * } else {System.out.println("No chord.");}
     */

    // If melodic event instantiated, pass it on
    if (!(melEvent.pitch.isNull())) {
	
      //Check mode and move to mode if necessary
      if (MelodyScale.checkFlag()) {
	if (hasChord) {
	  MelodyScale.moveToMode(melEvent, firstChord);
	} else if (lastMelody != null) {
	  MelodyScale.moveToMode(melEvent, lastMelody.key);
	}
      }
    
      //next
      assignLasts(melEvent);
      instantiateMelody(melEvent);
      return;
    }

    // Otherwise, continue with melody assignment
    assignMelody(melEvent);     //includes assignLasts
    instantiateMelody(melEvent);
    return;    
  }
  
/*
 * Assign a new melodic event pitch
 */
  private static void assignMelody( Event melEvent ) {
    boolean blockIntervals = false;

    /**** NEED PHRASE POSITION INFO!!!!!!! *********
     *   NEED TO CHECK TO SEE if LASTS EXIST *********/
   
    if (melEvent.eventInfo.getLocation().equals("sentence start")) {
      blockIntervals = true;
    }
    //System.out.println(melEvent.eventInfo.getLocation());
    Chord chord = new Chord();

    //Limit to scale if necessary:
    if (hasChord) {
      chord = ((Chord) melEvent.supConChord.getFirst());
      if (MelodyScale.checkFlag()) {  
	ArrayTools.equalizeArrays(scaleTones,
				  MelodyScale.getScaleTones(chord.key));
	ArrayTools.add2Array(newTones,scaleTones);
      }
      melEvent.key.setKeyType(chord.key.getIntKeyType());
      melEvent.key.setMode(chord.key.getMode());
    } else if (lastMelody!= null) {
      if (MelodyScale.checkFlag()) {
	ArrayTools.equalizeArrays(scaleTones,
				  MelodyScale.getScaleTones(lastMelody.key));
	ArrayTools.add2Array(newTones,scaleTones);
      }
      melEvent.key.setKeyType(lastMelody.key.getIntKeyType());
      melEvent.key.setMode(lastMelody.key.getMode());
    }

    //Limit to chord tones if necessary:
    if (hasChord) {
      for (int i = 0; i < melEvent.supConChord.size(); i++) {
	chord = ((Chord) melEvent.supConChord.getElemAt(i));
	if (!(chord.chordType.isNull())) {
	  ArrayTools.equalizeArrays(tempTones, newTones);
	  ArrayTools.equalizeArrays(chordTones, 
				  ChordTones.matchChordTones(melEvent,chord));
	  ArrayTools.add2Array(tempTones, chordTones);
	  if (zeroCheck(tempTones)) {
	    ArrayTools.equalizeArrays(newTones, tempTones);
	  }
	}
      }

      //Apply resolution rules:
      if (melEvent.eventInfo.getLocation().equals("resolution")) {
	
	ArrayTools.equalizeArrays(tempTones, newTones);
	ArrayTools.add2Array(tempTones, Resolution.getResolveTones(chord));
	if (zeroCheck(tempTones)) {
	  ArrayTools.equalizeArrays(newTones, tempTones);
	} else {System.out.println("Melody resolve failed");}
      }
      //ArrayTools.printArray(newTones);
    } 

    //Follow the pivot rules if necessary
    if ((lastMelody != null)&&(PivotRules.pivotRules)&&
	(melEvent.key.getMode().equals("minor"))&&
	(melEvent.key.getMode().equals(lastMelody.key.getMode()))&&
	(melEvent.key.getIntKeyType()==lastMelody.key.getIntKeyType())) {

      //MelodyPivotRules automatically checks to see if newTones passes
      // the zeroCheck test after applying its rules
      newTones = MelodyPivotRules.applyPivotRules(lastMelody, newTones);
    }
  
    if (!blockIntervals) {
      //Turn tones to intervals
      ArrayTools.add2Array(newIntervals,
			   tonesToIntervals(newTones,
				       MelodyScale.getKeyPitch(lastMelody)));
      //System.out.println("Tones to intervals for new Melody:");
      //  ArrayTools.printArray(newIntervals, 15);
     
      //Add in melody range rules:
      ArrayTools.equalizeArrays(tempIntervals, newIntervals);
      ArrayTools.add2Array(tempIntervals,
			 MelodyRange.applyRangeLimits(lastMelody));
      if (zeroCheck(tempIntervals)) {
	ArrayTools.equalizeArrays(newIntervals, tempIntervals);}
     
 
      //Add in preference rules:
      ArrayTools.equalizeArrays(tempIntervals, newIntervals);
      ArrayTools.add2Array(tempIntervals,preferredIntervals);
      if (zeroCheck(tempIntervals)) {
	ArrayTools.equalizeArrays(newIntervals, tempIntervals);}
  

      //Add in harshness rules:
      ArrayTools.add2Array(newIntervals,harshIntervals);
      
      //Apply Interval following rules:
      ArrayTools.add2Array(newIntervals,
			 IntervalFollowing.applyFollowingRules(lastInterval));
      
      //ArrayTools.printArray(newIntervals, 15);

      //Apply contour instructions:
      ArrayTools.equalizeArrays(tempIntervals, newIntervals);
      ArrayTools.add2Array(tempIntervals,
			 ContourInstructions.applyContour(melEvent));
      if (zeroCheck(tempIntervals)) {
	ArrayTools.equalizeArrays(newIntervals, tempIntervals);}

      //ArrayTools.printArray(newIntervals,15);

      //Make a selection:
      int interIndex = ArrayTools.probableSelect(newIntervals);
      interIndex = interIndex - 14;
   
    
      //Make assignments
      Interval newInterval = new Interval(interIndex);
  
      //Add new interval to last pitch and set result to be the new pitch
      melEvent.pitch = PitchTools.add(lastMelody.pitch, newInterval);
      assignLasts(melEvent, newInterval);
  
      return;
    }
    //If blockIntervals is true...
    else {  
      int pitch = ArrayTools.probableSelect(newTones);

      //newTones is relative to the tonic pitch
      int tonic = melEvent.key.getIntKeyType();
      pitch = (tonic + pitch) % 12;
      if (pitch < 0) pitch = pitch + 12;

      //Center the starting pitch about the melody center
      int center = Instrumentation.getIntMelodyCenter();
      if (pitch < (center - 5)) {pitch = pitch + 12;}
      else if (pitch > (center + 6)) {pitch = pitch - 12;}
      
      melEvent.pitch.setRelPitch(pitch);
      lastInterval.setInterval(0);
      assignLasts(melEvent, lastInterval);
    }
    return;
     
  }

/**
 *  For debugging.
 */
  public static void main(String[] args) {
    /*  KeyControl.setCurrentKeyType(0);
    KeyControl.setCurrentMode("major");
    DissonanceLevel.setDissonanceLevel(20.0f);
    DissonanceHandling.setDissonanceHandling(0);

    Chord chord1 = new Chord();
    inQueue.push(chord1);
    Chord chord2 = new Chord();
    inQueue.push(chord2);
    Chord chord3 = new Chord();
    inQueue.push(chord3);
    Chord chord4 = new Chord();
    inQueue.push(chord4);
    Chord chord5 = new Chord();
    inQueue.push(chord5);
    Chord chord6 = new Chord();
    inQueue.push(chord6);
    Chord chord7 = new Chord();
    inQueue.push(chord7);
    Chord chord8 = new Chord();
    inQueue.push(chord8);
    Chord chord9 = new Chord();
    inQueue.push(chord9);
    Chord chord10 = new Chord();
    inQueue.push(chord10);
    
    chord1.setMusicDefn("sentence start");

    System.out.println("Starting....");
    System.out.println(" ");
    for (int i = 1; i<=4; i++) {
    getNewChord();
    System.out.println(lastChord.chordType + " in the key of " 
          + lastChord.key);
    }

    Modulation.setModulation("minor");
    for (int i = 5; i<=7; i++) {
    getNewChord();
    System.out.println(lastChord.chordType + " in the key of " 
          + lastChord.key);
    }
  
    Cadence.setCadence("pre-cadence");
    for (int i = 8; i<=10; i++) {
    getNewChord();
    System.out.println(lastChord.chordType + " in the key of " 
          + lastChord.key);
    }
    
  }
  */
  }
}
