/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.pitch.melody;
import herman.elements.*;
import herman.pitch.tools.*;
import herman.pitch.harmony.*;
import herman.properties.*;
import herman.pitch.instruments.*;

/** 
*   Contains the pivot rules applicable to the melody
*/

public class MelodyPivotRules {

  private static float Yes = 1.0f;
  private static float no = -1000.0f;

   //only allows the octave
  private static final float[] octaveTone = 
  {Yes,no,no,no,no,no,no,no,no,no,no,no};
  
  //Indicates that the resolution to the octave has been delayed
  private static boolean delayOctave = false;

  //only allows the leading tone
  private static final float[] leadingTone = 
  {no,no,no,no,no,no,no,no,no,no,no,Yes};

  //prevents the leading tone
  private static final float[] noLead = 
  {Yes,Yes,Yes,Yes,Yes,Yes,Yes,Yes,Yes,Yes,Yes,no};

  //prevents the raised seventh, raised sixth, and tonic
  private static final float[] afterUnRaised7 = 
  {no,Yes,Yes,Yes,Yes,Yes,Yes,Yes,Yes,no,Yes,no};

  //prevents the raised sixth, raised 7 and unraised seventh
  private static final float[] afterUnRaised6 = 
  {Yes,Yes,Yes,Yes,Yes,Yes,Yes,Yes,Yes,no,no,no};

  //Indicates that the move to the leading tone has been delayed
  private static boolean delayLead = false;
  
  public static float[] applyPivotRules(Event lastMelody, float[] newTones) {
    float[] tempTones = new float[12];

    int lastKP = MelodyScale.getKeyPitch(lastMelody).getKeyPitch() % 12;
    if (lastKP < 0) lastKP = 12 + lastKP;


    //After unraised 7, must go down
    if (lastKP == 10) {
      ArrayTools.equalizeArrays(tempTones, newTones);
      ArrayTools.add2Array(tempTones, afterUnRaised7);
      if (zeroCheck(tempTones)) {
	ArrayTools.equalizeArrays(newTones, tempTones);
      }
    }
    //After unraised 6, no unraised 7, raised 7, or raised 6
    else if (lastKP == 8) {
      ArrayTools.equalizeArrays(tempTones, newTones);
      ArrayTools.add2Array(tempTones, afterUnRaised6);
      if (zeroCheck(tempTones)) {
	ArrayTools.equalizeArrays(newTones, tempTones);
      }
    }
    //After leading tone, must eventually go to octave.  Allowed to delay
    // until this is possible
    else if ((lastKP == 11)||(delayOctave)) {
      delayOctave = false;
      ArrayTools.equalizeArrays(tempTones, newTones);
      ArrayTools.add2Array(tempTones, octaveTone);
      if (zeroCheck(tempTones)) {
	ArrayTools.equalizeArrays(newTones, tempTones);
      } else {delayOctave = true;}    //delay resolution till next event
    } 

    //Can't go from octave to leading tone
    else if (lastKP == 0) {
      ArrayTools.equalizeArrays(tempTones, newTones);
      ArrayTools.add2Array(tempTones, noLead);
      if (zeroCheck(tempTones)) {
	ArrayTools.equalizeArrays(newTones, tempTones);
      }
    }

    //After raised sixth, must eventually go to leading tone.  Allowed to 
    // delay until this is possible
    else if ((lastKP == 9)||(delayLead)) {
      delayLead = false;
      ArrayTools.equalizeArrays(tempTones, newTones);
      ArrayTools.add2Array(tempTones, leadingTone);
      if (zeroCheck(tempTones)) {
	ArrayTools.equalizeArrays(newTones, tempTones);
      } else {delayLead = true;}     //delay leading tone till next event
    }
    
    return newTones;
  }

  // Check to see if at least one event probability > 0 
  private static boolean zeroCheck( float[] prob) {
    int greaterThanZeroFlag = 0;
    for (int i = 0; i < prob.length; i++)
      if (prob[i] >= 0.0f)
	greaterThanZeroFlag++;
    return (greaterThanZeroFlag > 0) ? true : false;
  }
}
