/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.pitch.melody;
import herman.properties.*;
import herman.pitch.tools.*;
import java.lang.Math;
/** 
 * The IntervalPreferences class contains the rules and methods that allow 
 *  specification of a preferred interval size.  This takes the form of 
 *  a gaussian normal distribution, where the centerpoint and standard
 *  deviation are paramters.
 *  <p>
 *  At the present time, the maximum interval size, in either direction, is a 
 *  M9.
 **/

public class IntervalPreference {

  //The "Not Possible Number"
  private static final float no = -1000.0f;
  private static final float Yes = 1.0f;

  //The centerpoint of the distribution
  private static Interval center = new Interval(1);

  //The standard deviation
  private static float standardDev = 2.0f;
  
  //The powerLevel
  private static float powerLevel = 1.0f;

  //The manual preference array
  private static float[] manualPrefer = new float[15];
 

  //The current preferences;
  private static float[] currentPreference = new float[15];

  public IntervalPreference() { /* Nothing */}

/** 
 *  Intialize the IntervalPreference class.
 */
  public static void initialize() {
    setPowerLevel(500.0f);
    //updateIntervalPreference();
  }
  
  private static void updateIntervalPreference() {
    double base = 1/Math.sqrt(2.0*Math.PI*Math.pow(standardDev,2.0));
    double exponentialBottom  = 2.0*Math.pow(standardDev,2.0);
    float[] normal = new float[15];
    float[] preferIntervals = new float[29];

    for (int i = 0; i<15; i++) {
      double exponentialTop = Math.pow((i - center.getIntInterval()),2.0);
      normal[i] = (float)(base*Math.exp(-exponentialTop/exponentialBottom));
    }
    
    


    ArrayTools.equalizeArrays(currentPreference, normal);
 
    //Set the negative intervals
    for (int i = 0; i<14; i++) {
      preferIntervals[i] = normal[14 - i];
    }
 
    //Set the positive (and zero) intervals:
    for (int i = 0; i<15; i++) {
      preferIntervals[14 + i] = normal[i];
    }

    //Normalize & multiply by the powerLevel
    ArrayTools.normalizeArray(preferIntervals);
    for (int i = 0; i<29; i++) {
      preferIntervals[i] = preferIntervals[i] * powerLevel;
    }
    
    //ArrayTools.printArray(preferIntervals);

    ArrayTools.equalizeArrays(MelodyProgression.preferredIntervals, 
			      preferIntervals);
  }

  private static void updateManualIntervalPreference() {
  
    float[] normal = manualPrefer;
    float[] preferIntervals = new float[29];


    ArrayTools.equalizeArrays(currentPreference, normal);

    //Set the negative intervals
    for (int i = 0; i<14; i++) {
      preferIntervals[i] = normal[14 - i];
    }
 
    //Set the positive (and zero) intervals:
    for (int i = 0; i<15; i++) {
      preferIntervals[14 + i] = normal[i];
    }

    //Normalize & multiply by the powerLevel
    ArrayTools.normalizeArray(preferIntervals);
    for (int i = 0; i<29; i++) {
      preferIntervals[i] = preferIntervals[i] * powerLevel;
    }
    
    //ArrayTools.printArray(preferIntervals);

    ArrayTools.equalizeArrays(MelodyProgression.preferredIntervals, 
			      preferIntervals);
  }

// Access Methods

/**
 *  Set the centerpoint interval
 */
  public static void setCenterInterval(Interval interval) {
    if (interval.getIntInterval() < 0) {
      interval.setInterval(Math.abs(interval.getIntInterval()));
    }
    center.setInterval(interval.getIntInterval());
    updateIntervalPreference();
  }

/**
 *  Set the standard deviation
 */
  public static void setStandardDeviation(float stanDev) {
    standardDev = stanDev;
    updateIntervalPreference();
  }

/**
 *  Set the power level
 */
  public static void setPowerLevel(float power) {
    powerLevel = power;
    updateIntervalPreference();
  }

/**
 *  Set the centerpoint interval and standard deviation
 */
  public static void setNormalFunction(Interval interval, float stanDev) {
    if (interval.getIntInterval() < 0) {
      interval.setInterval(Math.abs(interval.getIntInterval()));
    }
    center.setInterval(interval.getIntInterval());
    standardDev = stanDev;
    updateIntervalPreference();
  }

/**
 *  Set the centerpoint interval, standard deviation, & powerLevel
 */
  public static void setAll(Interval interval, float stanDev, float power) {
    if (interval.getIntInterval() < 0) {
      interval.setInterval(Math.abs(interval.getIntInterval()));
    }
    center.setInterval(interval.getIntInterval());
    standardDev = stanDev;
    powerLevel = power;
    updateIntervalPreference();
  }

/**
 *  Maually set the interval preference array.  The array is 15 floats
 * in length, with the index corresponding to the interval (in tones).  
 *  Example: prefer[1] refers to the m2 interval.
 */
  public static void setManualPreference(float[] prefer) {
    manualPrefer = prefer;
    updateManualIntervalPreference();
  }

/** 
 *  Get the current preference float[]
 */
  public static float[] getCurrentPreferences() {
    return currentPreference;
  }



 
/**
 *  For debugging.
 */
  public static void main(String[] args) {
  
  }
}
