/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.pitch.melody;
import herman.properties.*;
import herman.pitch.tools.*;
import java.lang.Math;
/** 
 * The IntervalHarshness class contains rules that modify the probability of
 * particular intervals being selected in the MelodyProgression module based 
 * on the harshness level parameter.
 **/

public class IntervalHarshness {

  //The "Not Possible Number"
  private static final float no = -1000.0f;
  private static final float Yes = 1.0f;

  //The harshness level
  private static float harshness = 0.0f;
  
  //The powerLevel
  private static float powerLevel = 100.0f;
 
  //The starting values for each interval:
  private static final float[] startValues = 
  {50.0f,                  // no change
   10.0f,25.0f,            //m2, M2
   35.0f,50.0f,            //m3, M3
   35.0f,10.0f,           //P4, A4
   50.0f,                  //P5,
   35.0f,35.0f,            //m6, M6
   25.0f,10.0f,           //m7, M7
   50.0f,                  //octave
   10.0f, 25.0f};          //m9, M9


  //The multipliers for each interval:
  private static final float[] multipliers = 
  {0.0f,                  // no change
   2.0f,0.75f,            //m2, M2
   0.3f,0.0f,            //m3, M3
   0.3f,1.0f,           //P4, A4
   0.0f,                  //P5,
   0.3f,0.3f,            //m6, M6
   0.75f,2.0f,           //m7, M7
   0.0f,                 //octave
   2.0f, 0.75f};          //m9, M9


  public IntervalHarshness() { /* Nothing */}

/**
 *  Initializes the IntervalHarshness class.
 */
  public static void initialize() {
    updateIntervalHarshness();
  }

  private static void updateIntervalHarshness() {
    
    float[] positive = new float[15];
    float[] hIntervals = new float[29];

    for (int i = 0; i<15; i++) {
      positive[i] = startValues[i] + multipliers[i]*harshness;
    }

    //Set the negative intervals
    for (int i = 0; i<14; i++) {
      hIntervals[i] = positive[14 - i];
    }
 
    //Set the positive (and zero) intervals:
    for (int i = 0; i<15; i++) {
      hIntervals[14 + i] = positive[i];
    }

    //Normalize & multiply by the powerLevel
    ArrayTools.normalizeArray(hIntervals);
    for (int i = 0; i<29; i++) {
      hIntervals[i] = hIntervals[i] * powerLevel;
    }
    
    //ArrayTools.printArray(hIntervals);

    ArrayTools.equalizeArrays(MelodyProgression.harshIntervals, hIntervals);
  }

// Access Methods


/**
 *  Set the harshness level.  
 * @param harsh the harshness level.  A float > 0.0f
 */
  public static void setHarshness(float harsh) {
    harshness = harsh;
    updateIntervalHarshness();
  }

/**
 *  Set the power level.  A float > 0.0f
 */
  public static void setPowerLevel(float power) {
    powerLevel = power;
    updateIntervalHarshness();
  }

/**
 *  Set the harshness and the power level
 */
  public static void setAll(float harsh, float power) {
    powerLevel = power;
    harshness = harsh;
    updateIntervalHarshness();
  }

 
/**
 *  For debugging.
 */
  public static void main(String[] args) {
  
  }
}
