/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.pitch.melody;
import herman.properties.*;
import herman.pitch.tools.*;
import herman.elements.*;
import herman.pitch.harmony.PivotRules;

/** 
 * The ChordTones class contains rules that relate chords to the individual
 *  pitches that compose those chords.
 **/

public class ChordTones {

  //The "Not Possible Number"
  private static final float no = -1000.0f;
  private static final float Yes = 1.0f;

  /* The matchAt percentage determines the ratio of melodic event size to 
   * chord size above which all melodic events *must* be matched to their
   * corresponding chords
   */
  private static float matchAt = 0.0f;
  private static float minMatchProb = 20.0f;
 
  //The maximum time (in milliseconds) that an event can overlap
  // with a chord without being forced to match a chord tone
  private static float mustMatchThreshold = 1000.0f;

  public ChordTones() { /* Nothing */}

  /**
   *  Returns true if the melody event must be instantiated to a tone of
   * the supplied chord (assumed to be associated with the melody event)
   */
  public static boolean mustBeAChordTone(Event mel, Chord chord) {
    long melT = mel.startTime;
    long melD = mel.duration;
    long chordT = chord.startTime;
    long chordD = chord.duration;
    long overlap = 0;
 
    if (melT < chordT) {                   //Melody starts first...
      if ((melT+melD)<(chordT+chordD)) {     //...and ends first
	overlap = melT+melD - chordT;
      } else {                                //...but chord ends first
	overlap = chordD;
      }
    } else {                              //Chord starts first
      if ((chordT+chordD)<(melT+melD)) {   //...and ends first
	overlap = chordT + chordD - melT;
      } else {                             //...but melody ends first
	overlap = melD;
      }
    }

    float overlapToChordRatio = ((float)overlap) /((float) chord.duration);
    if (overlapToChordRatio > 100.0f) overlapToChordRatio = 100.0f;
    float probInChord = 0.0f;

    //Calculate the probability the tone should be in the chord
    if (matchAt > 0.0f) {
      probInChord = minMatchProb + (100.0f - minMatchProb)/matchAt
                                           *overlapToChordRatio*100.0f;
    } else probInChord = 100.0f;

    //Is overlap bigger than threshold?
    boolean mustMatch = false;
    if (overlap > mustMatchThreshold) mustMatch = true;
      

    //Determine if should be in chord
    float rand = ((float)java.lang.Math.random())*100.0f;
  
    if ((mustMatch)||(rand <= probInChord)) return true;
    return false;
  }

  /**
   *  Checks to see if the event mustBeAChordTone, and if so, returns
   *  a float[] indicating which tones are possible (= 1.0) and which
   *  are impossible (= -1000.0).  Additionally, it makes sure that the
   *  status of the pivot tones in the chord and melody is the same.
   */
  public static float[] matchChordTones(Event mel, Chord chord) {
    float[] chordTones = new float[12];
    ArrayTools.setArrayValue(chordTones,Yes);
    
    if (mustBeAChordTone(mel,chord)) {             //If must be in the chord...
      ArrayTools.setArrayValue(chordTones,no);
      //  System.out.println("Must have chord Tones");
      int[] tones = getChordTones(chord);
      for (int i = 0; i<tones.length; i++) {
	chordTones[tones[i]] = Yes;
      }
    }
    else {                          //Even if not matching chord Tones
                                    // don't violate pivot rules
      //System.out.println("Not chord Tones");
      if (PivotRules.hasRaised7(chord)) {
	//System.out.println("Chord has Raised7th!!");
	chordTones[10] = no;
	chordTones[9] = no;
      }
      if (PivotRules.hasRaised6(chord)) {
	//System.out.println("Chord has Raised6th!!");
	chordTones[8] = no;
	chordTones[11] = no;
      }
      if (PivotRules.hasUnRaised7(chord)) {
	chordTones[11] = no;
	chordTones[9] = no;               
      }
      if (PivotRules.hasUnRaised6(chord)) {
	chordTones[9] = no;
      }
    }
    return chordTones;
  }

/**
 *  Returns an int[] listing the relative pitches that are tones in the
 *  supplied chord
 */
  public static int[] getChordTones(Chord chord) {
    String degree = chord.chordType.getStringDegree();
    String type = chord.chordType.getStringType();
    String extras = chord.chordType.getStringExtras();
    int deg = chord.chordType.getIntDegree();

    int[] triadTones = new int[3];
    int root = 0;

    //Intialize triad root:
    if (degree.equals("I")) root = 0;
    else if (degree.equals("II")) root = 2;
    else if (degree.equals("III")) {
      if (chord.key.getMode().equals("minor")) root = 3;
      else root = 4;
    }
    else if (degree.equals("IV")) root = 5;
    else if (degree.equals("V")) root = 7;
    else if (degree.equals("VI")) {
      if ((chord.key.getMode().equals("minor"))&&(type.equals("major"))) 
	                                      root = 8;
      else root = 9;
    }
    else if (degree.equals("VII")) {
      if ((chord.key.getMode().equals("minor"))&&(type.equals("major"))) 
	                                      root = 10;
      else root = 11;
    }

    triadTones[0] = root;

    //Put into type:
    if (type.equals("major")) {
      triadTones[1] = root + 4;
      triadTones[2] = root + 7;
    }
    else if (type.equals("minor")) {
      triadTones[1] = root + 3;
      triadTones[2] = root + 7;
    }
    else if (type.equals("diminished")) {
      triadTones[1] = root + 3;
      triadTones[2] = root + 6;
    }  
    else if (type.equals("augmented")) {
      triadTones[1] = root + 4;
      triadTones[2] = root + 8;
    }

    //Add extra tones
    int extraNum = 0;
    int[] extraTones = new int[2];
    if (extras.equals("m7")) {
      extraNum = 1;
      extraTones[0] = root + 10;
    }
    else if (extras.equals("M7")) {
      extraNum = 1;
      extraTones[0] = root + 11;
    }  
    else if (extras.equals("m9")) {
      extraNum = 1;
      extraTones[0] = root + 1;
    }
    else if (extras.equals("M9")) {
      extraNum = 1;
      extraTones[0] = root + 2;
    }
    else if (extras.equals("m7 & m9")) {
      extraNum = 2;
      extraTones[0] = root + 10;
      extraTones[1] = root + 1; 
    }  
    else if (extras.equals("M7 & m9")) {
      extraNum = 2;
      extraTones[0] = root + 1;
      extraTones[1] = root + 11;
    }
    else if (extras.equals("m7 & M9")) {
      extraNum = 2;
      extraTones[0] = root + 2;
      extraTones[1] = root + 10;
    }
    else if (extras.equals("M7 & M9")) {
      extraNum = 2;
      extraTones[0] = root + 2;
      extraTones[1] = root + 11;
    }

    int[] chordTones = new int[(extraNum + 3)];
    
    for (int i = 0; i<3; i++) {
      triadTones[i] = triadTones[i] % 12;
      chordTones[i] = triadTones[i];
    }
    for (int i = 0; i<extraNum; i++) {
      extraTones[i] = extraTones[i] % 12;
      chordTones[i+3] = extraTones[i];
    }
    return chordTones;
  }

/**
 *  Returns a float[][] describing which chords include the supplied pitch
 *    in the supplied key.  The float[][] has 28 columns (one per chord type)
 *    and 5 rows (one per extra tone).
 *      It uses scale knowledge, and relies on the scale to eliminate some 
 *      chords that don't share the pitch, but that aren't possible under
 *      the given key anyway.
 */
  public static float[][] getChords(Pitch pitch, Key key) {
    /* A float describing whether each chord type is possible or not.  It
       has 28 columns (one per chord type) and 5 rows (one per extra tone).
       It USES scale knowledge, and relies on the scale to eliminate some 
       chords that don't share the pitch, but that aren't possible under
       the given key
    */
    float[][] chords = new float[28][5];

    //Initialize so that nothing is possible
    for (int i = 0; i<28; i++) {
      ArrayTools.setArrayValue(chords[i], no);
    }

    //kp is the base scale tone of the pitch in the given key
    int kp = (pitch.getRelPitch() - key.getIntKeyType()) % 12;
    if (kp < 0 ) kp = kp + 12;

    int scaleD = MelodyScale.changeToScaleDegree(kp, key);
    int okChord = 0;

    if (key.getMode().equals("major")) {
    //Everything with scaleD as the root is OK
      for (int i = 0; i<4; i++) {
	ArrayTools.setArrayValue(chords[scaleD*4+i], Yes);
      }
      
      //Everything with scaleD as a third is OK
      okChord = (scaleD + 5) %7;            //6th up = 3rd down
	for (int i = 0; i<4; i++) {
	  ArrayTools.setArrayValue(chords[okChord*4 + i], Yes);  
	} 

      //Everything with scaleD as a fifth is OK
      okChord = (scaleD + 3) %7;            //4th up = 5th down
	for (int i = 0; i<4; i++) {
	  ArrayTools.setArrayValue(chords[okChord*4 + i], Yes);  
	}
      
      //A second up with the correct seventh is OK
      // (the scale ensures that the seventh is correct)
      okChord = (scaleD + 1) %7;            //2nd up
	for (int i = 0; i<4; i++) {
	  chords[okChord*4 + i][1]= Yes; 
	  chords[okChord*4 + i][2]= Yes; 
	}
    
      //A seventh up with the correct ninth is OK
      // (the scale ensures that the ninth is correct)
      okChord = (scaleD + 6) %7;            //7th up
	for (int i = 0; i<4; i++) {
	  chords[okChord*4 + i][3]= Yes; 
	  chords[okChord*4 + i][4]= Yes; 
	}
    }
    else {   // minor mode
      if (scaleD < 5) {        // If degree lower than VI (0-6 scale!) 
	//Everything with scaleD as the root is OK
	for (int i = 0; i<4; i++) {
	  ArrayTools.setArrayValue(chords[scaleD*4+i], Yes);
	}
	
	//Everything with scaleD as a third is OK
	okChord = (scaleD + 5) %7;            //6th up = 3rd down
	for (int i = 0; i<4; i++) {
	  ArrayTools.setArrayValue(chords[okChord*4 + i], Yes);  
	} 

	//Everything with scaleD as a fifth is OK
	okChord = (scaleD + 3) %7;            //4th up = 5th down
	for (int i = 0; i<4; i++) {
	  ArrayTools.setArrayValue(chords[okChord*4 + i], Yes);  
	}

	//A second up with the correct seventh is OK
	// (the scale ensures that the seventh is correct)
	okChord = (scaleD + 1) %7;            //2nd up
	  for (int i = 0; i<4; i++) {
	    chords[okChord*4 + i][1]= Yes; 
	    chords[okChord*4 + i][2]= Yes; 
	  }
    
	//A seventh up with the correct ninth is OK
	// (the scale ensures that the ninth is correct)
	okChord = (scaleD + 6) %7;            //7th up
	  for (int i = 0; i<4; i++) {
	    chords[okChord*4 + i][3]= Yes; 
	    chords[okChord*4 + i][4]= Yes; 
	  }
      }
      else {       //If the degree is VI (5&6) or VII (7&8)
	if (scaleD == 5) {
	  //VI major OK
	  ArrayTools.setArrayValue(chords[scaleD*4], Yes);
	
	//IV minor has scaleD as a third --> OK
	  ArrayTools.setArrayValue(chords[13], Yes);  
     
	//II dimin. has scaleD as a fifth --> OK
	  ArrayTools.setArrayValue(chords[6], Yes);  

	//A second up with the correct seventh is OK
	  chords[24][1]= Yes;                //VII major m7
       
	//A seventh up with the correct ninth is OK
	  chords[16][3]= Yes;              //V major m9
	  chords[17][3]= Yes;              //V minor m9
	}
	if (scaleD == 6) {
	  //VI diminished OK
	  ArrayTools.setArrayValue(chords[22], Yes);
	
	//IV major has scaleD as a third --> OK
	  ArrayTools.setArrayValue(chords[12], Yes);  
     
	//II minor has scaleD as a fifth --> OK
	  ArrayTools.setArrayValue(chords[5], Yes);  

	//A second up with the correct seventh is OK
	  chords[24][2]= Yes;                //VII major M7
	  chords[26][1]= Yes;                //VII dim. m7

	//A seventh up with the correct ninth is OK
	  chords[16][4]= Yes;              //V major M9
	  chords[17][4]= Yes;              //V minor M9
	}
	if (scaleD == 7) {
	  //VII major OK
	  ArrayTools.setArrayValue(chords[24], Yes);
	
	//V minor has scaleD as a third --> OK
	  ArrayTools.setArrayValue(chords[17], Yes);  
     
	//III major has scaleD as a fifth --> OK
	  ArrayTools.setArrayValue(chords[8], Yes);  

	//A second up with the correct seventh is OK
	  chords[1][1]= Yes;                //I minor m7

	//A seventh up with the correct ninth is OK
	  chords[20][4]= Yes;              //VI major M9
	}
	if (scaleD == 8) {
	  //VII diminished OK
	  ArrayTools.setArrayValue(chords[26], Yes);
	
	//V major has scaleD as a third --> OK
	  ArrayTools.setArrayValue(chords[16], Yes);  
     
	//III augmented has scaleD as a fifth --> OK
	  ArrayTools.setArrayValue(chords[11], Yes);  

	//A second up with the correct seventh is OK
	  chords[1][2]= Yes;                //I minor M7

	//A seventh up with the correct ninth is OK
	  chords[22][4]= Yes;              //VI diminished M9
	}
      }
	
    }
    return chords;
  }

// Access Methods

/**
 *  Set the maximum percentage of chord duration that a melody event
 *     can be a non-chord tone. 
 */
  public static void setMaxNonChordTonePercent(float percent) {
    matchAt = percent;
  }

/**
 *  Set the minimum chord tone matching probability.  This is the <em>
 *  minimum </em> probability that any melody event will match a chord tone.
 */
  public static void setMinMatchProb(float prob) {
    minMatchProb = prob;
  }

/**
 *  Set the maximum time (in milliseconds) that an event may overlap a chord
 *    without being forced to match a chord tone
 */
  public static void setMustMatchThreshold(float time) {
    mustMatchThreshold = time;
  }

/**
 *  For debugging.
 */
  public static void main(String[] args) {
  Chord chord = new Chord();
  chord.chordType.setDegree("V");
  chord.chordType.setType("minor");
  chord.chordType.setExtras("M7");
  chord.key.setKeyType(0);
  chord.key.setMode("major");

  System.out.print("{");
  int[] chordTones = getChordTones(chord);
  for (int i = 0; i<chordTones.length; i++) {
    System.out.print(chordTones[i] + ", "); 
  }
  System.out.println("}");
  }
}
