/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.pitch.harmony;
import herman.elements.*;
import herman.pitch.tools.*;

/** 
 * The PivotRules class stores and applies the rules governing the use of 
 *   pivot tones (i.e., the raised 6th and 7th) in the minor mode.  
 **/

public class PivotRules {

/**
 *  pivotRules is a boolean describing whether or not the rules should be 
 *    applied
 */
  public static boolean pivotRules = true;

  //The "Not Possible Number"
  private static final float no = -1000.0f;
  private static final float Yes = 1.0f;


  public PivotRules() { /* Nothing */}

/**
 *  Returns true if the supplied chord has a raised 7th
 */
  public static boolean hasRaised7(Chord chord) {
    String lDeg = chord.chordType.getStringDegree();
    String lEx = chord.chordType.getStringExtras();
    String lTy = chord.chordType.getStringType();

    int lExtra = chord.chordType.getIntExtras();
    if (((lDeg.equals("III"))&&(lTy.equals("augmented")))||
	    ((lDeg.equals("V"))&&(lTy.equals("major")))||
	    ((lDeg.equals("VII"))&&(lTy.equals("diminished")))||
	 ((lDeg.equals("I"))&&((lEx.equals("M7"))||(lExtra==6)||(lExtra==8)))||
	                         // I with a M7
	    ((lDeg.equals("VI"))&&(lTy.equals("diminished"))&&
	                ((lEx.equals("M9"))||(lExtra>6)))) {
      return true;
    }
    return false;
  }

/**
 *  Returns true if the supplied chord has a raised 6th
 */
  public static boolean hasRaised6(Chord chord) {
    String lDeg = chord.chordType.getStringDegree();
    String lEx = chord.chordType.getStringExtras();
    String lTy = chord.chordType.getStringType();

    int lExtra = chord.chordType.getIntExtras();
    if (((lDeg.equals("II"))&&(lTy.equals("minor")))||
	    ((lDeg.equals("IV"))&&(lTy.equals("major")))||
	    ((lDeg.equals("VI"))&&(lTy.equals("diminished")))||
	    ((lDeg.equals("V"))&&((lEx.equals("M9"))||(lExtra>6)))||
		                       //V major M9 not poss.
	    ((lDeg.equals("VII"))&&((lEx.equals("M7"))||(lExtra==6)||
				    (lExtra==8)))) { 
                          //VII dim. M7 not poss.
      return true;
    }
    return false;
  }

/**
 *  Returns true if the supplied chord has an unraised 6th
 */
  public static boolean hasUnRaised6(Chord chord) {
    String lDeg = chord.chordType.getStringDegree();
    String lEx = chord.chordType.getStringExtras();
    String lTy = chord.chordType.getStringType();

    int lExtra = chord.chordType.getIntExtras();
    if (((lDeg.equals("II"))&&(lTy.equals("diminished")))||
	    ((lDeg.equals("IV"))&&(lTy.equals("minor")))||
	    ((lDeg.equals("VI"))&&(lTy.equals("major")))||
      ((lDeg.equals("V"))&&((lEx.equals("m9"))||(lExtra==5)||(lExtra==6))&&
               (lTy.equals("minor")))|| 
      ((lDeg.equals("VII"))&&((lEx=="m7")||(lExtra==5)||(lExtra==7))&&
             (lTy.equals("major")))) {
      return true;
    }
    return false;
  }

/**
 *  Returns true if the supplied chord has an unraised 7th
 */
  public static boolean hasUnRaised7(Chord chord) {
    String lDeg = chord.chordType.getStringDegree();
    String lEx = chord.chordType.getStringExtras();
    String lTy = chord.chordType.getStringType();

    int lExtra = chord.chordType.getIntExtras();
    if (((lDeg.equals("III"))&&(lTy.equals("major")))||
	    ((lDeg.equals("V"))&&(lTy.equals("minor")))||
	    ((lDeg.equals("VII"))&&(lTy.equals("major")))||
	    ((lDeg.equals("I"))&&((lEx.equals("m7"))||(lExtra==5)||
				  (lExtra==7)))||
	    ((lDeg.equals("VI"))&&(lTy.equals("major"))&&((lEx.equals("M9"))||
						    (lExtra>6)))) {
      return true;
    }
    return false;
  }

/**  
 *  Applies the pivot rules when in a minor key.  Works by affecting the
 *   pivotRules array in the HarmonyProgression module, thereby limiting
 *   the possible new chords to ones that are possible under the pivot
 *   rules.
 */

  public static void applyPivotRules(Chord lastChord) {
 
    String lDeg = lastChord.chordType.getStringDegree();
    String lEx = lastChord.chordType.getStringExtras();
    String lTy = lastChord.chordType.getStringType();

    int lExtra = lastChord.chordType.getIntExtras();

    float[] pivotArray = HarmonyProgression.pivotRules;
   
 
    if (((pivotRules)&&((KeyControl.getCurrentMode()).equals("minor")))&& 
                     //If the pivot rules apply...
       ((!(lDeg.equals("I")))||(lEx.equals("m7"))||(lEx.equals("M7"))||
	                                       (lExtra>4))) { 
             //...and if the degree is not I, or is I with a 7th
	 ArrayTools.setArrayValue(pivotArray,no);

	/* Raised Sevenths go to octave (no raised tones) or to raised 7th*/
	if (((lDeg.equals("III"))&&(lTy.equals("augmented")))||
	    ((lDeg.equals("V"))&&(lTy.equals("major")))||
	    ((lDeg.equals("VII"))&&(lTy.equals("diminished")))||
	 ((lDeg.equals("I"))&&((lEx.equals("M7"))||(lExtra==6)||(lExtra==8)))||
	                         // I with a M7
	    ((lDeg.equals("VI"))&&(lTy.equals("diminished"))&&
	                ((lEx.equals("M9"))||(lExtra>6)))) {
	  pivotArray[1] = Yes;     //I minor OK (octave)
	  pivotArray[13] = Yes;      //IV minor OK (octave)
	  pivotArray[20] = Yes;      // VI major OK  (octave)
	  pivotArray[11] = Yes;      //III augmented OK (R7)

	  pivotArray[16] = Yes;     //V major OK (R7)
	  pivotArray[26] = Yes;     //VII diminished OK (R7)

	  if ( ExtraTonesControl.force(6,"m7")) {  //If IIdim can be set to m7
	    pivotArray[6] = Yes;     //II diminished OK... (octave)
	  }
	  if (ExtraTonesControl.force(24,"M9")) {   //If VIIM can be set to M9
	    pivotArray[24] = Yes;     //VII major OK... (octave)
	  }
	}

	/* Raised 6ths go to raised 7th  or to raised 6th*/
	else if (((lDeg.equals("II"))&&(lTy.equals("minor")))||
	    ((lDeg.equals("IV"))&&(lTy.equals("major")))||
	    ((lDeg.equals("VI"))&&(lTy.equals("diminished")))||
	    ((lDeg.equals("V"))&&((lEx.equals("M9"))||(lExtra>6)))||
		                       //V major M9 not poss.
	    ((lDeg.equals("VII"))&&((lEx.equals("M7"))||(lExtra==6)||
				    (lExtra==8)))) { 
                          //VII dim. M7 not poss.
	  pivotArray[5] = Yes;     //II minor OK (R6)
	  pivotArray[12] = Yes;      //IV major OK (R6)
	  pivotArray[22] = Yes;     //VI diminished OK (R6)
	 
	  pivotArray[11] = Yes;      //III augmented OK (R7)
	  pivotArray[16] = Yes;     //V major OK (R7)
	  pivotArray[26] = Yes;     //VII diminished OK (R7)

	  if ( ExtraTonesControl.force(1,"M7")) {  //If Im can be set to M7
	    pivotArray[1] = Yes;     //I minor OK... (R7)
	  }
	  if (ExtraTonesControl.force(17,"M9")) {   //If Vm can be set to M9
	    pivotArray[17] = Yes;     //V minor OK... R6
	  }
	  if (ExtraTonesControl.force(24,"M7")) {   //If VIIM can be set to M7
	    pivotArray[24] = Yes;     //VII major OK... (R6)
	  }
	}
	
        /* Unraised 6ths go to raised 7th, unraised 6th, or octave*/
	else if (((lDeg.equals("II"))&&(lTy.equals("diminished")))||
	    ((lDeg.equals("IV"))&&(lTy.equals("minor")))||
	    ((lDeg.equals("VI"))&&(lTy.equals("major")))||
      ((lDeg.equals("V"))&&((lEx.equals("m9"))||(lExtra==5)||(lExtra==6))&&
               (lTy.equals("minor")))|| 
      ((lDeg.equals("VII"))&&((lEx=="m7")||(lExtra==5)||(lExtra==7))&&
             (lTy.equals("major")))) {

	  pivotArray[6] = Yes;     //II diminished OK (U6)
	  pivotArray[13] = Yes;      //IV minor OK (U6)
	  pivotArray[20] = Yes;     //VI major OK (U6)
	 
	  pivotArray[1] = Yes;      //I minor OK (octave)
	  pivotArray[11] = Yes;     //III augmented OK (R7)
	  pivotArray[16] = Yes;     //V major OK (R7)
	  pivotArray[26] = Yes;     //VII diminshed OK (R7)

	  if ( ExtraTonesControl.force(17,"m9")) {  //If Vm can be set to m9
	    pivotArray[17] = Yes;     //V minor OK... (U6)
	  }
	  if (ExtraTonesControl.zeroProb(24,"M7")) {   //If VIIM not M7
	    pivotArray[24] = Yes;     //VII major OK... (U6, octave)
	  }
	}

        /* Unraised 7ths go to U6 or to U7 */
	else if (((lDeg.equals("III"))&&(lTy.equals("major")))||
	    ((lDeg.equals("V"))&&(lTy.equals("minor")))||
	    ((lDeg.equals("VII"))&&(lTy.equals("major")))||
	    ((lDeg.equals("I"))&&((lEx.equals("m7"))||(lExtra==5)||
				  (lExtra==7)))||
	    ((lDeg.equals("VI"))&&(lTy.equals("major"))&&((lEx.equals("M9"))||
						    (lExtra>6)))) {
	  pivotArray[8] = Yes;       //III major OK (U7)
	  if (ExtraTonesControl.zeroProb(17,"M9")) {   //If Vm not M9
	    pivotArray[17] = Yes;      //V minor OK (U7
	  }
	 if (ExtraTonesControl.zeroProb(7,"M7")) {   //If VIIM not M7
	    pivotArray[24] = Yes;      //VII major OK  (U7)
	  }

	  pivotArray[6] = Yes;     //II diminished OK (U6)
	  pivotArray[13] = Yes;     //IV minor OK (U6)
          pivotArray[20] = Yes;     //VI major OK (U6)

	  if ( ExtraTonesControl.force(1,"m7")) {  //If Im can be set to m7
	    pivotArray[1] = Yes;     //I minor OK... (U7)
	  }
	}
      
    }
    else {ArrayTools.setArrayValue(pivotArray,Yes);}
  }
// Access Methods

/**
 *  For debugging.
 */
  public static void main(String[] args) {
   
  }
 
}
