/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.pitch.harmony;
import herman.elements.*;
import herman.pitch.tools.*;
/** 
 * The Modulation class handles the rules governing modulation from key
 *   to key. Possibilities include: major-major, minor-minor, major-minor,
 *   and minor-major.
 *  <p>
 *  At the present times, the Modulation class only switches to the new
 *   key; it does not force a cadence or drive the use of modulatory chords
 **/

public class Modulation {
  private static float STRAIGHT = 0.40f;
  private static float UP = 0.70f;
  private static float OFFSET = 0.15f;
  private static float no = -1000.0f;
  private static float Yes = 1.0f;

  private static boolean modulationFlag = false;
  private static StringBuffer modulation = new StringBuffer("major");
  
  private static float[] noRaisedTones = 
  {Yes,Yes,Yes,Yes,           //I OK
   Yes,no,Yes,Yes,            //II minor bad
   Yes,Yes,Yes,no,            //III augmented bad
   no,Yes,Yes,Yes,            //IV major bad
   no,Yes,Yes,Yes,            //V major bad
   Yes,Yes,no,Yes,            //VI diminished bad
   Yes,Yes,no,Yes};            //VII diminished bad
  
  private static boolean modulateMajorMinor( Chord lastChord) {
    String deg = lastChord.chordType.getStringDegree();
    String ex = lastChord.chordType.getStringExtras();
    float rand = ((float)java.lang.Math.random());

    if (((deg.equals("I"))&&(!ex.equals("M7")))||
	  ((deg.equals("VI"))&&(!ex.equals("M9")))) {
      if (rand < STRAIGHT) {
	shiftMajorToStraightMinor(lastChord);
      }
      else if (rand < UP) {
	shiftMajorToUpMinor(lastChord);
      }
      else {shiftMajorToDownMinor(lastChord);}
    }

    else if (((deg.equals("III"))&&(!ex.equals("m9")))||
	  ((deg.equals("V"))&&(!ex.equals("m7")))||
	    (deg.equals("I"))||(deg.equals("VI"))) {
      if (rand < STRAIGHT) {
	shiftMajorToStraightMinor(lastChord);
      }
      else {
	shiftMajorToUpMinor(lastChord);
      }
    }

    else if ((deg.equals("II"))||
	  (deg.equals("IV"))) {
      if (rand < STRAIGHT) {
	shiftMajorToStraightMinor(lastChord);
      }
      else {
	shiftMajorToDownMinor(lastChord);
      }
    }

    else {shiftMajorToStraightMinor(lastChord);}
    return true;
  }

  private static boolean modulateMajorMajor( Chord lastChord) {
    String deg = lastChord.chordType.getStringDegree();
    String ex = lastChord.chordType.getStringExtras();
    float rand = ((float)java.lang.Math.random());

    if (((deg.equals("I"))&&(!ex.equals("M7")))||
	  ((deg.equals("VI"))&&(!ex.equals("M9")))) {
      if (rand < UP*2) {
	shiftMajorToUpMajor(lastChord);
      }
      else  {
	shiftMajorToDownMajor(lastChord);
      }
    }

    else if (((deg.equals("III"))&&(!ex.equals("m9")))||
	  ((deg.equals("V"))&&(!ex.equals("m7")))||
	    (deg.equals("I"))||(deg.equals("VI"))) {
     
	shiftMajorToUpMajor(lastChord);
    }

    else if ((deg.equals("II"))||
	  (deg.equals("IV"))) {
	shiftMajorToDownMajor(lastChord);
    }

    else {return false;}
    return true;
  }

  private static boolean modulateMinorMajor(Chord lastChord) {
    String deg = lastChord.chordType.getStringDegree();
    String type = lastChord.chordType.getStringType();
    String ex = lastChord.chordType.getStringExtras();
    int extra = lastChord.chordType.getIntExtras();
    float rand = ((float)java.lang.Math.random());
   
    if (((deg.equals("II"))&&(type.equals("minor")))||
	((deg.equals("III"))&&(type.equals("augmented")))||
	((deg.equals("IV"))&&(type.equals("major")))||
	((deg.equals("V"))&&(type.equals("major")))||
	((deg.equals("VI"))&&(type.equals("diminished")))||
	((deg.equals("VII"))&&(type.equals("diminished")))||
	((deg.equals("I"))&&((ex.equals("M7"))||(extra==6)||(extra==8)))||
	((deg.equals("V"))&&((ex.equals("M9"))||(extra==7)||(extra==8)))||
	((deg.equals("VII"))&&((ex.equals("M7"))||(extra==6)||(extra==8)))) {
      
      //If the last chord had a raised tone...
      //...and if the sixth was raised
        if (((deg.equals("II"))&&(type.equals("minor")))||
	    ((deg.equals("IV"))&&(type.equals("major")))||
	    ((deg.equals("VI"))&&(type.equals("diminished")))||
	    ((deg.equals("V"))&&((ex.equals("M9"))||(extra>6)))||
	    ((deg.equals("VII"))&&((ex.equals("M7"))||(extra==6)||
				    (extra==8)))) { 
                         
	  //Make raised 6th impossible for next chord
	  HarmonyProgression.modulate[5] = no;   //II minor no (R6)
	  HarmonyProgression.modulate[12] = no;   //IV major no (R6)
	  HarmonyProgression.modulate[22] = no;   //VI diminished no (R6)
	 
	  if (ExtraTonesControl.zeroProb(17,"M9")) { //If Vm can be set to M9
	    HarmonyProgression.modulate[17] = Yes;     //V minor OK... R6
	  }
	  else {HarmonyProgression.modulate[17] = no;}

	  if (ExtraTonesControl.zeroProb(24,"M7")) { //If VIIM can be set to M7
	    HarmonyProgression.modulate[24] = Yes;  //VII major OK... (R6)
	  }
	  else {HarmonyProgression.modulate[24] = no;}
	}

	//Or if the seventh was raised...
	else {
	  //Make raised tones impossible
     ArrayTools.equalizeArrays(noRaisedTones,HarmonyProgression.modulate);
     ExtraTonesControl.zeroProb(1,"M7");
     ExtraTonesControl.zeroProb(17,"M9");
     ExtraTonesControl.zeroProb(24,"M7");
	}
     //Failure to modulate this turn..
     return false;
    }

    //Otherwise, last chord was unraised, so...
    //Shift to straight major...
    shiftMinorToStraightMajor(lastChord);

    //...and either stay here or modulate Major-Major
    if (rand > (STRAIGHT - OFFSET)) {
      modulateMajorMajor(lastChord);
    }
    
    //Success!
    return true;
  }
  
  private static boolean modulateMinorMinor(Chord lastChord) {
    String deg = lastChord.chordType.getStringDegree();
    String type = lastChord.chordType.getStringType();
    String ex = lastChord.chordType.getStringExtras();
    int extra = lastChord.chordType.getIntExtras();
    float rand = ((float)java.lang.Math.random());
   
    if ((deg.equals("II"))||
	((deg.equals("III"))&&(type.equals("augmented")))||
	((deg.equals("IV"))&&(type.equals("major")))||
	((deg.equals("V"))&&(type.equals("major")))||
	((deg.equals("VI"))&&(type.equals("diminished")))||
	((deg.equals("VII"))&&(type.equals("diminished")))||
	((deg.equals("I"))&&((ex.equals("M7"))||(extra==6)||(extra==8)))||
	((deg.equals("V"))&&((extra>2)))||
	((deg.equals("VII"))&&((extra==1)||(extra==2)||(extra==5)||
				(extra==6)))) {
      
      //If the last chord had a raised tone OR was the II, OR v9 OR VII7
      //...and if the sixth was raised
        if (((deg.equals("II"))&&(type.equals("minor")))||
	    ((deg.equals("IV"))&&(type.equals("major")))||
	    ((deg.equals("VI"))&&(type.equals("diminished")))||
	    ((deg.equals("V"))&&((ex.equals("M9"))||(extra>6)))||
	    ((deg.equals("VII"))&&((ex.equals("M7"))||(extra==6)||
				    (extra==8)))) { 
                         
	  //Make raised 6th impossible for next chord
	  HarmonyProgression.modulate[5] = no;   //II minor no (R6)
	  HarmonyProgression.modulate[6] = no;   // No II period!!
	  HarmonyProgression.modulate[12] = no;   //IV major no (R6)
	  HarmonyProgression.modulate[22] = no;   //VI diminished no (R6)
	 
	}
	//Or if the seventh was raised...(OR chord was a II)
	else {
	  //Make raised tones impossible
     ArrayTools.equalizeArrays(noRaisedTones,HarmonyProgression.modulate);
     HarmonyProgression.modulate[5] = no;   //No II chords at all
     ExtraTonesControl.zeroProb(1,"M7");
     ExtraTonesControl.zeroProb(17,"M9");
     ExtraTonesControl.zeroProb(24,"M7");
     ExtraTonesControl.zeroProb(17,"m9");       //No v9
     ExtraTonesControl.zeroProb(24,"m7");       //No VII 7
	}
     //Failure to modulate this turn..
     return false;
    }

    //Otherwise, last chord was unraised and was not a II or v9 or VII7, so...
      //Shift to straight major...
      shiftMinorToStraightMajor(lastChord);
      
      //Set STRAIGHT possibility to zero and UP to the middle
      float oldStraight = STRAIGHT;
      float oldUP = UP;
      STRAIGHT = 0.0f;
      UP = 0.50f;

      //...and modulate Major-minor
      modulateMajorMinor(lastChord);
     
      //Set everything back
      STRAIGHT = oldStraight;
      UP = oldUP;  
    
      //Success!
      return true;
  }

  private static void shiftMinorToStraightMajor(Chord lastChord) {
    //Set Key to straight major
      int newKey = (KeyControl.getIntCurrentKeyType()+3) % 12;
      KeyControl.setCurrentKeyType(newKey);
      KeyControl.setCurrentMode("major");

   //Set new degree to a sixth up
     int newDegree = (((lastChord.chordType.getIntDegree() - 1)+5)%7)+1;
      lastChord.chordType.setDegree(newDegree);

   //Set new type
      if ((newDegree==1)||(newDegree==4)||(newDegree==5)) {
	lastChord.chordType.setType(0);
      }
      else if ((newDegree==2)||(newDegree==3)||(newDegree==6)) {
	lastChord.chordType.setType(0);
      }
      else {lastChord.chordType.setType(2);}
  }

  private static void shiftMajorToStraightMinor(Chord lastChord) {
    //Set Key to straight minor
      int newKey = (KeyControl.getIntCurrentKeyType()+9) % 12;
      KeyControl.setCurrentKeyType(newKey);
      KeyControl.setCurrentMode("minor");

   //Set new degree to a third up
     int newDegree = (((lastChord.chordType.getIntDegree() - 1)+2)%7)+1;
      lastChord.chordType.setDegree(newDegree);

   //Set new type
      if ((newDegree==1)||(newDegree==4)||(newDegree==5)) {
	lastChord.chordType.setType(1);
      }
      else if ((newDegree==3)||(newDegree==6)||(newDegree==7)) {
	lastChord.chordType.setType(0);
      }
      else {lastChord.chordType.setType(2);}
    }

  private static void shiftMajorToUpMinor(Chord lastChord) {
    //Set Key to up minor
      int newKey = (KeyControl.getIntCurrentKeyType()+4) % 12;
      KeyControl.setCurrentKeyType(newKey);
      KeyControl.setCurrentMode("minor");

   //Set new degree to a 6th up
     int newDegree = (((lastChord.chordType.getIntDegree() - 1)+5)%7)+1;
      lastChord.chordType.setDegree(newDegree);

   //Set new type
      if ((newDegree==1)||(newDegree==4)||(newDegree==5)) {
	lastChord.chordType.setType(1);
      }
      else if ((newDegree==3)||(newDegree==6)||(newDegree==7)) {
	lastChord.chordType.setType(0);
      }
      else {lastChord.chordType.setType(2);}
  }
  
  private static void shiftMajorToUpMajor(Chord lastChord) {
    //Set Key to up major
      int newKey = (KeyControl.getIntCurrentKeyType()+7) % 12;
      KeyControl.setCurrentKeyType(newKey);
      KeyControl.setCurrentMode("major");

   //Set new degree to a 4th up
     int newDegree = (((lastChord.chordType.getIntDegree() - 1)+3)%7)+1;
      lastChord.chordType.setDegree(newDegree);

   //Set new type
      if ((newDegree==1)||(newDegree==4)||(newDegree==5)) {
	lastChord.chordType.setType(0);
      }
      else if ((newDegree==3)||(newDegree==6)||(newDegree==2)) {
	lastChord.chordType.setType(1);
      }
      else {lastChord.chordType.setType(2);}
  }

  private static void shiftMajorToDownMinor(Chord lastChord) {
    //Set Key to down minor
      int newKey = (KeyControl.getIntCurrentKeyType()+2) % 12;
      KeyControl.setCurrentKeyType(newKey);
      KeyControl.setCurrentMode("minor");

   //Set new degree to a 7th up
     int newDegree = (((lastChord.chordType.getIntDegree() - 1)+6)%7)+1;
      lastChord.chordType.setDegree(newDegree);

   //Set new type
      if ((newDegree==1)||(newDegree==4)||(newDegree==5)) {
	lastChord.chordType.setType(1);
      }
      else if ((newDegree==3)||(newDegree==6)||(newDegree==7)) {
	lastChord.chordType.setType(0);
      }
      else {lastChord.chordType.setType(2);}
  }
  
  private static void shiftMajorToDownMajor(Chord lastChord) {
    //Set Key to down major
      int newKey = (KeyControl.getIntCurrentKeyType()+5) % 12;
      KeyControl.setCurrentKeyType(newKey);
      KeyControl.setCurrentMode("major");

   //Set new degree to a 5th up
     int newDegree = (((lastChord.chordType.getIntDegree() - 1)+4)%7)+1;
      lastChord.chordType.setDegree(newDegree);

   //Set new type
      if ((newDegree==1)||(newDegree==4)||(newDegree==5)) {
	lastChord.chordType.setType(0);
      }
      else if ((newDegree==2)||(newDegree==3)||(newDegree==6)) {
	lastChord.chordType.setType(1);
      }
      else {lastChord.chordType.setType(2);}
  }

// Access Methods

/**
 *  Applies the modulation rules.  The key will be modulated to the mode
 *    specified in the "modulation" String.
 *@param lastChord  the last instantiated chord
 */
  public static void modulate(Chord lastChord) {
    boolean result = false;
    if (modulation.toString().equals("minor")) {
      if (KeyControl.getCurrentMode().equals("minor")) {
	result = modulateMinorMinor(lastChord);
      }
      else {result = modulateMajorMinor(lastChord);}
    }
    else if (modulation.toString().equals("major")) {
      if (KeyControl.getCurrentMode().equals("minor")) {
	result = modulateMinorMajor(lastChord);
      }
      else {result = modulateMajorMajor(lastChord);}
    }
    
    //Clean up
    if (result) {
      modulationFlag = false;
    }
  }

/**
 *  Begins a new modulation.  Modulates to the mode described in the String
 *  argument (major or minor).
 *
 *@param mode  A String specifying the desired resultant mode.
 */
  public static void setModulation(String mode) {
    modulation.insert(0,mode);
    modulation.setLength(mode.length());
    modulationFlag = true;
  }

/** 
 *  Return the status of the boolean modulateFlag.  If "true", a modulation is
 *    in progress.
 */
   public static boolean checkFlag() {
    if (modulationFlag) {return true;}
    return false;
  }

/** 
 *  Over-ride a modulation and force it to stop
 */
   public static void stopModulation() {
    modulationFlag = false;
  }
/**
 *  For debugging.
 */
  public static void main(String[] args) {

  }
}
