/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.pitch.harmony;
import herman.properties.*;
import herman.pitch.tools.*;
/** 
 * The KeyControl class stores the current musical key for the Pitch 
 *  Instatiator and uses that to set the scales that are used by the
 *  Pitch sub-modules.  KeyControl represents the current key with a 
 *  Key() object.
 *  <p>
 *  @see Key
 **/

public class KeyControl {

  // currentKey stores the current Key ! :) 
  private static Key currentKey = new Key();
  private static Key lastKey = new Key();

  //The "Not Possible Number"
  private static final float no = -1000.0f;
  private static final float Y = 1.0f;

  // The basic Chord types for a major scale as represented in the 
  //    Harmony Progression module:
  private static final float[] majorScaleChords =
  {Y,no,no,no,         // I   Major
   no,Y,no,no,         // II  minor
   no,Y,no,no,         // III minor
   Y,no,no,no,         // IV  Major
   Y,no,no,no,         // V   Major
   no,Y,no,no,         // VI  minor
   no,no,Y,no};         // VII diminished

  // The basic Chord types for a minor scale as represented in the 
  //    Harmony Progression module:
  private static final float[] minorScaleChords =
  {no,Y,no,no,          // I   minor
   no,Y,Y,no,           // II  diminished, raised => minor
   Y,no,no,Y,           // III Major, raised => augmented
   Y,Y,no,no,           // IV  minor, raised => Major
   Y,Y,no,no,           // V   minor, raised => Major
   Y,no,Y,no,           // VI  Major, raised => diminished
   Y,no,Y,no};           // VII Major, raised => diminished

  // The basic extra tones possibilities in a major scale:
  private static final float[][] majorScaleExtras =
  {{no,Y,no,Y,no,no,no,Y},   //I has M7 and M9
   {no,Y,no,Y,no,no,no,Y},
   {no,Y,no,Y,no,no,no,Y},
   {no,Y,no,Y,no,no,no,Y},
   {Y,no,no,Y,no,no,Y,no},   //II has m7 and M9
   {Y,no,no,Y,no,no,Y,no},
   {Y,no,no,Y,no,no,Y,no},
   {Y,no,no,Y,no,no,Y,no},
   {Y,no,Y,no,Y,no,no,no},   //III has m7 and m9
   {Y,no,Y,no,Y,no,no,no},
   {Y,no,Y,no,Y,no,no,no},
   {Y,no,Y,no,Y,no,no,no},
   {no,Y,no,Y,no,no,no,Y},   //IV has M7 and M9
   {no,Y,no,Y,no,no,no,Y},
   {no,Y,no,Y,no,no,no,Y},
   {no,Y,no,Y,no,no,no,Y},
   {Y,no,no,Y,no,no,Y,no},   //V has m7 and M9
   {Y,no,no,Y,no,no,Y,no},
   {Y,no,no,Y,no,no,Y,no},
   {Y,no,no,Y,no,no,Y,no},
   {Y,no,no,Y,no,no,Y,no},   //VI has m7 and M9
   {Y,no,no,Y,no,no,Y,no},
   {Y,no,no,Y,no,no,Y,no},
   {Y,no,no,Y,no,no,Y,no},
   {Y,no,Y,no,Y,no,no,no},   //VII has m7 and m9
   {Y,no,Y,no,Y,no,no,no},
   {Y,no,Y,no,Y,no,no,no},
   {Y,no,Y,no,Y,no,no,no}};

  // The basic extra tones possibilities in a minor scale:
  private static final float[][] minorScaleExtras =
  {{Y,Y,no,Y,no,no,Y,Y},   //I has m7,M7, & M9
   {Y,Y,no,Y,no,no,Y,Y},
   {Y,Y,no,Y,no,no,Y,Y},
   {Y,Y,no,Y,no,no,Y,Y},
   {Y,no,Y,no,Y,no,no,no},   //II has m7 and m9
   {Y,no,Y,no,Y,no,no,no},
   {Y,no,Y,no,Y,no,no,no},
   {Y,no,Y,no,Y,no,no,no},
   {no,Y,no,Y,no,no,no,Y},   //III has M7 and M9
   {no,Y,no,Y,no,no,no,Y}, 
   {no,Y,no,Y,no,no,no,Y}, 
   {no,Y,no,Y,no,no,no,Y}, 
   {Y,no,no,Y,no,no,Y,no},   //IV has m7 and M9
   {Y,no,no,Y,no,no,Y,no},
   {Y,no,no,Y,no,no,Y,no},
   {Y,no,no,Y,no,no,Y,no},
   {Y,no,Y,no,Y,no,no,no},   //V major has m7 and m9
   {Y,no,Y,Y,Y,no,Y,no},   //V minor has m7 and m9,M9
   {Y,no,Y,Y,Y,no,Y,no},
   {Y,no,Y,Y,Y,no,Y,no},
   {no,Y,no,Y,no,no,no,Y},   //VI major has M7 and M9
   {no,Y,no,Y,no,no,no,Y}, 
   {Y,no,Y,no,Y,no,no,no},   //VI diminished has m7 and m9
   {Y,no,Y,no,Y,no,no,no},
   {Y,Y,no,Y,no,no,Y,Y},   //VII major has m7,M7 and M9
   {Y,Y,no,Y,no,no,Y,Y},
   {no,no,Y,no,no,no,no,no},   //VII diminished has m9
   {no,no,Y,no,no,no,no,no}};

  public KeyControl() { /* Nothing */}
  
/**
 * Allocates a new KeyControl object and sets the key, using an integer 
 * base type
 *@param kt     an integer representing the base type
 *@param m      a String representing the mode
 */
  public KeyControl(int kt, String m) {
    currentKey.setMode(m);
    currentKey.setKeyType(kt);
    return;
  }

/**
 * Allocates a new KeyControl object and sets the key using a String for 
 * the base type.
 *@param kt     an String representing the base type
 *@param m      a String representing the mode
 */
  public KeyControl( String kt, String m ) {
    currentKey.setMode(m);
    currentKey.setKeyType(kt);
    return;
  }

  private static void updateScale() {
    // The Harmony (only mode important):
  
    if ((currentKey.getMode()).equals("major")) {
     ArrayTools.equalizeArrays(HarmonyProgression.basicScale,majorScaleChords);
      ExtraTonesControl.scaleExtras= majorScaleExtras;
    } else {
     ArrayTools.equalizeArrays(HarmonyProgression.basicScale,minorScaleChords);
      ExtraTonesControl.scaleExtras = minorScaleExtras;
    }
    
    //Update the sustainNotes
    int lastKT = lastKey.getIntKeyType() % 12;
    if (lastKT < 0) lastKT = lastKT + 12;
    int curKT = currentKey.getIntKeyType() % 12;
    if (curKT < 0) curKT = lastKT + 12;
    
    int[] oldSustain = HarmonyProgression.sustainNotes;
    int[] newSustain = new int[oldSustain.length];
    int index = 0;
    for (int i = 0; i<oldSustain.length; i++) {
      index = (lastKT - curKT + i) % 12;
      if (index < 0) index = index +12;
      newSustain[index] = oldSustain[i];
    }
    HarmonyProgression.sustainNotes = newSustain;

    //Need Melody info.
  }
    

// Access Methods
/**
 * Returns true if the key has not been set yet.
 */
  public static boolean isNull() { if (currentKey.isNull()) 
    return true; else return false; }

/**
 * Sets base type using an integer.
 */
  public static void setCurrentKeyType( int kt ) {
    lastKey.setKeyType(currentKey.getIntKeyType());
    currentKey.setKeyType(kt); 
    updateScale();
  }

/**
 * Sets base type using a String
 */
  public static void setCurrentKeyType( String kt ) {
    lastKey.setKeyType(currentKey.getIntKeyType());
    currentKey.setKeyType(kt);
    updateScale();
  }

/**
 * Sets the mode using a String
 */
  public static void setCurrentMode( String m ) { 
    lastKey.setMode(currentKey.getMode());
    currentKey.setMode(m);
    updateScale();
  }

/**
 * Gets the base type as a String
 */
  public static String getStrCurrentKeyType() { 
    return currentKey.getStrKeyType();}

/**
 * Gets base type as an integer
 */
  public static int getIntCurrentKeyType() { 
    return currentKey.getIntKeyType();}

/**
 * Gets mode as a String
 */
  public static String getCurrentMode() { return currentKey.getMode();}



/**
 *  For debugging.
 */
  public static void main(String[] args) {
  
  }
}
