/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.pitch.harmony;
import herman.elements.*;
import herman.properties.*;
import herman.pitch.melody.*;
import herman.pitch.tools.ArrayTools;

/** The HarmonyProgression class 
 *   generates the chord progressions of the music.  It serves as the central
 *   control center that calls each of the harmony parameter modules in 
 *   turn.
 */

public class HarmonyProgression {
  /** Number of chords */
  private static final int NUMchordTYPES = 4;
  static final int chordNum = 28;
  static final int extraTonesNum = 63;
  private static final int NUMextraTYPES = extraTonesNum / 7;

  /**
   * The in queue containing chords to be instantiated.
   */
  public static Queue inQueue = new Queue();
  private static float Yes = 1.0f;
  private static float no = -1000.0f;

  // Parameter arrays
  static float[] basicScale = new float[chordNum];
  static float[] commonTones = new float[chordNum];
  static float[] cadenceRules = new float[chordNum];
  static float[] modulate = new float[chordNum];
  static float[] pivotRules = new float[chordNum];
  static float[] generalProgression = new float[chordNum];
  static float[] dissonanceHandling = new float[chordNum];
  static float[] dissonanceLevel = new float[chordNum];
  static float[] baseChordType = new float[chordNum];
  static float[] tempChordType = new float[chordNum];
  static float[] sameChord = new float[chordNum];
  static float[] melodyMatching = new float[chordNum];
  static float[] sustainChords = new float[chordNum];

  static float[] extraTones = new float[NUMextraTYPES];
  static float[][] chordPoss = new float[28][5];
  static float[][] totalChordPoss = new float[28][5];

  // Last Chord
  static Chord lastChord;

  // Last Progression
  private static  int lastProg = 0;

  //Sustained melody flag
  static boolean sustainFlag = false;
  
  //Sustained melody possibilities
  static int[] sustainNotes = new int[12];

  //"Special" Blocks:
  private static boolean modulateBlock = false;
  private static boolean cadenceBlock = false;
  static boolean melodyMatchFlag = false;

  // Check to see if at least one chord probability > 0 
  private static boolean zeroCheck( float[] prob) {
    int greaterThanZeroFlag = 0;
    for (int i = 0; i < chordNum; i++)
      if (prob[i] >= 0.0f)
	greaterThanZeroFlag++;
    return (greaterThanZeroFlag > 0) ? true : false;
  }

  //Clear the "special" blocks:
  private static void clearBlocks() {
    modulateBlock = false;
    cadenceBlock = false;
    melodyMatchFlag = false;
  }

  // Send an event to the Chord Instantiator
  private static void instantiateChord (Chord chord) {

  ChordInstantiator.inQueue.push(chord);

  }

  private static int scaleDissonanceMelody(Chord chord) {
    int failed = 0;
     //Add in the scale information
    ArrayTools.add2Array(baseChordType, basicScale);

    //Match instantiated melody, if necessary
    chordPoss = MatchMelody.applyMelodyMatching(chord);

    if (melodyMatchFlag) {
      melodyMatching = limitChords(chordPoss);

      //add in the melody matching
      ArrayTools.equalizeArrays(tempChordType, baseChordType);
      ArrayTools.add2Array(tempChordType, melodyMatching);
      if (zeroCheck(tempChordType)) {
	ArrayTools.equalizeArrays(baseChordType, tempChordType);
      } else {failed++; System.out.println("Melody matching failure");}
    }

     /* Rules that require a lastChord: */
    if (lastChord != null) { 
 
     //Pivot Rules    
      ArrayTools.equalizeArrays(tempChordType, baseChordType);
      PivotRules.applyPivotRules(lastChord);
      ArrayTools.add2Array(tempChordType, pivotRules);
      if (zeroCheck(tempChordType)) 
           ArrayTools.equalizeArrays(baseChordType, tempChordType);
      else {failed++; System.out.println("Pivot Rules Failure");}
      //ArrayTools.printArray(baseChordType,4);

   
    
      //Dissonance Handling
      ArrayTools.equalizeArrays(tempChordType, baseChordType);
      DissonanceHandling.applyDissonanceHandlingRules(lastChord);
      ArrayTools.add2Array(tempChordType, dissonanceHandling);
      if (zeroCheck(tempChordType)) {
           ArrayTools.equalizeArrays(baseChordType, tempChordType);        
      }
      else {failed++; System.out.println("Dissonance handling failure");}
    }

    //Sustained melody checks:
    if ((sustainFlag)&&(!melodyMatchFlag)&&(chord.partMelEvent.size() > 0)) {
   
      //Must chord match the sustained melody note? 
      if (ChordTones.mustBeAChordTone((Event)chord.partMelEvent.getFirst(),
				      chord)) {
	//If so, get the chord limitations float[][]
	totalChordPoss = MatchSustain.matchSustain(chord); 
  
       //map that float into chord type limitations and extra tone restrictions
	sustainChords = limitChords(totalChordPoss);

	//add in the sustain matching
	ArrayTools.equalizeArrays(tempChordType, baseChordType);
	ArrayTools.add2Array(tempChordType, sustainChords);
	if (zeroCheck(tempChordType)) {
	  ArrayTools.equalizeArrays(baseChordType, tempChordType);
	} else {failed++; System.out.println("Sustain matching failure");}
      }  
    }
    return failed;
  }



  // Translation functions .  NOTE THE "4" THAT COMES FROM M,m,d,a!!!!!!
  private static int translateDegree( int index ) {
    return (index/NUMchordTYPES + 1);
  }
  
  private static int translateType( int index ) {
    return (index % NUMchordTYPES);
  }

  //Assign lastChord and lastProg
  private static void assignLasts(Chord chord) {
    int prog = 0;
    int lastDegree = 0;
    //  System.out.println(chord.chordType);
    if (!(lastChord == null)) {
     lastDegree = lastChord.chordType.getIntDegree();
    }
    else {lastDegree = chord.chordType.getIntDegree();}
    int chordDegree = chord.chordType.getIntDegree();
    int degInterval = chordDegree - lastDegree;
    if (degInterval == 0) {prog = 0;}
    else if (degInterval > 0) {prog = degInterval + 1;}
    else if (degInterval < 0) {prog = 8 + degInterval;}
    
    lastProg = prog;
    //System.out.println(prog);
    lastChord = chord;
  }

  //Takes a chordPoss float[][] and returns an equivalent chordType float
  // Also sets the ExtraTones correctly
  private static float[] limitChords(float[][] chordPoss) {
    float[] results = new float[chordNum];
    
    //Set the chord types possibilities...
      for (int i = 0; i<chordNum; i++) {
	ArrayTools.clipNegative(chordPoss[i]);
	if (ArrayTools.sumAbsArray(chordPoss[i])>0.0f) {
	  results[i] = Yes;}
	else {results[i] = no;}
      }

      int failures = 0;
      boolean success = false;
      //Set the Extra tones possibilities
      for (int i = 0; i<chordNum; i++) {
	failures = 0;
        for (int j = 0; j<5; j++) {
	  if (chordPoss[i][j]==0.0f) {                 //If chordPoss says
	    success = ExtraTonesControl.zeroProb(i,j);  // no to Extra, zeroProb
	     if (!success) {failures++;}
	  }
	}
	// If all extra tones poss. set to zero, chord type impossible
	if (failures >= ArrayTools.sumAbsArray(chordPoss[i])) {
	  results[i] = no;
	}
      }
      return results;
  }


  // matchMelody function 
  private static void matchMelody(Event mel, Chord chord) {
    melodyMatchFlag = true;
    Key key = new Key(KeyControl.getIntCurrentKeyType(), 
			KeyControl.getCurrentMode());
    //Move to mode if nec. and if chord is the first supCon of mel
    if ((mel.supConChord.size()==1)||
	((Chord)mel.supConChord.getFirst()==chord)) {
      if (MelodyScale.checkFlag()) MelodyScale.moveToMode(mel, key);
    }

    //Get chord possibilities
    float[][] newChords = ChordTones.getChords(mel.pitch, key);
    for (int i = 0; i<28; i++) {
      ArrayTools.add2Array(chordPoss[i], newChords[i]);
    }
  }

  /*
   *  Assigns chord inversions.
   */
  private static void assignInversion(Chord chord) {
    Inversions.assignInversions(chord, lastChord);
  }
  
/**
 *  Handles requests for new instantiated chords.  Returns true if successful.
 */
  public static boolean getNewChord() {
    makeNewChord();
    return true;
  }

  //Start new chord instantiation process
  private static void makeNewChord() {

    // if (needMoreHarmony()) /* wait */;

    //Get new harmonic event
    Chord chord = ((Chord)inQueue.pop());
    
    // Reset the baseChordType
    ArrayTools.setArrayValue( baseChordType, 0.0f );

    // If chord instantiated, pass it on
    if (!(chord.chordType.isNull())) {
      //Set the current key to be the chord key
      KeyControl.setCurrentKeyType(chord.key.getIntKeyType());
      KeyControl.setCurrentMode(chord.key.getMode());
 
      assignInversion(chord);
      assignLasts(chord);
      instantiateChord(chord);
      return;
    }
    

    // Continue with chord assignment
    assignChord(chord);
    assignInversion(chord);
    assignLasts(chord);
    instantiateChord(chord);
    return;    
  }

  //Assign a new Chord
  private static void assignChord( Chord chord ) {
     int chordTypeIndex = 0;
     int startIndex = 0;
     int failed = 0;

    // Reset the baseChordType
     // THis is now done in the makeNewChord method
     System.out.println(chord.harmonyInfo.getLocation());
    // Cadence, Modulation, Phrase Start, or Other
    if ((chord.harmonyInfo.getLocation().length() > 0)||
	(chord.harmonyInfo.getSecond().length() > 0)){
      ArrayTools.equalizeArrays(tempChordType, baseChordType);

      
      //CADENCE
    if (chord.harmonyInfo.getLocation().equals("pre-cadence")) {
      Cadence.setCadence("pre-cadence");
      modulateBlock = true;
    }
    else if (chord.harmonyInfo.getLocation().equals("cadence")) {
      Cadence.setCadence("cadence");
      modulateBlock = true;
    }
    else if (chord.harmonyInfo.getLocation().equals("resolution")) {
      Cadence.setCadence("resolution");
      modulateBlock = true;
    }

    //MODULATION
    else if (chord.harmonyInfo.getSecond().equals("modulate minor")) {
      Modulation.setModulation("minor");
    }
    else if (chord.harmonyInfo.getSecond().equals("modulate major")) {
      Modulation.setModulation("major");
    }

    //PHRASE MARKERS
    else if (chord.harmonyInfo.getLocation().equals("sentence start")) {
      ArrayTools.setArrayValue(baseChordType,no);
      lastChord = null;                         //Last chords irrelevant
      modulateBlock = true;
      for (int i=0; i<4; i++) {      
	baseChordType[i] = Yes;                 //I a good start
	baseChordType[16+i] = Yes;              //V a good start
      }
    }
    else if (chord.harmonyInfo.getLocation().equals("phrase start")) {
      // ArrayTools.setArrayValue(baseChordType,no);
      //lastChord = null;         //Last chords irrelevant???????!!!!!
    }

    }


    if ((Cadence.checkFlag())&&(!cadenceBlock)) {
	Cadence.doCadence();
	ArrayTools.add2Array(baseChordType,cadenceRules);
	
	//ArrayTools.printArray(baseChordType,4);
       
    }

    failed = scaleDissonanceMelody(chord);

    if (lastChord != null) { 
/*  CHECK cadence and modulation etc. 
 *
 *
 *
 */
      if ((Modulation.checkFlag())&&(!modulateBlock)) {
	ArrayTools.setArrayValue(modulate, Yes);
	Modulation.modulate(lastChord);
	ArrayTools.equalizeArrays(tempChordType,baseChordType);
	int keyType = KeyControl.getIntCurrentKeyType();
	String mode = new String(KeyControl.getCurrentMode());
	Key oldKey = lastChord.key;
	ChordType oldChordType = lastChord.chordType;

	if (!Modulation.checkFlag()) {        //if modulation occurred
	  ArrayTools.setArrayValue(baseChordType,0.0f);
	  int failNum = scaleDissonanceMelody(chord); 
	                          //redo early steps in new key
	  if (failNum > failed) {   //if causes more failures, reset...
	    ArrayTools.equalizeArrays(baseChordType,tempChordType);
	    String newMode  = new String(KeyControl.getCurrentMode());
	    KeyControl.setCurrentKeyType(keyType);
	    KeyControl.setCurrentMode(mode);
	    Modulation.setModulation(newMode);
	    lastChord.key = oldKey;
	    lastChord.chordType = oldChordType;
	  }
	}
	else {                              //modulation did not occur
	  //Check to see if restrictions cause failure
	  ArrayTools.add2Array(tempChordType, modulate);
	  if (zeroCheck(tempChordType)) 
	    ArrayTools.add2Array(baseChordType,modulate);
	  else {System.out.println("Modulation failure!!!");}
	}   
      }

     

    //Same tone rules
      ArrayTools.equalizeArrays(tempChordType, baseChordType);
    SameChordRules.applySameChordRules(lastChord);
    ArrayTools.add2Array(tempChordType, sameChord);
    if (zeroCheck(tempChordType)) 
            ArrayTools.equalizeArrays(baseChordType, tempChordType);
  

      //Common Tones
      ArrayTools.equalizeArrays(tempChordType, baseChordType);
      CommonTones.applyCommonTonesRules(lastChord);
      ArrayTools.add2Array(tempChordType, commonTones);
    if (zeroCheck(tempChordType)) 
            ArrayTools.equalizeArrays(baseChordType, tempChordType);
   

      //General Progression
      ArrayTools.equalizeArrays(tempChordType, baseChordType);
      GeneralProgression.applyGeneralProgressionRules(lastChord, lastProg);
      ArrayTools.add2Array(tempChordType, generalProgression);
    }

  //Rules that change only when parameters change
    
    // Dissonnace Level:
    ArrayTools.add2Array(baseChordType, dissonanceLevel);
 
    //    System.out.println("The chord array...");
    // ArrayTools.printArray(baseChordType,4);
    


    //Select base degree and type:

    chordTypeIndex = ArrayTools.probableSelect(baseChordType);
    chord.chordType.setDegree(translateDegree(chordTypeIndex));
    chord.chordType.setType(translateType(chordTypeIndex));

/* ***  if (lastChord != null) {
 *     float[] old2Extras = ExtraTonesControl.getExtrasVector(
 *				      lastChord.chordType.getIntDegree(),
 *			      lastChord.chordType.getIntType()    );
 *  ArrayTools.printArray(old2Extras);
 *  }
*/

    //Decide if extra tones in chord
    extraTones = ExtraTonesControl.getExtrasVector(
				      chord.chordType.getIntDegree(),
				      chord.chordType.getIntType()    );
    ExtraTonesControl.resetExtraTones();
  
    /*  *****THE ABOVE IS FOR DEBUG <---What!!??!! */

    chordTypeIndex = ArrayTools.probableSelect(extraTones); 
                  // Need to change this ?
    chord.chordType.setExtras(chordTypeIndex);
    
    //Clean-up and return

    chord.key.setKeyType(KeyControl.getIntCurrentKeyType());
    chord.key.setMode(KeyControl.getCurrentMode());
    //   ArrayTools.setArrayValue(baseChordType,0.0f);  Done at beginning!!
    ArrayTools.setArrayValue(extraTones,0.0f);

    //Clear the blocks on modulation, etc.
    clearBlocks();

    //Check for sustained event onto next chord
    MatchSustain.checkSustain(chord);

/*    if ((chord.partMelEvent.size() > 0)) {
 *	Event lastEvent = ((Event)chord.partMelEvent.getElemAt(
 *                         (chord.partMelEvent.size()-1)));
 *	//if lastEvent has other chords attached:
 *	if (lastEvent.supConChord.size()>1) {
 *	  //if there is only one event, stretching over this chord
 *	  if ((chord.partMelEvent.size()==1)&&(sustainFlag)) {
 *	    int[] tempNotes = new int[12];
 *	    int[] chordTones = ChordTones.getChordTones(chord);
 *	    boolean flag = false;
 *	    for (int i = 0; i<chordTones.length; i++) {
 *	      if (sustainNotes[chordTones[i]] > 0) {
 *		tempNotes[chordTones[i]] = 1;
 *		flag = true;
 *	      }
 *	    }
 *	    //if there are common tones, those notes are the new sustain notes
 *	    // Otherwise, the sustain notes remain the same
 *	    if (flag) sustainNotes = tempNotes;
 *	  }
 *	  //Otherwise, if this is a new event to be sustained
 *	  else {
 *	    int[] chordTones = ChordTones.getChordTones(chord);
 *	    for (int i = 0; i<12; i++) {
 *	      sustainNotes[i] = 0;
 *	    }
 *	    for (int i = 0; i<chordTones.length; i++) {
 *	      sustainNotes[chordTones[i]] = 1;
 *	    }
 *	  }
 *	  sustainFlag = true;
 *	}
 *      	else {sustainFlag = false;}     //lastEvent not sustained...
 *	}
 */

    return;
 
  }

/**
 *  For debugging.
 */
  public static void main(String[] args) {
    KeyControl.setCurrentKeyType(0);
    KeyControl.setCurrentMode("major");
    DissonanceLevel.setDissonanceLevel(20.0f);
    DissonanceHandling.setDissonancePrep(0);
    DissonanceHandling.setDissonanceRes(2);

    Chord chord1 = new Chord();
    inQueue.push(chord1);
    Chord chord2 = new Chord();
    inQueue.push(chord2);
    Chord chord3 = new Chord();
    inQueue.push(chord3);
    Chord chord4 = new Chord();
    inQueue.push(chord4);
    Chord chord5 = new Chord();
    inQueue.push(chord5);
    Chord chord6 = new Chord();
    inQueue.push(chord6);
    Chord chord7 = new Chord();
    inQueue.push(chord7);
    Chord chord8 = new Chord();
    inQueue.push(chord8);
    Chord chord9 = new Chord();
    inQueue.push(chord9);
    Chord chord10 = new Chord();
    inQueue.push(chord10);
    
    chord1.setMusicDefn("sentence start");

    System.out.println("Starting....");
    System.out.println(" ");
    for (int i = 1; i<=4; i++) {
    getNewChord();
    System.out.println(lastChord.chordType + " in the key of " 
          + lastChord.key);
    }

    Modulation.setModulation("minor");
    for (int i = 5; i<=7; i++) {
    getNewChord();
    System.out.println(lastChord.chordType + " in the key of " 
          + lastChord.key);
    }
  
    Cadence.setCadence("pre-cadence");
    for (int i = 8; i<=10; i++) {
    getNewChord();
    System.out.println(lastChord.chordType + " in the key of " 
          + lastChord.key);
    }
    
  }
}
