/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.pitch.harmony;
import herman.elements.*;
import herman.pitch.tools.*;
/** 
 * The GeneralProgression class contains the rules governing the proper
 *   general progressions of harmonies (by degree)
 *  <p>
 **/

public class GeneralProgression {

  /** Float value describing the extent to which the general progression
   *   rules should be ignored (the higher the value, the more the rules
   *   are disregarded).  0 - 10 is the useful range.
   */

  public static float ignoreLevel = 0.0f;

  /** Float value describing the relative effect that the general 
   *   progression rules have with respect to the other harmony parameters.
   */ 
  public static float powerLevel = 100.0f;

  private static final int chordNUM = 28;

/**
 *  Applies the general progression rules based on the current ignoreLevel.  
 *    Automatically modifies the generalProgression probability array in 
 *    the Harmony Progression sub-module.
 *@param lastEvent  the last harmonic event 
 *@param lastProg  the integer of the interval (degree) of the last progression
 */
  public static void applyGeneralProgressionRules( Chord lastChord,
						   int lastProg) {
    int lDegree = lastChord.chordType.getIntDegree();
    float[] genProgArray = HarmonyProgression.generalProgression;
    ArrayTools.setArrayValue(genProgArray,0.0f);
    int resolve = 0;

   if (lastProg != 0) {                 // If there was a lastProg
    float GOOD = (4.0f + ignoreLevel);    //Set base values
    float OK = (3.0f + ignoreLevel);
    float SPARE = (2.0f + ignoreLevel);

    if (lastProg==5) { // If last progression was a 5th up
      
	resolve = (((lDegree - 1) + 1) % 7);  //resolve to 2nd above     
	for (int i = 0; i <= 3; i++) {   //Make those degrees more likely
	  genProgArray[resolve*4+i] = GOOD;
	}
	resolve = (((lDegree - 1) + 7) % 7);  //Or resolve to 2nd below      
	for (int i = 0; i <= 3; i++) {   //Make those degrees more likely
	  genProgArray[resolve*4+i] = GOOD;
	}
    }    
    else if (lastProg==3) { // If last progression was a 3rd up
      
	resolve = (((lDegree - 1) + 1) % 7);  //resolve to 2nd above       
	for (int i = 0; i <= 3; i++) {   //Make those degrees more likely
	  genProgArray[resolve*4+i] = GOOD;
	}
	resolve = (((lDegree - 1) + 3) % 7);  //Or resolve to 4th above
	for (int i = 0; i <= 3; i++) {   //Make those degrees more likely
	  genProgArray[resolve*4+i] = GOOD;
	}
    }
    else {    //Otherwise follow normal probabilities:
        resolve = (((lDegree - 1) + 3) % 7);  //resolve to 4th above       
	for (int i = 0; i <= 3; i++) {          //Good
	  genProgArray[resolve*4+i] = GOOD;
	}
	resolve = (((lDegree - 1) + 5) % 7);  //resolve to 6th above
	for (int i = 0; i <= 3; i++) {          //Good
	  genProgArray[resolve*4+i] = GOOD;
	} 
	resolve = (((lDegree - 1) + 2) % 7);  //resolve to 3rd above       
	for (int i = 0; i <= 3; i++) {          //OK
	  genProgArray[resolve*4+i] = OK;
	}
	resolve = (((lDegree - 1) + 4) % 7);  //resolve to 5th above
	for (int i = 0; i <= 3; i++) {          //OK
	  genProgArray[resolve*4+i] = OK;
	} 
	resolve = (((lDegree - 1) + 1) % 7);  //resolve to 2nd above       
	for (int i = 0; i <= 3; i++) {          //sparingly
	  genProgArray[resolve*4+i] = SPARE;
	}
	resolve = (((lDegree - 1) + 6) % 7);  //resolve to 7th above
	for (int i = 0; i <= 3; i++) {          //Sparingly
	  genProgArray[resolve*4+i] = SPARE;
	} 
    }

    //Normalize and multiply by powerLevel
    ArrayTools.normalizeArray(genProgArray);
    ArrayTools.multiplyArrayBy(genProgArray,powerLevel);
   }
  }

// Access Methods


/**
 *  For debugging.
 */
  public static void main(String[] args) {
   
  }
}
