/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.pitch.harmony;
import herman.pitch.tools.*;

/** 
 * The DissonanceLevel class contains rules that adjust the probabilities 
 *   in the HarmonyProgression sub-module towards selecting/avoiding dissonant
 *   chords.  
 *  <p>
 *   From consonant to dissonant, the chord types are:
 * <ul>
 * <li> major
 * <li> minor
 * <li> diminished
 * <li> augmented
 * </ul>
 *
 * It also influences the ExtraTonesControl towards selecting/avoiding
 *   dissonant extra tones.
 * <p>
 * From consonant to dissonant, the extra tones are:
 *  <ul>
 *  <li> none
 *  <li> sevenths
 *  <li> ninths
 *  <li> sevenths and ninths
 *  </ul>
 *
 */

public class DissonanceLevel {

/**
 * dissonanceLevel is a float > 0.0 that controls the level of dissonance 
 *   in the accompaniment chords (larger numbers => more dissonance)
 */
  private static float dissonanceLevel = 0.0f;

/**
 * powerLevel is a float > 0.0 that controls relative effect of the
 *    dissonance level rules verus the other harmony parameters
 */
  private static float powerLevel = 100.0f;

  private static void setTypeArray(float[] array, String type, float value) {
    int index = 0;
    if (type.equals("major")) index = 0;
    else if (type.equals("minor")) index = 1;
    else if (type.equals("diminished")) index = 2; 
    else index = 3; 

    for (int i = 0; i < 28; i = i+4) {
      array[i+index] = value;
    }
    return;
  }

 
  private static void updateDissonance() {
    float[] typeArray = HarmonyProgression.dissonanceLevel;
    float[] extraArray = new float[9];

    // Set the type probabilities
    float value = 100.0f - 0.5f*dissonanceLevel;
    setTypeArray(typeArray,"major", value);
    
     value = 25.0f + 0.5f*dissonanceLevel;
    setTypeArray(typeArray,"minor", value);

    value = 15.0f + 0.75f*dissonanceLevel;
    setTypeArray(typeArray,"diminished", value);

     value = 0.0f + 0.75f*dissonanceLevel;
    setTypeArray(typeArray,"augmented", value);

    // Set the extra tones probabilities
     value = 100.0f - dissonanceLevel;
     extraArray[0] = value;
    
    value = 0.0f + 1.0f*dissonanceLevel;
    extraArray[1] = value;
    extraArray[2] = value;

    value = -74.0f + 1.5f*dissonanceLevel;
    extraArray[3] = value;
    extraArray[4] = value;

    value = -150.0f + 2f*dissonanceLevel;
    extraArray[5] = value;
    extraArray[6] = value;
    extraArray[7] = value;
    extraArray[8] = value;

    //Normalize and multiply by powerLevel
    ArrayTools.normalizeArray(typeArray);
    ArrayTools.normalizeArray(extraArray);
    ArrayTools.multiplyArrayBy(typeArray, powerLevel);
    ArrayTools.multiplyArrayBy(extraArray, powerLevel);

    //Send the array to the ExtraTonesControl
    ExtraTonesControl.disLevelExtras = extraArray;

  }

// Access Methods

/**
 * Sets the power level.  NEEDS EXCEPTION HANDLING
 *@param power    A float > 0.0 that controls relative effect of the
 *    dissonance level rules verus the other harmony parameters
 */
  public static void setPowerLevel(float power) {
    powerLevel = power;
    updateDissonance();
  }

/**
 * Sets the dissonance level. NEEDS EXCEPTION HANDLING
 *@param dissonance A float greater than zero that controls the level of
 *    dissonance in the accompaniment chords (larger numbers => more 
 *    dissonance).  0 - 100 is probably the useful range.
 */
  public static void setDissonanceLevel(float dissonance) {
    dissonanceLevel = dissonance;
    updateDissonance();
  }

/**
 * Gets the power level
 */
  public static float getPowerLevel() {return powerLevel;}
  
/**
 * Gets the dissonance level
 */
  public static float getDissonanceLevel() {return dissonanceLevel;}

/**
 *  For debugging.
 */
  public static void main(String[] args) {

  }
}
