/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.pitch.harmony;
import herman.elements.*;
import herman.pitch.tools.*;

/** 
 * The DissonanceHandling class contains the rules for preparing and 
 * resolving dissonances.  It enables or disables the application of 
 * these rules in the Harmony Progression sub-module.
 *  <p>
 * Preparation and resolution are important for the diminished and augmented
 *  chord types, as well as for sevenths and ninths.  At the present time,
 *  though, augmented and ninth chords are ignored.
 **/

public class DissonanceHandling {

  /** Semi-booleans describing whether or not a particular rule should be 
   *   applied.  
   *     2 = full application
   *     1 = looser application
   *     0 = no application
   */

  private static int dimPrep = 2;
  private static int dimRes  = 2;
  private static int augPrep = 2;
  private static int augRes  = 2;
  private static int seventhPrep = 2;
  private static int seventhRes  = 2;
  private static int ninthPrep = 2;
  private static int ninthRes  = 2;

  

  private static final float Yes = 1.0f;
  private static final float no = -1000.0f;
 

/**
 *  Applies the currently true rules for dissonance handling and uses the 
 *   result to set the dissonanceHandling array in HarmonyProgression
 * @param lastEvent the last harmonic event
 *
 */
  public static void applyDissonanceHandlingRules( Chord lastChord) {
    int lType = lastChord.chordType.getIntType();
    int lExtra = lastChord.chordType.getIntExtras();
    int lDegree = lastChord.chordType.getIntDegree();
    float[] dissArray = HarmonyProgression.dissonanceHandling;

    ArrayTools.setArrayValue(dissArray,Yes);      //Everythings OK
    // ExtraTonesControl.resetDisHandleExtras();     //All extra tones OK

    int resolve = 0;
    int oKTone = 0;

//Preparation:
    if (dimPrep!=0) {
      for (int i = 2; i <4; i = i+4) {
	dissArray[i] = no;                //Set diminished to no
      }
      oKTone = (((lDegree - 1) + 3) % 7);  //prepared by 5th above
      dissArray[oKTone*4+2] = Yes;
      oKTone = (((lDegree - 1) + 6) % 7);  //prepared by 3rd above
      dissArray[oKTone*4+2] = Yes;
    }
    if (seventhPrep==2) {
      oKTone = (((lDegree - 1) + 3) % 7);  //7th prepared by 5th above
      int oKTone2 = (((lDegree - 1) + 6) % 7);  //7th prepared by 3rd above
      for (int i = 0; i<HarmonyProgression.baseChordType.length; i++) {
	if (HarmonyProgression.baseChordType[i] > 0.0f) {
	  if (((i % 4) != oKTone)&&((i % 4) != oKTone2)) {
	    ExtraTonesControl.zeroProb(i,"m7");
	    ExtraTonesControl.zeroProb(i,"M7");
	  }
	}
      }
    }


//Resolution:
    if ((lType==2)||(lType==3)||(lExtra!=0)) {
      // If chord dim., aug., or has 7ths or 9ths

      if ((lType==2)&&(dimRes != 0)) {   //diminished needs resolving
	ArrayTools.setArrayValue(dissArray,no);
	resolve = (((lDegree - 1) + 3) % 7);  //resolve to 4th above
        
	for (int i = 0; i <= 3; i++) {   //Make those degrees possible
	  dissArray[resolve*4+i] = Yes;
	}
      }

      else if ((lType==3)&&(augRes != 0)) {   //augmented needs resolving
/* ******** NEED RULES FOR AUGMENTED CHORDS ************************
        
	for (int i = 0; i <= 3; i++) {   //Make those degrees possible
	  dissArray[resolve*4+i] = Yes;
	}   */
      }

      else if (((lExtra==1)||(lExtra==2)||(lExtra>4))&&(seventhRes==2)) {  
	                                   //sevenths need  resolving
	ArrayTools.setArrayValue(dissArray,no);
	resolve = (((lDegree - 1) + 3) % 7);  //resolve to 4th above
        
	for (int i = 0; i <= 3; i++) {   //Make those degrees possible
	  dissArray[resolve*4+i] = Yes;
	}

	if (lExtra == 2) {     //M7's can also resolve:
	  resolve = (((lDegree - 1) + 4) % 7);  // to a 5th above...
	  for (int i = 0; i <= 3; i++) {        // (make that degree possible)
	    dissArray[resolve*4+i] = Yes;
	  }
	}

      } 
      
      else if ((lExtra>2)&&(lExtra<5)&&(ninthRes!=0)) {   
	                                    //ninths need  resolving
	ArrayTools.setArrayValue(dissArray,no);

	resolve = (((lDegree - 1) + 5) % 7);  //resolve to 6th above       
	for (int i = 0; i <= 3; i++) {        //Make that degree possible
	  dissArray[resolve*4+i] = Yes;
	}
        
	if (lExtra == 4) {                      //M9's can also resolve:
	  resolve = (((lDegree - 1) + 6) % 7);  // to a 7th above (was 3rd!)
	  for (int i = 0; i <= 3; i++) {        // (make that degree possible)
	    dissArray[resolve*4+i] = Yes;
	  }

	  /*  resolve = (((lDegree - 1) + 3) % 7);   //...and to a 4th above      
	  for (int i = 0; i <= 3; i++) {         //(make that degree possible)
	    dissArray[resolve*4+i] = Yes;
	  }  */
	}	
      }
    }
  }

// Access Methods

/**
 *  Set the level of dissonance preparation. 
 *
 * <ul>
 * <li> 2 = full preparation 
 * <li> 1 = partial preparation 
 * <li> 0 = no preparation 
 *  </ul>
 */
  public static void setDissonancePrep(int handle) {
    dimPrep = handle;
    augPrep = handle; 
    seventhPrep = handle;  
    ninthPrep = handle;
   
  }

/**
 *  Set the level of dissonance resolution. 
 *
 * <ul>
 * <li> 2 = full  resolution
 * <li> 1 = partial resolution
 * <li> 0 = no resolution
 *  </ul>
 */
  public static void setDissonanceRes(int handle) {
    
    dimRes  = handle;
    augRes  = handle;
    seventhRes  = handle;
    ninthRes  = handle;
  }
    

/**
 *  For debugging.
 */
  public static void main(String[] args) {
  
  }
}
