/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.pitch.harmony;
import herman.elements.*;
import herman.pitch.tools.*;
import herman.form.PhraseForm;
/** 
 * The Cadence class handles the rules governing the use and application 
 * of cadences in the harmonic progression.
 * <p>
 * Cadence selects a cadence type based on (1) the location of the phrase
 * in the sentence and (2) the respective settings of sentenceResolution
 * and phraseResolution, both integers in [1,5].
 *<p>
 * Sentence Resolution:
 * <ol>
 * <li> authentic cadence only
 * <li> authentic & plagal cadences
 * <li> authentic, plagal, and dominant cadences
 * <li> plagal, dominant, and deceptive cadences
 * <li> no cadences
 * </ol>
 * <p>
 * Phrase Resolution:
 * <ol>
 * <li> authentic & plagal cadences
 * <li> authentic, plagal, and dominant cadences
 * <li> plagal & dominant
 * <li> plagal, dominant, and deceptive cadences
 * <li> no cadences
 * </ol>
 * <p>
 * Cadence uses the HarmonyProgression lastChord information, if available.
 * 
 **/

public class Cadence {
  private static boolean cadenceFlag = false;
  private static StringBuffer location = new StringBuffer("pre-cadence");
  private static float no = -1000.0f;
  private static float Yes = 1.0f;
  private static int sentenceResolution = 1;
  private static int phraseResolution = 1;
  private static int resolutionFinality = 0;

  // Check to see if at least one chord probability > 0 
  private static boolean zeroCheck( float[] prob) {
    int greaterThanZeroFlag = 0;
    for (int i = 0; i < prob.length; i++)
      if (prob[i] >= 0.0f)
	greaterThanZeroFlag++;
    return (greaterThanZeroFlag > 0) ? true : false;
  }

  private static void applyCadence() {
    float[] cadRules = HarmonyProgression.cadenceRules;
    ArrayTools.setArrayValue(cadRules, no);
    int lDegree = 1;
    int cType = 1;

    //Set the cadence type possibilities
    if (!(PhraseForm.getCurrentLocation().equals("sentence end"))) {
                 // if the phrase is not the last in the sentence:
	cType = phraseResolution + 1;
    }
    else {
      if (sentenceResolution > 3) {
	cType = sentenceResolution + 1;
      } else {cType = sentenceResolution;}
    }


    if (!(HarmonyProgression.lastChord == null)) {
      lDegree = HarmonyProgression.lastChord.chordType.getIntDegree();
    }

    float rand = ((float)java.lang.Math.random());

    //Pre-cadence:
    if (location.toString().equals("pre-cadence")) {
      //First, permit the authentic (and deceptive)
      for (int i = 0; i<4; i++) {
	cadRules[4+i] = Yes;            //II OK
	cadRules[12+i] = Yes;           //IV OK
	cadRules[20+i] = Yes;           //VI OK
      }
      if ((lDegree==2)||(lDegree==4)||(lDegree==6)) {
	for (int i = 0; i<4; i++) {
	  cadRules[i] = Yes;           //I OK
	}
      }

      //Next, add in plagal if necess.:
      if ((cType == 2)&&(rand>0.3f)) {
	cadRules[24] = Yes;            //VII major OK
	cadRules[17] = Yes;            //V minor OK
	cadRules[8] = Yes;             //III major OK
	if (KeyControl.getCurrentMode().equals("major")) {
	  cadRules[16] = Yes;         //V major OK
	  cadRules[9] = Yes;          //III minor OK  
	}
      }
	  
      //Finally, above this point anything can lead in:
      else if (cType > 2) {
	ArrayTools.setArrayValue(cadRules, Yes);
      }

      //Avoid weak cadences; eliminate same chord:
      for (int i = 0; i<4; i++) {
	  cadRules[(lDegree-1)*4+i] = no;           //kill same chord
      }
    }

    //Do a cadence (i.e., the second to last chord)
    else if (location.toString().equals("cadence")) {
      //First, set authentic (and deceptive) possibilities
      if ((lDegree==2)||(lDegree==4)||(lDegree==6)||(lDegree==1)) {
	cadRules[16] = Yes;            //V major OK
	if (cType > 4) cadRules[17] = Yes;  //V minor OK with deceptive
      }

      //Next, add in plagal
      if (cType > 1) {
	if ((lDegree!=1)&&((cType == 4)||(lDegree==3)||(lDegree==5)||
	    (rand > 0.3f))) {   //throw in some randomness
                               // to reduce probability  
	  cadRules[6] = Yes;              //II diminished OK
	  cadRules[13] = Yes;             // IV minor OK
	  if (KeyControl.getCurrentMode().equals("major")) {
	    cadRules[5] = Yes;            //II minor OK in major
	    cadRules[12] = Yes;           //IV major OK in major
	  }
	}
      }

      //Next add in dominant
      if ((cType > 2)&&(rand > 0.5f)) {   //throw in some randomness
	                                     // to reduce probability  
	for (int i = 0; i<4; i++) {
	  cadRules[i] = Yes;               //I OK
	  cadRules[4+i] = Yes;             //II OK
	  cadRules[12+i] = Yes;            //IV OK
	  cadRules[20+i] = Yes;            //VI OK
	}
      }

      //Subtract authentic if only have plagal and dominant:
      if (cType == 4) {
	  cadRules[16] = no;            //V major not OK now
      }
      
      //Finally, everything
      if (cType == 6) {
	for (int i = 0; i<4; i++) {
	  cadRules[8+i] = Yes;               //III OK
	  cadRules[24+i] = Yes;             //VII OK
	}
      } 

      //Avoid weak cadences; eliminate same chord:
      for (int i = 0; i<4; i++) {
	  cadRules[(lDegree-1)*4+i] = no;           //kill same chord
      }

      //Failsafe...force the V
      if (!(zeroCheck(cadRules))) {
	cadRules[16] = Yes;            //V Major OK 
	//cadRules[17] = Yes;            //V minor OK
	if (HarmonyProgression.lastChord == null) {    /* if this is a
							  first chord */
	  cadRules[0] = Yes;            //I Major OK 
	  cadRules[1] = Yes;            //I minor OK 
	}
      }
    }

    //Finally, resolutions
    else  {
      //check to see if none:
      if (cType == 6) {ArrayTools.setArrayValue(cadRules, Yes);}
      
      else {  //Otherwise, limit possibilities still...
	//Take care of authentic and plagal
	if ((lDegree == 5)||(lDegree==2)||(lDegree==4)) {
	  cadRules[0] = Yes;            //I Major OK
	  cadRules[1] = Yes;            //I minor OK
	  if (resolutionFinality == 0) {
	    ExtraTonesControl.force(0,"none");
	    ExtraTonesControl.force(1,"none");
	  }
	}

	//add in dominant:
	if ((cType > 2 )&&((lDegree==1)||(lDegree==2)||(lDegree==4)||
			    (lDegree==6))) {
	  for (int i = 0; i<4; i++) {
	    cadRules[16+i] = Yes;            //V OK
	  }
	}

	//add in deceptive
	if ((cType > 4)&&(lDegree==5)) {
	  for (int i = 0; i<4; i++) {
	    cadRules[12+i] = Yes;            //IV OK
	    cadRules[20+i] = Yes;            //VI OK
	  }
	}
	
	//Avoid weak cadences; eliminate same chord:
	for (int i = 0; i<4; i++) {
	  cadRules[(lDegree-1)*4+i] = no;           //kill same chord
	}

	//Failsafe...force resolution
	if (!(zeroCheck(cadRules))) {
	  if (lDegree != 1) {             //if last degree not I:
	    cadRules[0] = Yes;            //I Major OK
	    cadRules[1] = Yes;            //I minor OK
	    if (resolutionFinality == 0) {
	      ExtraTonesControl.force(0,"none");
	      ExtraTonesControl.force(1,"none");
	    }
       
	  } else {                        //if last degree == 1:
	    cadRules[16] = Yes;                  //V major OK
	    cadRules[17] = Yes;                  //V minor OK
	    ExtraTonesControl.force(16,"none");
	    ExtraTonesControl.force(17,"none");
	  }
	}
      }
    }
    //ArrayTools.printArray(cadRules,4);
  }

// Access Methods

/**
 *  Begins a new cadence.  Starts at the "cadence position" specified in 
 *   the String argument (either "pre-cadence", "cadence", or "resolution").
 */
  public static void setCadence(String type) {
     location.insert(0,type);
     location.setLength(type.length());
     cadenceFlag = true;
  }

/**
 *  Apply the cadence rules based on the current "cadence position."  
 *   Automatically handles setting the cadenceFlag appropriately to signal
 *   whether or not a cadence is in progress.
 */ 
  public static void doCadence() {
    applyCadence();
    if (location.toString().equals("pre-cadence")) {
      setCadence("cadence");
    }
    else if (location.toString().equals("cadence")) {
      setCadence("resolution");
    }
    else {cadenceFlag = false;}     //Cadence finished
  }

/**
 *  Return the status of the boolean cadenceFlag.  If "true", a cadence is
 *    in progress.
 */ 
  public static boolean checkFlag() {
    if (cadenceFlag) {return true;}
    return false;
  }

/** 
 * Set the phrase resolution type
 */
  public static void setPhraseResolution(int v) {
    phraseResolution = v;
  }

/** 
 * Set the sentence resolution type
 */
  public static void setSentenceResolution(int v) {
    sentenceResolution = v;
  }

/** 
 * Set the resolution finality.  Making this anything but 0 allows the 
 *    resolution I chord to have extra tones
 */
  public static void setResolutionFinality(int v) {
    resolutionFinality = v;
  }


/**
 *  For debugging.
 */
  public static void main(String[] args) {

  }
}
