/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.misc;

public abstract class TemporalQueue
{
  private QueueElement queue = null;
  
  protected static final int BEFORE = 0;
  protected static final int AFTER = 1;
  protected static final int EQUAL = 2;


  private static int numTimes=0;
  private static int numEvents=0;

  private Boolean lock = new Boolean(true);

  /**
   * Add a TemporalQueueObject to the queue
   *
   * @param e The object to add
   */
  public void addElement(TemporalQueueObject e)
  {
    synchronized(lock)
    {
      QueueElement el, newEl;
      
      // new event added, so increment numEvents
      numEvents++;
      
      // check we've not got an empty queue
      if (queue == null)
      {
	// create new QueueElement
	queue = new QueueElement(e.getKey());
	
	// there is NO previous element to this
	queue.previous = null;
	
	// add in the new TemporalQueueObject & exit
	queue.objects.add(e);
	return;
      }


      // OK, try and recurse to the QueueElement with the same key
      // as TemporalQueueObject e
      Object[] result = recurseTo(queue, e.getKey());


      // process the findings of recurseTo
      switch(((Integer) result[0]).intValue())
    {
    // Right, a QueueElement with the same key as e was not found, but we
    // should add a new QueueElement into the returned QueueElement's BEFORE 
    // pointer
    case BEFORE:
      // extract the previous QueueElement
      el = ((QueueElement) result[1]);

      // create a new QueueElement for things with this key
      newEl = new QueueElement(e.getKey());

      // OK, update it's "previous" QueueElement stuff
      newEl.wasBefore = true;
      newEl.previous = el;
      
      // add it into el's before pointer
      el.before = newEl;

      // finally, add in the TemporalQueueObject e to the new QueueElement
      newEl.objects.add(e);

      // new time added => increment numTimes
      numTimes++;
      break;
      


    // Right, a QueueElement with the same key as e was not found, but we
    // should add a new QueueElement into the returned QueueElement's AFTER 
    // pointer
    case AFTER:
      // extract the previous QueueElement
      el = ((QueueElement) result[1]);

      // create a new QueueElement for things with this key
      newEl = new QueueElement(e.getKey());

      // OK, update it's "previous" QueueElement stuff
      newEl.wasAfter = true;
      newEl.previous = el;
      
      // add it into el's after pointer
      el.after = newEl;

      // finally, add in the TemporalQueueObject e to the new QueueElement
      newEl.objects.add(e);

      // new time added => increment numTimes
      numTimes++;
      break;


    // OK, we found a QueueElement with the same key as the TemporalQueueObject	
    // just add the TemporalQueueObject into it's object list
    case EQUAL:
      // extract the previous QueueElement
      el = ((QueueElement) result[1]);

      // add the object in
      el.objects.add(e);

      // time was already there, so don't increment numTimes
      break;
    }
    }
  }


  /**
   * Retrive the list of TemporalQueueObjects occurring at the head of the 
   * queue. (note this of type Object[], NOT TemporalQueueObject[])
   * DOES NOT DELETE THEM
   * 
   * @return The array of objects
   */
  public Object[] firstElement()
  {
    synchronized(lock)
    {
      // check we've not got an empty queue
      if (queue == null)
	return(null);
      
      // get the QueueElement containing the first TemporalQueueObjects
      QueueElement el = recurseToFirst(queue);

      // if  it's null, then there's nothing in the queue
      if (el == null)
	return(null);
      
      // return it's events
      return(el.objects.getArray());
    }
  }
  

  /**
   * Retrive the list of TemporalQueueObjects occurring at the head of the 
   * queue. (note this of type Object[], NOT TemporalQueueObject[])
   * DOES DELETE THEM
   * 
   * @return The array of objects
   */
  public Object[] firstElementAndRemove()
  {
    synchronized(lock)
    {
      // check we've not got an empty queue
      if (queue == null)
	return(null);
      
      // get the QueueElement of the first event in the queue
      QueueElement el = recurseToFirst(queue);

      // extract the list of objects & nullify the list of objects
      Object[] objects = el.objects.getArray();
      el.objects = null;
      
      // OK, now tidy up the queue, starting from el
      fixTree(el);
    
      // update queue stats
      numTimes--;
      numEvents-=objects.length;
      
      return(objects);
    }
  }


  public int numTimes()
  {
    synchronized(lock)
    {
      return(numTimes);
    }
  }


  public int numEvents()
  {
    synchronized(lock)
    {
      return(numEvents);
    }
  }


  public void printQueue()
  {
    synchronized(lock)
    {
      _printQueue(0, queue);
    }
  }

  private void _printQueue(int indent, QueueElement q)
  {
    String ind = indent(indent);

    System.out.println(ind + "key = " + q.key);
    System.out.println(ind + "before = " + q.before);
    if (q.before != null) _printQueue(indent+1, q.before);
    System.out.println(ind + "after = " + q.after);
    if (q.after != null) _printQueue(indent+1, q.after);
    System.out.println(ind + "objects = " + q.objects);
    System.out.println(ind + "previous = " + q.previous);
    System.out.println(ind + "wasBefore = " + q.wasBefore);
    System.out.println(ind + "wasAfter = " + q.wasAfter);


  }

  private String indent(int level)
  {
    if (level == 0) return(new String(""));

    char[] tmp;

    tmp = new char[level];
    for(int i=0; i< level; i++)
      tmp[i] = '\t';
    
    return(new String(tmp));
  }


  /**
   * Recurses through the queue, looking for a TemporalQueueObject with the supplied 
   * key
   * 
   * @param el The Queue to search through
   * @param keyToLookFor The key to look for in the queue
   * @return A list indicating which QueueElement we got to, and whether the
   * a new QueueElement should be added BEFORE, or AFTER to handle the key,
   * or if a QueueElement for the key was found, EQUAL.
   */
  private Object[] recurseTo(QueueElement el, Object keyToLookFor)
  {
    // right, compare the key of the current QueueElement,
    // and the one we're looking for
    int compared = compareKeys(keyToLookFor, el.key);
    switch(compared)
    {
    // if there isn't a "before" element, then indicate why we got stuck,
    // and where we got to to the caller
    case BEFORE:
      if (el.before == null)
      {
	Object[] retval = {new Integer(BEFORE), (Object) el};
	return(retval);
      }

      // we didn't get stuck! keep recursing
      return(recurseTo(el.before, keyToLookFor));



    // if there isn't an "after" element, then indicate why we got stuck,
    // and where we got to to the caller
    case AFTER:
      if (el.after == null)
      {
	Object[] retval = {new Integer(AFTER), (Object) el};
	return(retval);
      }

      // we didn't get stuck! keep recursing
      return(recurseTo(el.after, keyToLookFor));


    // Woo! We found an appropriate QueueElement. Return it, and indicate
    // we were successful
    case EQUAL:
      Object[] retval = {new Integer(EQUAL), (Object) el};
      return(retval);
    }

    throw new Error(Integer.toString(compared));
  }




  /**
   * Recurses to the first element in the Queue.
   * 
   * @param el The Queue to use.
   * @return null if the queue is empty, or the QueueElement at the start
   */
  private QueueElement recurseToFirst(QueueElement el)
  {
    // first of all, recurse on the QueueElement BEFORE this
    // one, if it's present
    if (el.before != null)
      return(recurseToFirst(el.before));

    // Right, if there are some objects, then we know this is the
    // current queue element
    if (el.objects != null)
      return(el);

    // Therefore, we know that there MUST be elements after this	
    // QueueElement, otherwise it would have been tidied away
    if (el.after != null)
      return(recurseToFirst(el.after));

    // we shouldn't get here
    throw new Error("No data in QueueElement! (Tidying bug!)");
  }





  /**
   * Fixes up the Queue tree after an element deletion
   *
   * @param el the starting point to fix the Queue from
   */
  private void fixTree(QueueElement el)
  {
    // if there is anything pointed to by this element's
    // after, before, or object pointers, then we can't
    // remove it, or anything above it in the tree just now
    if ((el.before != null) ||
	(el.after != null) ||
	(el.objects != null))
      return;

    // Right, we've hit the top tree element, AND it has nothing in
    // it's pointers, so nullify the TemporalQueue.queue entry
    if (el.previous == null)
    {
      queue = null;
      return;
    }

    // OK, we've not at the top of the tree, but there is nothing in
    // this element. Wipe the pointer to this QueueElement from the
    // QueueElement which pointed to it (i.e. el.previous) accordingly
    if (el.wasBefore)
      el.previous.before = null;
    if (el.wasAfter)
      el.previous.after = null;

    // recurse back up to the previous element
    fixTree(el.previous);
  }



  /**
   * Compares two key objects together. Returns BEFORE, AFTER or EQUAL,
   * depending on the relationship between e1 & e2 (i.e. if e1 < e2, return 
   * BEFORE
   */
  protected abstract int compareKeys(Object e1, Object e2);
  





  /**
   * An element in the queue (note: distinct from the TemporalQueueObjects 
   */
  private final class QueueElement
  {
    /**
     * The key (this time the events in this element should "occur at")
     */
    public Object key;


    /**
     * Pointer to chain of QueueElements whose events occurbefore this one
     * i.e. temporally before
     */
    public QueueElement before = null;


    /**
     * Pointer to chain of QueueElements whose events occur after this one
     * i.e. temporally after
     */
    public QueueElement after = null;


    /**
     * The list of events which occur at time = key
     */
    public ItemList objects = new ItemList();



    /**
     * Pointer to the QueueElement preceding this one in the queue
     * i.e. before this in terms of the queue only
     */
    public QueueElement previous = null;


    /**
     * Indicators as to whether this QueueElement was pointed to the by the 
     * before or after pointers of the "previous" QueueElement
     */
    public boolean wasBefore = false;
    public boolean wasAfter = false;


    private QueueElement(){}
    public QueueElement(Object key)
    {
      this.key = key;
    }
  }
}
