/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.midiplayer;
import herman.midi.*;
import herman.misc.*;


/**
 * This implements the actual player functionality. It plays MidiEvents
 * (hopefully) at their correct times, using non-busy waiting
 */
public class MidiPlayerThread extends Thread
{
  /**
   * Just to remember the system clock time when we started
   */
  private long startTime = 0;

  /**
   * The device to output MIDI to
   */
  private MidiOut outDevice;



  /** 
   * Creates a new MidiPlayerThread, using outDevice as the MIDI output device
   *
   * @param outDevice The MIDI device to output to
   */
  protected MidiPlayerThread(MidiOut outDevice)
  {
    this.outDevice = outDevice;
  }
      

  /**
   * run method of this thread
   */
  public void run()
  {
    long curTime;


    // get the current tick starting time 
    this.startTime = System.currentTimeMillis();


    // this runs forever (well, sort of, ish, like)
    while(true)
    {
      // work out the current time, relative to where we started
      curTime = System.currentTimeMillis() - startTime;


      // get the next event from the queue
      Object[] queuedEvents = MidiPlayer.queue.firstElement();
      

      // eek! we've no queue elements left, so suspend ourselves until
      // we're informed we have some (and ask for some more)
      if ((queuedEvents == null) || (queuedEvents.length == 0))
      {
	herman.eventprocess.EventProcess.activate();
	MidiPlayer.playThreadSuspended = true;
	this.suspend();
	continue;
      }
      

      // right, if we're ahead of time for this event, we'll just sleep until
      // we're ready (and keep the current time updated)
      // (just use the first item in the queue, since we KNOW that will be 
      // there)
      if (((MidiTemporalQueueObject) queuedEvents[0]).time > curTime)
      {
	try
	{
	  this.sleep(((MidiTemporalQueueObject) queuedEvents[0]).time - curTime);
	}

	// if we get this, then we've been told to stop sleeping prematurely,
	// therefore, we must have recieved an event which should start sooner
	// therefore, we *continue* so that we are looking at the correct 
	// element (we may still need to sleep a bit)
	catch (InterruptedException e)
	{
	  continue;
	}

	curTime = System.currentTimeMillis() - startTime;
      }
      
      // right, we're definitely going to play this NOW, so remove it
      // from the queue
      MidiPlayer.queue.firstElementAndRemove();


      // OK, we should be ready to play this event. If we've already
      // missed the time for it, we'll just ignore it.
      long difference = 
	((MidiTemporalQueueObject) queuedEvents[0]).time - curTime;

      // this checks if the difference between the times is less than
      // 200 milliseconds (if so, then it plays!)
      if ((difference*difference) < 40000)
      {
	for(int i=0; i< queuedEvents.length; i++)
	{
	  try
	  {
	    (((MidiTemporalQueueObject) queuedEvents[i]).event).send(outDevice);
	  }
	  catch (InvalidMIDIChannelException e)
	  {
	    throw new FatalError(e.toString());
	  }
	  catch (NotInitialisedException e)
	  {
	    throw new FatalError(e.toString());
	  }
	  catch (MIDIDeviceBusyException e)
	  {
	    throw new FatalError(e.toString());
	  }	   
	  catch (UnknownInstrumentException e)
	  {
	    throw new FatalError(e.toString());
	  }	   
	}

	// update last played time
	MidiPlayer.lastPlayedTime = 
	  ((MidiTemporalQueueObject) queuedEvents[0]).time;
      }

      // only a few things left => get some more stuff from rhythm
      if (MidiPlayer.queue.numTimes() <= 3)
	herman.eventprocess.EventProcess.activate();
    }
  }
}

