/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.midiplayer;
import herman.misc.*;
import herman.midi.*;

/**
 * This is the publicly accessible interface to the MidiPlayer
 */
public class MidiPlayer
{
  /**
   * Pointer to the Midi player thread
   */
  protected static MidiPlayerThread playThread = null;

  /**
   * The queue of elements to play
   */
  protected static MidiTemporalQueue queue = null;

  /**
   * Flag indicating if the player thread has suspended itself due to lack
   * of queue elements or not
   */
  protected static boolean playThreadSuspended = false;

  /**
   * The time the current head-of-queue will be played at
   */
  protected static long curFirstTime = -1;


  public static long lastPlayedTime = -1;



  /**
   * No institations
   */
  private MidiPlayer(){}


  /**
   * Initialises the MidiPlayer to use the output device outDevice
   *
   * @param outDevice The output device to use
   */
  public static void init(MidiOut outDevice)
  {
    // make a new temporal queue to hold the events
    queue = new MidiTemporalQueue();

    // set variables up
    curFirstTime = -1;
    playThreadSuspended = false;

    // start up player thread for this device
    playThread = new MidiPlayerThread(outDevice);
    playThread.start();
  }



  /**
   * Sends a MidiEvent to the MidiPlayer to be played. It will be queued until
   * it's time has come (eek!)
   *
   * @param event The event to send
   */
  public static void send(MidiEvent event)
  {
    // ignore any events arriving too late
    if (event.time <= lastPlayedTime)
      return;
    
    // add the event into the queue
    queue.addElement(new MidiTemporalQueueObject(event));

    // Now, if the event's start time is less than the one the player is
    // currently sleeping till, wake it up NOW, instead of later
    if (event.time <= curFirstTime)
    {
      curFirstTime = event.time;
      playThread.interrupt();
    }

    if (curFirstTime == -1)
      curFirstTime = event.time;

    // wake up the player thread if need be
    if (playThreadSuspended)
      playThread.resume();
  }


  /**
   * Kills off the MidiPlayer. You can restart it by calling init() again
   */
  public static void kill()
  {
    playThread.stop();
    playThread = null;
    queue = null;
  }
}
