/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.midi;
import herman.misc.*;

public class MidiOut
{
  private MidiDevice device;
  private int handle;
  private int[] lastInstrument;

  private int MODE_CHANNEL =0;

  public static int MODE_LOCALCONTROL=122;
  public static int MODE_ALLNOTESOFF=123;
  public static int MODE_OMNIMODEOFF=124;
  public static int MODE_OMNIMODEON=125;
  public static int MODE_MONOPHONICMODE=126;
  public static int MODE_POLYPHONICMODE=127;


  /**
   * This would be meaningless, so I've stopped it
   */
  private MidiOut() {}

  public MidiOut(MidiDevice device)
    throws InvalidMIDIDeviceException, NotInitialisedException,
	   MIDIPortNotFoundException, MIDIDeviceBusyException,
	   InvalidMIDIChannelException
  {
    this.device = device;
    this.handle = open(device.getDeviceID());

    lastInstrument = new int[device.getNumVoices()];
    for(int i=0; i< lastInstrument.length; i++)
    {
      lastInstrument[i] = 0;
      changeProgram(i, 0);
    }
  }
  
  
  
  private int open(int deviceID)
    throws InvalidMIDIDeviceException, NotInitialisedException,
	   MIDIPortNotFoundException, MIDIDeviceBusyException
  {
    int handle;

    switch(handle = midiOutLowOpenDevice(deviceID))
    {
    case Midi.MIDI_NOTINITIALISED:
      throw new NotInitialisedException();
    case Midi.MIDI_FATALERROR:
      throw new FatalError();
    case Midi.MIDI_PORTNOTFOUND:
      throw new MIDIPortNotFoundException();
    case Midi.MIDI_INVALIDDEVICEID:
      throw new InvalidMIDIDeviceException();
    case Midi.MIDI_DEVICEBUSY:
      throw new MIDIDeviceBusyException();
    }

    return(handle);
  }



  public void close()
    throws NotInitialisedException, MIDIStillPlayingException,
	   BadHandleException, NotOpenedException
  {
    switch(midiOutLowCloseDevice(handle))
    {
    case Midi.MIDI_NOTINITIALISED:
      throw new NotInitialisedException();
    case Midi.MIDI_FATALERROR:
      throw new FatalError();
    case Midi.MIDI_STILLPLAYING:
      throw new MIDIStillPlayingException();
    case Midi.MIDI_BADHANDLE:
      throw new BadHandleException();
    case Midi.MIDI_NOTOPENED:
      throw new NotOpenedException();
    }
  }


  public MidiDevice getDevice()
  {
    return(device);
  }
      




  public void noteOn(int channel, int pitch, int velocity, String inst)
    throws InvalidMIDIChannelException, NotInitialisedException,
	   MIDIDeviceBusyException, UnknownInstrumentException
  {
    // check channel is OK
    if ((channel <0) || (channel > device.getNumVoices()))
      throw new InvalidMIDIChannelException();

    // get the instrument number
    int instrument = Midi.translateProgram(inst);
    
    // work out whether we should send a new instrument selection
    if (instrument != lastInstrument[channel])
    {
      changeProgram(channel, instrument);
      lastInstrument[channel] = instrument;
    }

    // send the MIDI message
    switch(midiOutLowSendMessage(handle, 
				 Midi.MIDI_NOTEON | channel, pitch, velocity))
    {
    case Midi.MIDI_NOTINITIALISED:
      throw new NotInitialisedException();

    case Midi.MIDI_FATALERROR:
      throw new FatalError();

    case Midi.MIDI_DEVICEBUSY:
      throw new MIDIDeviceBusyException();      
    }
  }

  public void noteOff(int channel, int pitch, int velocity)
    throws InvalidMIDIChannelException, NotInitialisedException,
	   MIDIDeviceBusyException
  {
    // check channel is OK
    if ((channel <0) || (channel > device.getNumVoices()))
      throw new InvalidMIDIChannelException();

    // send the MIDI message
    switch(midiOutLowSendMessage(handle, 
				 Midi.MIDI_NOTEOFF |channel, pitch, velocity))
    {
    case Midi.MIDI_NOTINITIALISED:
      throw new NotInitialisedException();

    case Midi.MIDI_FATALERROR:
      throw new FatalError();

    case Midi.MIDI_DEVICEBUSY:
      throw new MIDIDeviceBusyException();      
    }
  }


  public void changeProgram(int channel, int instrument)
    throws InvalidMIDIChannelException, NotInitialisedException,
	   MIDIDeviceBusyException
  {
    // check channel is OK
    if ((channel <0) || (channel > device.getNumVoices()))
      throw new InvalidMIDIChannelException();

    // send the MIDI message
    switch(midiOutLowSendMessage(handle, 
				 Midi.MIDI_PROGRAMCHANGE |channel, 
				 instrument, 0))
    {
    case Midi.MIDI_NOTINITIALISED:
      throw new NotInitialisedException();

    case Midi.MIDI_FATALERROR:
      throw new FatalError();

    case Midi.MIDI_DEVICEBUSY:
      throw new MIDIDeviceBusyException();      
    }
  }



  
  public void modeMessage(int MsgType)
    throws InvalidMIDIChannelException, NotInitialisedException,
	   MIDIDeviceBusyException
  {
    modeMessage(MsgType, 0);
  }

  public void modeMessage(int MsgType, int param)
    throws InvalidMIDIChannelException, NotInitialisedException,
	   MIDIDeviceBusyException
  {
    // check message type is OK
    if ((MsgType < 122) || (MsgType > 127))
      throw new InvalidMIDIChannelException();
      

    // send the MIDI message
    switch(midiOutLowSendMessage(handle, 
				 0xb0 | MODE_CHANNEL, MsgType, param))
    {
    case Midi.MIDI_NOTINITIALISED:
      throw new NotInitialisedException();

    case Midi.MIDI_FATALERROR:
      throw new FatalError();

    case Midi.MIDI_DEVICEBUSY:
      throw new MIDIDeviceBusyException();      
    }
  } 




  protected static MidiDevice[] init()
    throws AlreadyInitialisedException
  {
    // load the library file
    System.load("midiout");

    // init the low level stuff
    switch(midiOutLowInit())
    {
    case Midi.MIDI_ALREADYINITIALISED:
      throw new AlreadyInitialisedException();
    case Midi.MIDI_FATALERROR:
      throw new FatalError();
    }
    
    // work out the array of (functional) MIDI devices
    MidiDevice[] devices = new MidiDevice[0];
    for(int i=0; i< midiOutLowGetNumDevices(); i++)
    {
      // get device information
      MidiDevice newDev = midiOutLowGetDeviceInfo(i);
      
      // skip to next if device is invalid in some way
      if (newDev == null)
	continue;
      
      // add new device into array
      MidiDevice[] tmpDevices = new MidiDevice[devices.length +1];
      System.arraycopy(devices, 0, tmpDevices, 0, devices.length);
      tmpDevices[devices.length] = newDev;
      devices = tmpDevices;
    }    

    return(devices);
  }



  protected static void shutdown()
    throws NotInitialisedException
  {
    switch(midiOutLowShutdown())
    {
    case Midi.MIDI_NOTINITIALISED:
      throw new NotInitialisedException();
    case Midi.MIDI_FATALERROR:
      throw new FatalError();
    }
  }
      




  // native calls to MIDI subsystem
  private synchronized static native int midiOutLowInit();
  private synchronized static native int midiOutLowShutdown();
  private synchronized static native int midiOutLowOpenDevice(int deviceNum);
  private synchronized static native int midiOutLowCloseDevice(int handle);
  private synchronized static native int midiOutLowSendMessage(int handle,
							       int statusByte, 
							       int dataByte1, 
							       int dataByte2);
  private synchronized static native int midiOutLowGetNumDevices();
  private synchronized static native MidiDevice midiOutLowGetDeviceInfo(int deviceID);
}









