/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

/**
 * MidiOut.c
 * (c) 1998 Andrew de Quincey
 * Version: 0.1
 *
 * Provides low level windows-based MIDI functionality
 */


#include <stdio.h>
#include <stdlib.h>
#include <windows.h>
#include <mmsystem.h>
#include <mmreg.h>
#include <jni.h>

#include "MidiMisc.h"
#include "herman_midi_MidiOut.h"


static HMIDIOUT** __midiLowHandleTable;
static MIDIOUTCAPS** __midiLowCapabilityTable;
static int* __midiLowLastStatusSentTable;
static int __midiLowInitialised = 0;
static int __midiLowNumDevices = 0;

static int __midiLowGetDeviceCapabilities(void);
static HMIDIOUT* __midiLowGetHandle(int handle, int* status);


JNIEXPORT jint JNICALL Java_herman_midi_MidiOut_midiOutLowInit
  (JNIEnv *env, jclass class)
{
  int i;
  int status;

  // check we've not already been setup
  if (__midiLowInitialised)
    return(MIDI_ERRALREADYINITIALISED);


  // get the number of devices present in the system
  __midiLowNumDevices = midiOutGetNumDevs();

  
  // setup memory for handle & capabilities tables
  if ((__midiLowHandleTable =
       (HMIDIOUT**) malloc(sizeof(HMIDIOUT*) * __midiLowNumDevices)) == NULL)
    return(MIDI_ERRFATALERROR);

  if ((__midiLowCapabilityTable =
       (MIDIOUTCAPS**) malloc(sizeof(MIDIOUTCAPS*) * __midiLowNumDevices)) == NULL)
  {
    free(__midiLowHandleTable);
    return(MIDI_ERRFATALERROR);
  }

  if ((__midiLowLastStatusSentTable =
       (int*) malloc(sizeof(int) * __midiLowNumDevices)) == NULL)
  {
    free(__midiLowHandleTable);
    free(__midiLowCapabilityTable);
    return(MIDI_ERRFATALERROR);
  }


  // set all the entries to NULL
  for(i=0; i< __midiLowNumDevices; i++)
  {
    __midiLowHandleTable[i] = NULL;
    __midiLowCapabilityTable[i] = NULL;
    __midiLowLastStatusSentTable[i] = 0;
  }

  
  // OK, determine individual MIDI device's capabilities
  status = __midiLowGetDeviceCapabilities();
  if (status < 0)
  {
    free(__midiLowHandleTable);
    free(__midiLowCapabilityTable);
    free(__midiLowLastStatusSentTable);
    return(status);
  }

  // everything went OK!!!
  __midiLowInitialised = 1;
  return((jint) MIDI_OK);
}



JNIEXPORT jint JNICALL Java_herman_midi_MidiOut_midiOutLowShutdown
  (JNIEnv *env, jclass class)
{
  int i;

  // check we've been setup
  if (!__midiLowInitialised)
    return(MIDI_ERRNOTINITIALISED);


  // zip through the tables, clearing them
  for(i=0; i< __midiLowNumDevices; i++)
  {
    if (__midiLowHandleTable[i] != NULL)
    {
      midiOutClose(__midiLowHandleTable[i]);
      free(__midiLowHandleTable[i]);
    }
    
    if (__midiLowCapabilityTable[i] != NULL)
      free(__midiLowCapabilityTable[i]);
  }
  
  // clear up the last few things
  free(__midiLowHandleTable);
  free(__midiLowCapabilityTable);
  free(__midiLowLastStatusSentTable);
  __midiLowNumDevices = 0;

  
  // all went OK
  __midiLowInitialised = 0;  
  return(MIDI_OK);
}
      


JNIEXPORT jint JNICALL Java_herman_midi_MidiOut_midiOutLowOpenDevice
  (JNIEnv *env , jclass class, jint deviceID)
{
  HMIDIOUT* handle;
  int status;

  // check we've been setup
  if (!__midiLowInitialised)
    return(MIDI_ERRNOTINITIALISED);

  // get memory for handle
  if ((handle = (HMIDIOUT*) malloc(sizeof(HMIDIOUT))) == NULL)
    return(MIDI_ERRFATALERROR);
  
  // open MIDI device & check for errors
  status = (int) midiOutOpen(handle, deviceID, 0, 0, CALLBACK_NULL);
  switch(status)
  {
  case MMSYSERR_BADDEVICEID:
    free(handle);
    return(MIDI_ERRINVALIDDEVICEID);

  case MMSYSERR_INVALPARAM:
  case MMSYSERR_NOMEM:
    free(handle);
    return(MIDI_ERRFATALERROR);
    
  case MIDIERR_NODEVICE:
    free(handle);
    return(MIDI_ERRPORTNOTFOUND);

  case MMSYSERR_ALLOCATED:
    free(handle);
    return(MIDI_ERRDEVICEBUSY);
  }

  // keep a note of the Windows MIDI device handle
  __midiLowHandleTable[deviceID] = handle;

  // we'll just use the device number as a handle; it's unique per device, 
  // so why not?
  return(deviceID);
}



JNIEXPORT jint JNICALL Java_herman_midi_MidiOut_midiOutLowCloseDevice
  (JNIEnv *env, jclass class, jint handle)
{
  int status;
  HMIDIOUT* windowsHandle;

  // check we've been setup
  if (!__midiLowInitialised)
    return(MIDI_ERRNOTINITIALISED);


  // work out the windows handle & check it's OK
  if ((windowsHandle = __midiLowGetHandle(handle, &status)) == NULL)
    return(status);

  // close the device
  status = (int) midiOutClose(*windowsHandle);
  switch(status)
  {
  case MIDIERR_STILLPLAYING:
    return(MIDI_ERRSTILLPLAYING);
    
  case MMSYSERR_INVALHANDLE:
  case MMSYSERR_NOMEM:
    return(MIDI_ERRFATALERROR);
  }

  // OK, free up memory & update MIDI handle table
  free(windowsHandle);
  __midiLowHandleTable[handle] = NULL;


  // twelve of the clock, and all's well
  return(MIDI_OK);
}


JNIEXPORT jint JNICALL Java_herman_midi_MidiOut_midiOutLowSendMessage
   (JNIEnv* env, jclass class,
    jint handle, jint statusByte, jint dataByte1, jint dataByte2)
{
  union
  {
    DWORD dwData;
    BYTE bData[4];
  } u;
  HMIDIOUT* windowsHandle;
  int status;

  // check we've been setup
  if (!__midiLowInitialised)
    return(MIDI_ERRNOTINITIALISED);

  // get handle for this device & check it's OK
  if ((windowsHandle = __midiLowGetHandle(handle, &status)) == NULL)
    return(status);


  // setup message to be sent
  if (__midiLowLastStatusSentTable[handle] == statusByte)
  {
    // don't need to send the statusByte again if it's the same as the 
    // previous one
    u.bData[0] = dataByte1 & 0xff;
    u.bData[1] = dataByte2 & 0xff;
    u.bData[2] = 0;    
    u.bData[3] = 0; 
  }
  else
  {
    u.bData[0] = statusByte & 0xff;
    u.bData[1] = dataByte1 & 0xff;
    u.bData[2] = dataByte2 & 0xff;
    u.bData[3] = 0;
  }
  

  // send it!
  status = midiOutShortMsg(*windowsHandle, u.dwData);
  switch(status)
  {
  case MIDIERR_BADOPENMODE:
    return(MIDI_ERRFATALERROR);

  case MIDIERR_NOTREADY:
    return(MIDI_ERRDEVICEBUSY);
    
  case MMSYSERR_INVALHANDLE:
    return(MIDI_ERRFATALERROR);
  }


  // remember the last status byte sent on this channel
  __midiLowLastStatusSentTable[handle] = statusByte;


  // Woo!
  return(MIDI_OK);
}



JNIEXPORT jint JNICALL Java_herman_midi_MidiOut_midiOutLowGetNumDevices
  (JNIEnv * env, jclass class)
{
  // check we've been setup
  if (!__midiLowInitialised)
    return(MIDI_ERRNOTINITIALISED);

  return(__midiLowNumDevices);
}  

JNIEXPORT jobject JNICALL Java_herman_midi_MidiOut_midiOutLowGetDeviceInfo
  (JNIEnv* env, jclass class, jint deviceID)
{
  jobject deviceInfo;
  jclass midiDeviceClass;

  // check deviceID in range
  if ((deviceID < 0) || (deviceID >= __midiLowNumDevices))
    return(NULL);

  // check the device was detected OK
  if (__midiLowCapabilityTable[deviceID] == NULL)
    return(NULL);
  
  // find the MidiDevice class
  midiDeviceClass = (*env)->FindClass(env, "herman/midi/MidiDevice");
  
  // make up new MidiDevice class
  deviceInfo = 
    (*env)->NewObject(env, 
		      midiDeviceClass, 
		      (*env)->GetMethodID(env, 
					  midiDeviceClass, 
					  "<init>", 
					  "(IILjava/lang/String;III)V"),
		      __midiLowCapabilityTable[deviceID]->wMid,
		      __midiLowCapabilityTable[deviceID]->wPid,
		      (*env)->NewStringUTF(env,__midiLowCapabilityTable[deviceID]->szPname),
		      __midiLowCapabilityTable[deviceID]->wVoices,
		      __midiLowCapabilityTable[deviceID]->wNotes,
		      deviceID);
  
  return(deviceInfo);
}





static HMIDIOUT* __midiLowGetHandle(int handle, int* status)
{
  HMIDIOUT* windowsHandle;

  // check handle in range
  if ((handle < 0) || (handle >= __midiLowNumDevices))
  {
    *status = MIDI_ERRBADHANDLE;
    return(NULL);
  }

  // check device was detected OK when we started up
  if (__midiLowCapabilityTable[handle] == NULL)
  {
    *status = MIDI_ERRBADHANDLE;
    return(NULL);
  }

  // get the handle
  windowsHandle = __midiLowHandleTable[handle];

  // check if it has actually been opened
  if (windowsHandle == NULL)
  {
    *status = MIDI_ERRNOTOPENED;
    return(NULL);
  }

  // all went well!
  return(windowsHandle);
}




static int __midiLowGetDeviceCapabilities(void)
{
  MMRESULT result;
  MIDIOUTCAPS* capabilities;
  int i;


  // get device capabilities
  for(i=0; i < __midiLowNumDevices; i++)
  {
    // get memory & setup
    __midiLowCapabilityTable[i] = NULL;
    if ((capabilities = (MIDIOUTCAPS*) malloc(sizeof(MIDIOUTCAPS))) == NULL)
    {
      // error => free memory & exit
      for(i =0; i < __midiLowNumDevices; i++)
	if (__midiLowCapabilityTable[i] != NULL)
	  free(__midiLowCapabilityTable[i]);

      return(MIDI_ERRFATALERROR);
    }

    // actually get the device capabilities
    result = midiOutGetDevCaps(i, capabilities, sizeof(MIDIOUTCAPS));

    // check for errors
    switch(result)
    {
    case MMSYSERR_BADDEVICEID:
    case MMSYSERR_INVALPARAM:
      // error => free memory & exit
      for(i =0; i < __midiLowNumDevices; i++)
	if (__midiLowCapabilityTable[i] != NULL)
	  free(__midiLowCapabilityTable[i]);
      return(MIDI_ERRFATALERROR);

    case MMSYSERR_NODRIVER: 
      // we'll just ignore these, but we'll mark them by setting their
      // capabilities entry to NULL
      free(capabilities);
      capabilities = NULL;
      continue;
    }

    // update the device capabilities table
    __midiLowCapabilityTable[i] = capabilities;
  }
  
  return(MIDI_OK);
}
