/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.form;
import herman.elements.*;
import herman.properties.*;
import herman.form.types.*;
import herman.form.transforms.*;
import herman.mapper.Mapper;
import herman.pitch.harmony.KeyControl;
import herman.pitch.io.PitchInstInput;
import java.util.Vector;
import java.util.Enumeration;


/** 
 * The PhraseForm class plans the basic form of each phrase.
 */
 
public class PhraseForm {
  private static PhraseType[] currentSentence = new PhraseType[1]; 
  private static boolean sentenceEnd = false;
  private static int sentenceIndex = 10;
  public static long basePhraseDuration = 10000;
  private static boolean runOnce = false;

  //original phrases that have been used in this sentence
  private static Vector usedPhrases = new Vector();

/**
 *  The currently active phrase
 */
  public static Phrase currentPhrase;

/** 
 *  The percentage of time that a new phrase is used rather than an old phrase
 */
  public static float variety = 50.0f;

/** 
 *  The current tension level : this will be replace
 */
  public static long currentTension = 0;

/** 
 *  The valid range around the current tension level (in either direction).
 *  Phrases with tension levels within this range can be used. 
 */
  public static long validTensionRange = 5;

/**
 *  Initialize the PhraseForm module
 */
  public static void initialize() {
    currentPhrase = new Phrase();
    currentPhrase.phraseInfo.setLocation("sentence start");
    Labeller.newPhrase = true;
  }

/* 
 * Store the new phrases used in this sentence if the tension level did not
 *  change too drastically during their generation.
 */
  private static void storeUsed() {
    Phrase phrase = null;
    if (usedPhrases.size() > 0) {
	Enumeration elems = usedPhrases.elements();
	while (elems.hasMoreElements()) {
	  phrase = (Phrase)elems.nextElement();
	if ((!(phrase.phraseInfo.getSecond().equals("large tension change")))&&
	     (phrase.transformation.getStatus().equals("new"))) {
	  PhraseLibrary.storePhrase(phrase);
	}
	}
    }
    //Replace the phrases that were drawn from the Library:
    PhraseLibrary.replacePhrases();
  }

/* 
 * Get the next phrase
 */
  private static Phrase getNextPhrase() {
    Phrase result = null;

    //If need new sentence, get one
    if ((sentenceIndex > (currentSentence.length - 1))||
	(sentenceEnd)) {
      
       //Store usedPhrases:
      System.out.println("Storing original phrases for last sentence...");
      storeUsed();

      //Get new sentence
      System.out.println("Getting new sentence...");
      currentSentence = SentenceForm.getNewSentence();
      sentenceIndex = 0;
      sentenceEnd = false;

      //Kill after one sentence, temporarily
      //if (runOnce) {
      //while (true) { }
      //}
     
      runOnce = true;
      //Set new values
      usedPhrases.removeAllElements();
      result = getNextPhrase();
      result.phraseInfo.setLocation("sentence start");
    } 

    //Otherwise, ++ the index and return the phrase
    else {
      //apply the sentenceType rules for the phrase position
      SentenceType s = SentenceForm.currentSentenceType;
      s.applyPhrasePositionRules(sentenceIndex);

      result = instantiatePhraseType(currentSentence[sentenceIndex]);

      //check transformation info, and assign as necessary:
      if (!(result.transformation.getStatus().equals("new"))) {
	s.applyPhraseTransformations(result, sentenceIndex);
      }

      if (sentenceIndex == (currentSentence.length - 1)) {
	result.phraseInfo.setLocation("sentence end");
	
	//Clear the resolution if not a new phrase:
	if (!(result.transformation.getStatus().equals("new"))) {
	  HarmonyClearResolution hcr = new HarmonyClearResolution();
	  MelodyClearResolution mcr = new MelodyClearResolution();
	  result.transformation.addTransform(hcr);
	  result.transformation.addTransform(mcr);
	  result.transformation.setStatus("transform");
	}
      }

      sentenceIndex++;
    }
    return result;
  }

/*
 *  Replace the supplied phrase type with a phrase
 */
  private static Phrase instantiatePhraseType(PhraseType phraseType) {
    Phrase phrase = null;

    if (usedPhrases.size() < (phraseType.ID+1)) {  /*this phrase hasn't been
						    * used in this sentence */
      float rand = ((float)java.lang.Math.random())*100.0f;
      if ((rand <= variety)||(!PhraseLibrary.hasPhrases())) {   //New phrase
	phrase = new Phrase();
	phrase.transformation.setStatus("new");
	phrase.duration = basePhraseDuration;
      } else {                                        //use an old phrase
	phrase = PhraseLibrary.getOrigPhrase();
	phrase.transformation.setStatus("direct copy");

	System.out.println("GETTING A PHRASE FROM THE LIBRARY");
	//Check to make sure it matches the current key, if not, make it match
	matchCurrentKey(phrase);
	
	//recreate links between harmony and melody
	Labeller.reCreateLinks(phrase);

      }
      
      //add it to the used phrases vector:
      usedPhrases.addElement(phrase);
      
    }

    else {   /*  This phrase was already used */
     phrase = ((Phrase)((Phrase)usedPhrases.elementAt(phraseType.ID)).clone());

     //determine if transformation required
     if (phraseType.transformation) {
       phrase.transformation.setStatus("transform");
     } else {phrase.transformation.setStatus("direct copy");}

     
     //Check to make sure it matches the current key, if not, make it match
     matchCurrentKey(phrase);

     //recreate links between harmony and melody:
     Labeller.reCreateLinks(phrase);

     //If tension level changed too much, replace with new phrase:
     if ((phrase.tension > (currentTension + validTensionRange))||
	 (phrase.tension < (currentTension - validTensionRange))||
	 (phrase.phraseInfo.getSecond().equals("large tension change"))) {
       phrase = new Phrase();
       phrase.transformation.setStatus("new");
       phrase.duration = basePhraseDuration;
     }
     else {        //diagnostic printing:
       System.out.println("Used Phrase # of chords: "+phrase.partChord.size());
     }
    }
    
    return phrase;
  }
  
/**
 *  Force the current sentence to end at the end of this phrase
 */
  public static void killSentence() {
    sentenceEnd = true;
  }

/* 
 * Assign transformations to a phrase
 */
  private static void assignTransformation(Phrase phrase) {
    //Don't have any yet

  }

/**
 *  Request that the current phrase be terminated and a new phrase started.
 */
  public static void getNewPhrase() {
    //store the current phrase
    //System.out.println("Storing old phrase...");
    //storeCurrent();

    System.out.println("Getting new phrase...");

    //Clear the sentence location information from the old phrase
    currentPhrase.phraseInfo.setLocation(" ");

    //update phrase tension level:
    if ((currentPhrase.tension > (currentTension + validTensionRange))||
	(currentPhrase.tension < (currentTension - validTensionRange))) {
	currentPhrase.phraseInfo.setSecond("large tension change");
    } else  {
      currentPhrase.tension -= ((currentPhrase.tension - currentTension)/2);
    }					       

    //Allow phrase change developments from the mapper:
    Mapper.newPhrase();

    //get the next phrase (getting a new sentence if necessary)
    currentPhrase = getNextPhrase();

    //Call rhythm changes
    Mapper.applyRhythm(currentPhrase);

    //assign tension level:
    currentPhrase.tension = currentTension;

    //Alert the Labeller, Contour, & Pitch that starting a new phrase
    Labeller.newPhrase = true;
    Contour.reset();
    PitchInstInput.newPhrase();

    //Send on to the transformer:
    Transformer.process(currentPhrase);
  }
								

/**
 *  Return the location of the current phrase within the sentence.  Generally,
 *  will be "sentence start", "middle", or "sentence end".
 */
  public static String getCurrentLocation() {
    return currentPhrase.phraseInfo.getLocation();
  }

/* 
 *  Makes sure that the phrase matches the current key.  If mode doesn't
 *  match, the harmony information is cleared.
 */
  private static void matchCurrentKey(Phrase phrase) {
    int currentKT = KeyControl.getIntCurrentKeyType();
    String currentMode = KeyControl.getCurrentMode();
    String phraseMode = ((Chord)phrase.partChord.getFirst()).key.getMode();
    int phraseKT = ((Chord)phrase.partChord.getFirst()).key.getIntKeyType();

    if (phraseKT != currentKT) {
      Key key = new Key(currentKT, currentMode);
      HarmonyKeyChange hkc = new HarmonyKeyChange(key);
      MelodyDirectKeyChange mdkc = new MelodyDirectKeyChange(key);
      phrase.transformation.addTransform(hkc);
      phrase.transformation.addTransform(mdkc);
      phrase.transformation.setStatus("transform");
    }

    if (!(phraseMode.equals(currentMode))) {
      HarmonyClearAll hca = new HarmonyClearAll();
      phrase.transformation.addTransform(hca);
      phrase.transformation.setStatus("transform");
    }
 
  }


  public static void main(String[] args) {
    PhraseForm.initialize();
    SentenceForm.initialize();
    getNewPhrase();
  }
}
