/* 
   Herman
   (c) 1998 Andrew de Quincey, adq@tardis.ed.ac.uk
   (c) 1998 Thomas Stapleford
   See README.TXT for copying/distribution/modification details.
*/

package herman.form;
import herman.elements.*;
import herman.properties.*;
import herman.pitch.harmony.Queue;
import herman.pitch.io.*;
import herman.player.Player;
import java.util.Enumeration;

/** 
 * The Labeller class attaches pitch information labels to the elements and 
 *   creates the connections between associated melodic events and chords.
 */
 
public class Labeller {

/** 
 *  The in queue for accompaniment events and chords 
 */
  public static Queue accompInQueue = new Queue();
/** 
 *  The in queue for melody events and event groups 
 */
  public static Queue melodyInQueue = new Queue();
/** 
 *  The in queue for melody events
 */
  public static Queue percussionInQueue = new Queue();
/** 
 *  A boolean indicating whether the elements in the in queues are the last
 *   elements for this phrase.
 */
  public static boolean phraseEnd = false;
/** 
 *  A boolean indicating whether the elements in the in queues are the first
 *   elements for this phrase.
 */
  public static boolean newPhrase = false;  

  private static Queue melHoldQueue = new Queue();
  private static Queue accHoldQueue = new Queue();
  private static long phraseMelProgress = 2;
  private static long phraseChordProgress = 2;
  private static long chordProgress = 0;
  private static Chord currentChord;
  private static Event currentMel; 
  private static long lastOverlap = 0;
  private static Chord lastChord;
  private static long threshold = 1;

/**
 *  Sets the threshold time used to permit suspensions
 */
  public static void setThreshold(long thresh) {
    threshold = thresh;
  }
 

/**
 *  Activates the Labeller module, causing it to process the elements in its
 *  in queues and pass them on to the Pitch Instantiator.
 */
  public static void activate() {
 
    long phraseLength = PhraseForm.currentPhrase.duration;
    long chordNum = (long)PhraseForm.currentPhrase.partChord.size();
    if (newPhrase) {
      //Alert the Player that a new phrase is coming
      Player.newPhrase(PhraseForm.currentPhrase.duration);
      newPhrase = false;
    }
    boolean saveLabels = false;

    melHoldQueue.removeAllElements();
    accHoldQueue.removeAllElements();

    MusicElement elem;
    Event event = null;

     //First handle the melody:
    while (melodyInQueue.size() > 0) {
      elem = (MusicElement)melodyInQueue.pop();
      if (elem.getMusicDefn().equals("note")) { 
	event = (Event) elem;

	//check to see if phrase over:
	if (event.startTime < phraseMelProgress) {  //This is a new phrase
	  if (PhraseForm.currentPhrase.phraseInfo.getLocation().equals(
    						       "sentence start")) {
	    event.eventInfo.setLocation("sentence start");
	  }
	  else {event.eventInfo.setLocation("phrase start");} 
	  phraseEnd = false;
	  //Can't have stuff hanging over phrases, so reset currentMel
	  currentMel = null;
	}
	
	phraseMelProgress = event.startTime + event.duration;
	if (phraseMelProgress >= PhraseForm.currentPhrase.duration) {
	  event.eventInfo.setLocation("resolution");
	}
	
       	saveLabels = false;
	if (event.supConChord.size() > 0) {
	  saveLabels = true;
	}
	
	if (!saveLabels) {
	  event.supConChord.removeElems();
	  melHoldQueue.push(event);
    	}

        PitchInstInput.melodyInQueue.push(event);
      }
    }
    //Label the end of the phrase (if not already done above):
    if ((event != null)&&(phraseEnd)) {
      event.eventInfo.setLocation("resolution");
    }
      

     //Next handle the accompaniment:
    while (accompInQueue.size() > 0) {
      elem = (MusicElement)accompInQueue.pop();
      if (elem.getMusicDefn().equals("chord")) { 
	Chord chord = (Chord) elem;
	//System.out.println("Got new chord, time = " + chord.startTime);
	
       
	//check to see if phrase over:
	if (chord.startTime < phraseChordProgress) {  //This is a new phrase
	  if (PhraseForm.currentPhrase.phraseInfo.getLocation().equals(
    						       "sentence start")) {
	    chord.harmonyInfo.setLocation("sentence start");
	  }
	  else {chord.harmonyInfo.setLocation("phrase start");}
	  //Reset the progress
	  chordProgress = 1;
	  //Can't have stuff hanging over phrases, so reset currentChord
	  currentChord = null;
	}
	
	if ((chordNum - chordProgress) == 2) {
	  chord.harmonyInfo.setLocation("pre-cadence");}
	else if ((chordNum - chordProgress) == 1) {
	  chord.harmonyInfo.setLocation("cadence");}
	else if ((chordNum - chordProgress) == 0) {
	  chord.harmonyInfo.setLocation("resolution");}

	/* saveLabels = false;
	if (chord.partMelEvent.size() > 0) {
	  saveLabels = true;
	  Event evnt = null;
	  Enumeration partMel = chord.partMelEvent.getElements();
	  while (partMel.hasMoreElements()) {
	    evnt = (Event)partMel.nextElement(); 
	    if (evnt.pitch.isNull()) {
	      saveLabels = false;
	    }
	  }
	} */
	
	if (!saveLabels) { 
	  chord.partMelEvent.removeElems();
	  accHoldQueue.push(chord);
	}

        PitchInstInput.accompInQueue.push(chord);
	phraseChordProgress = chord.startTime + chord.duration;
	chordProgress++;
      }
      else {//System.out.println("Got new Event, time = " + elem.startTime);
      PitchInstInput.accompInQueue.push(elem);}
    }

     //Next handle the percussion:
    while (percussionInQueue.size() > 0) {
      elem = (MusicElement)percussionInQueue.pop();
      PitchInstInput.percussionInQueue.push(elem);
    }
    
    if (!saveLabels) {
      connectElements();
    }

    PitchInstInput.activate();
  }

  private static void connectElements() {
    //early out if don't have any stuff
    //  if ((accHoldQueue.size()==0)||(melHoldQueue.size()==0)) {
    //   return;
    //  }

    if (currentChord == null) {currentChord = (Chord)accHoldQueue.pop();}
    if (currentMel == null) {currentMel = (Event)melHoldQueue.pop();}

    boolean continueFlag = true;
    while (continueFlag) {
      continueFlag = compareTimes();
    }

    continueFlag = true;
    if (getNextMelody() == false) {             //Out of melody...
      while (continueFlag) {
	continueFlag = getNextChord();          //...so cycle through accomp
      }
    } else {                              //Otherwise, out of accomp...
      while (continueFlag) {
	continueFlag = getNextMelody();   //...so cycle through melody
      }
    }

  }

  //Compare the times of the currentMelody and currentChord and link
  //  as appropriate.
  private static boolean compareTimes() {
    long melT = currentMel.startTime;
    long melD = currentMel.duration;
    long chordT = currentChord.startTime;
    long chordD = currentChord.duration;
    //  System.out.println(melT+", "+ melD+", "+ chordT+", "+ chordD);

    if (melT == chordT) {
      linkCurrents("equal");
      if ((melT+melD) <= (chordT + chordD)) {   //Mel event ends first or at
	                                        // the same time.
	return getNextMelody();
      } else { return getNextChord();}
    }
    else if (melT >= (chordT + chordD)) {     //Mel event starts after chord
      return getNextChord();
    }
    else if ((melT > chordT)&&(melT < (chordT+chordD))) {  //Mel starts during
                                                          // chord.
      linkCurrents("chord");
      if ((melT+melD) <= (chordT + chordD)) {   //Mel event ends first or at
	                                        // the same time.
	return getNextMelody();
      } else { return getNextChord();}
    }
    else if ((melT+melD) <= chordT) {     //Mel event ends before chord
      return getNextMelody();
    }
    else if ((melT < chordT)&&((melT+melD) > chordT)) {  //chord starts during
                                                          // melody.
      linkCurrents("melody");
      if ((melT+melD) <= (chordT + chordD)) {   //Mel event ends first or at
	                                        // the same time.
	return getNextMelody();
      } else { return getNextChord();}
    }
    return false;
  }

  //Get a new melody event, if possible
  private static boolean getNextMelody() {
    if (melHoldQueue.size() > 0) {
 //  System.out.println("supConChord size: " + currentMel.supConChord.size());
      currentMel = (Event)melHoldQueue.pop();
      
      return true;
    }
//System.out.println("supConChord size (l): " + currentMel.supConChord.size());
    return false;
  }

  //Get a new chord, if possible
  private static boolean getNextChord() {
    if (accHoldQueue.size() > 0) {
//System.out.println("partMelEvent size: " + currentChord.partMelEvent.size());
      lastChord = currentChord;
      currentChord = (Chord)accHoldQueue.pop();
      return true;
    }
    // System.out.println("partMelEvent size(l): " + 
    // currentChord.partMelEvent.size());
    return false;
  }

  //Link a melody event and a chord, if necessary
  private static String linkCurrents(String first) {
    long melT = currentMel.startTime;
    long melD = currentMel.duration;
    long chordT = currentChord.startTime;
    long chordD = currentChord.duration;
    long overlap = 0;
    StringBuffer end = new StringBuffer(" ");

    if (first.equals("equal")) {            //Both start at same time
      if ((melT+melD) <= (chordT+chordD)) {   //if mel ends first
	overlap = melD;
	end.insert(0,"melody");
      }
      else {overlap = chordD;}
    }
    else if (first.equals("chord")) {            //chord starts first
      if ((melT+melD) <= (chordT+chordD)) {   //if mel ends first
	overlap = melD;
	end.insert(0,"melody");
      }
      else {overlap = (chordT+chordD) - melT;}
    }  
    else if (first.equals("melody")) {            //melody starts first
      if ((melT+melD) <= (chordT+chordD)) {   //if mel ends first
	overlap = (melT+melD) - chordT;
	end.insert(0,"melody");
      }
      else {overlap = chordD;}
    } 
    
    if (overlap >= threshold) {
      currentChord.partMelEvent.addElem(currentMel);
      currentMel.supConChord.addElem(currentChord);
    }
    else if ((!(first.equals("melody")))&&(end.toString().equals("melody"))) {
             // if the melody contained by the chord
      currentChord.partMelEvent.addElem(currentMel);
      currentMel.supConChord.addElem(currentChord);
    }
    else if (!(first.equals("melody"))) {
            // the melody stretches over.  Assign to this chord temporarily.
      currentChord.partMelEvent.addElem(currentMel);
      currentMel.supConChord.addElem(currentChord);
    }
    else {  //The melody starts first and ends first
      if ((currentMel.supConChord.size() > 0)&&
	  (lastOverlap >= overlap)) {
	//Do nothing
      }
      else {
	if (!(lastChord == null)) {
	  lastChord.partMelEvent.removeLast();
	  currentMel.supConChord.removeLast();
	}
	currentChord.partMelEvent.addElem(currentMel);
	currentMel.supConChord.addElem(currentChord);
      }
    }
    
    lastOverlap = overlap;
    if (end.length() < 2) {end.insert(0,"chord");}
    return end.toString();
  }   

  public static void reCreateLinks(Phrase phrase) {

    MusicElement elem;
    Event event = null;
    Chord chord = null;

    //Can't have stuff hanging over phrases, so reset currentMel
    currentMel = null;
     //First handle the melody:
    Enumeration egEnum = phrase.partMelEventGroup.getElements();
    Enumeration eventEnum = null;
    while (egEnum.hasMoreElements()) {
     eventEnum = ((EventGroup)egEnum.nextElement()).partMelEvent.getElements();
     while (eventEnum.hasMoreElements()) {
	event = (Event) eventEnum.nextElement();
	event.supConChord.removeElems();
	melHoldQueue.push(event);
     }
    }
   
      

     //Next handle the accompaniment:
    Enumeration chordEnum = phrase.partChord.getElements();
    chordProgress = 1;
     //Can't have stuff hanging over phrases, so reset currentChord
     currentChord = null;
    while (chordEnum.hasMoreElements()) {
        chord = (Chord)chordEnum.nextElement();
	chord.partMelEvent.removeElems();
	accHoldQueue.push(chord);
    }
    
    connectElements();

    
  }
}


