unit Profile;
{
  Profile component unit for Digital Signal Analyser
  Researched and Developed by Boo Khan Ming

  E-mail: bookm@tm.net.my
  WWW:    http://come.to/khanming
}

interface

function ReadProfileString(FileID,SectionID,FieldID:string):string;
function ReadProfileWord(FileID,SectionID,FieldID:string):word;
function ReadProfileByte(FileID,SectionID,FieldID:string):byte;
function ReadProfileBoolean(FileID,SectionID,FieldID:string):boolean;
function ReadProfileInteger(FileID,SectionID,FieldID:string):integer;
function ReadProfileReal(FileID,SectionID,FieldID:string):real;

function WriteProfileString(FileID,SectionID,FieldID,ValueStr:string):boolean;
function WriteProfileWord(FileID,SectionID,FieldID:string;ValueWord:word):boolean;
function WriteProfileByte(FileID,SectionID,FieldID:string;ValueByte:byte):boolean;
function WriteProfileBoolean(FileID,SectionID,FieldID:string;ValueBool:boolean):boolean;
function WriteProfileInteger(FileID,SectionID,FieldID:string;ValueInt:integer):boolean;
function WriteProfileReal(FileID,SectionID,FieldID:string;ValueReal:real):boolean;

implementation

uses Outface;

const
  SectionPrefix='[';
  SectionSuffix=']';
  FieldSeparator='=';
  CommentIdentifier=';';

type
  SwapBufferType=array [1..250] of string; { 64KB buffer allocation }

var
  FileHandle:text;
  StringVar:string;
  WordVar:word;
  ByteVar:byte;
  BooleanVar:boolean;
  IntegerVar:integer;
  RealVar:real;
  Counter:word;
  FindStatus:boolean;
  ValError:integer;
  SwapBuffer:^SwapBufferType;

function RetrieveSectionID:string;
begin
  {$I-}
  ReadLn(FileHandle,StringVar);
  {$I+}
  if IOResult<>0 then
  begin
    RetrieveSectionID:='';
    Exit;
  end;

  StringVar:=TrimText(StringVar);
  if (StringVar[1]=SectionPrefix) and (StringVar[Length(StringVar)]=SectionSuffix) then
    RetrieveSectionID:=TrimText(Copy(StringVar,2,Length(StringVar)-2))
  else
    RetrieveSectionID:='';
end;

function RetrieveFieldID(var Value:string):string;
begin
  {$I-}
  ReadLn(FileHandle,StringVar);
  {$I+}
  if IOResult<>0 then
  begin
    RetrieveFieldID:='';
    Exit;
  end;

  StringVar:=TrimText(StringVar);

  { Skip comment and section identifier }
  if (not (StringVar[1] in [CommentIdentifier,SectionPrefix]))
  and (Pos(FieldSeparator,StringVar)>0) then
  begin
    RetrieveFieldID:=TrimText(Copy(StringVar,1,Pos(FieldSeparator,StringVar)-1));
    Value:=TrimText(Copy(StringVar,Pos(FieldSeparator,StringVar)+1,Length(StringVar)));
  end
  else
  begin
    RetrieveFieldID:='';
    Value:='';
  end;
end;

function ReadProfileString(FileID,SectionID,FieldID:string):string;
begin
  ReadProfileString:='';

  {$I-}
  Assign(FileHandle,FileID);
  Reset(FileHandle);
  {$I+}

  if IOResult<>0 then
    Exit;

  FindStatus:=False;

  repeat
    FindStatus:=UpperCase(RetrieveSectionID)=UpperCase(SectionID);
  until (FindStatus) or (EOF(FileHandle));

  while (not EOF(FileHandle)) and (FindStatus) do
  begin
    if UpperCase(RetrieveFieldID(StringVar))=UpperCase(FieldID) then
    begin
      ReadProfileString:=StringVar;
      Break;
    end;
  end;

  Close(FileHandle);
end;

function ReadProfileWord(FileID,SectionID,FieldID:string):word;
begin
  StringVar:=ReadProfileString(FileID,SectionID,FieldID);
  Val(StringVar,WordVar,ValError);

  if ValError<>0 then
    WordVar:=0;

  ReadProfileWord:=WordVar;
end;

function ReadProfileByte(FileID,SectionID,FieldID:string):byte;
begin
  StringVar:=ReadProfileString(FileID,SectionID,FieldID);
  Val(StringVar,ByteVar,ValError);

  if ValError<>0 then
    ByteVar:=0;

  ReadProfileByte:=ByteVar;
end;

function ReadProfileBoolean(FileID,SectionID,FieldID:string):boolean;
begin
  ReadProfileBoolean:=False;

  StringVar:=ReadProfileString(FileID,SectionID,FieldID);
  if UpperCase(StringVar)='YES' then
    ReadProfileBoolean:=True
  else if UpperCase(StringVar)='NO' then
    ReadProfileBoolean:=False;
end;

function ReadProfileInteger(FileID,SectionID,FieldID:string):integer;
begin
  StringVar:=ReadProfileString(FileID,SectionID,FieldID);
  Val(StringVar,IntegerVar,ValError);

  if ValError<>0 then
    IntegerVar:=0;

  ReadProfileInteger:=IntegerVar;
end;

function ReadProfileReal(FileID,SectionID,FieldID:string):real;
begin
  StringVar:=ReadProfileString(FileID,SectionID,FieldID);
  Val(StringVar,RealVar,ValError);

  if ValError<>0 then
    RealVar:=0;

  ReadProfileReal:=RealVar;
end;

function WriteProfileString(FileID,SectionID,FieldID,ValueStr:string):boolean;
var
  ProfileSize:word;
  SectionLine:word;
  FieldLine:word;

begin
  WriteProfileString:=False;

  {$I-}
  Assign(FileHandle,FileID);
  Reset(FileHandle);
  {$I+}

  { Create new file, new section, and new field }
  if IOResult<>0 then
  begin
    {$I-}
    Rewrite(FileHandle);
    {$I+}
    if IOResult<>0 then
      Exit;

    WriteLn(FileHandle,SectionPrefix+SectionID+SectionSuffix);
    Write(FileHandle,FieldID);
    Write(FileHandle,FieldSeparator);
    WriteLn(FileHandle,ValueStr);

    WriteProfileString:=True;

    Close(FileHandle);
    Exit;
  end;

  ProfileSize:=0;

  { Find total line }
  while not EOF(FileHandle) do
  begin
    ReadLn(FileHandle,StringVar);
    Inc(ProfileSize);
  end;

  Reset(FileHandle);

  Counter:=1;
  FindStatus:=False;

  { Search for matching section }
  while not EOF(FileHandle) do
  begin
    if UpperCase(RetrieveSectionID)=UpperCase(SectionID) then
    begin
      FindStatus:=True;
      Break;
    end;

    Inc(Counter);
  end;

  { Create new section, and new field }
  if not FindStatus then
  begin
    Close(FileHandle);
    Append(FileHandle);

    { Separate with blank line }
    Write(FileHandle,Chr(13)+Chr(10));

    WriteLn(FileHandle,SectionPrefix+SectionID+SectionSuffix);
    Write(FileHandle,FieldID);
    Write(FileHandle,FieldSeparator);
    WriteLn(FileHandle,ValueStr);

    WriteProfileString:=True;

    Close(FileHandle);
    Exit;
  end;

  SectionLine:=Counter+1;

  Inc(Counter);
  FindStatus:=False;

  { Search for matching field }
  while not EOF(FileHandle) do
  begin
    if UpperCase(RetrieveFieldID(StringVar))=UpperCase(FieldID) then
    begin
      FindStatus:=True;
      Break;
    end;

    Inc(Counter);
  end;

  FieldLine:=Counter;

  if MaxAvail<SizeOf(SwapBufferType) then
    Exit;

  GetMem(SwapBuffer,SizeOf(SwapBufferType));

  Reset(FileHandle);

  for Counter:=1 to ProfileSize do
  begin
    ReadLn(FileHandle,StringVar);
    SwapBuffer^[Counter]:=StringVar;
  end;

  {$I-}
  Close(FileHandle);
  Erase(FileHandle);

  Rewrite(FileHandle);
  {$I+}
  if IOResult<>0 then
  begin
    FreeMem(SwapBuffer,SizeOf(SwapBufferType));
    Exit;
  end;

  { Create new field }
  if not FindStatus then
  begin
    for Counter:=1 to ProfileSize do
    begin
      if Counter=SectionLine then
      begin
        Write(FileHandle,FieldID);
        Write(FileHandle,FieldSeparator);
        WriteLn(FileHandle,ValueStr);
      end;

      StringVar:=SwapBuffer^[Counter];
      WriteLn(FileHandle,StringVar);
    end;
  end
  else
  { Update value of existing field }
  begin
    for Counter:=1 to ProfileSize do
    begin
      if Counter=FieldLine then
      begin
        Write(FileHandle,FieldID);
        Write(FileHandle,FieldSeparator);
        WriteLn(FileHandle,ValueStr);
      end
      else
      begin
        StringVar:=SwapBuffer^[Counter];
        WriteLn(FileHandle,StringVar);
      end;
    end;
  end;

  Close(FileHandle);
  FreeMem(SwapBuffer,SizeOf(SwapBufferType));

  WriteProfileString:=True;
end;

function WriteProfileWord(FileID,SectionID,FieldID:string;ValueWord:word):boolean;
begin
  Str(ValueWord,StringVar);
  WriteProfileWord:=WriteProfileString(FileID,SectionID,FieldID,StringVar);
end;

function WriteProfileByte(FileID,SectionID,FieldID:string;ValueByte:byte):boolean;
begin
  Str(ValueByte,StringVar);
  WriteProfileByte:=WriteProfileString(FileID,SectionID,FieldID,StringVar);
end;

function WriteProfileBoolean(FileID,SectionID,FieldID:string;ValueBool:boolean):boolean;
begin
  if ValueBool then
    StringVar:='yes'
  else
    StringVar:='no';

  WriteProfileBoolean:=WriteProfileString(FileID,SectionID,FieldID,StringVar);
end;

function WriteProfileInteger(FileID,SectionID,FieldID:string;ValueInt:integer):boolean;
begin
  Str(ValueInt,StringVar);
  WriteProfileInteger:=WriteProfileString(FileID,SectionID,FieldID,StringVar);
end;

function WriteProfileReal(FileID,SectionID,FieldID:string;ValueReal:real):boolean;
begin
  Str(ValueReal,StringVar);
  WriteProfileReal:=WriteProfileString(FileID,SectionID,FieldID,StringVar);
end;

end.