unit Outface;
{
  Outface component unit for Digital Signal Analyser
  Researched and Developed by Boo Khan Ming

  E-mail: bookm@tm.net.my
  WWW:    http://come.to/khanming
}

interface

type
  MenuItemArrayType=array [1..25] of string[80];

const
  ScreenSurfaceFrontColor=7;
  ScreenSurfaceBackColor=0;
  ScreenBorderFrontColor=15+16;
  ScreenBorderBackColor=1;
  FrameSurfaceFrontColor=0;
  FrameSurfaceBackColor=7;
  FrameBorderFrontColor=15+16;
  FrameBorderBackColor=1;
  FrameSelectionBarFrontColor=7;
  FrameSelectionBarBackColor=1;
  SingleFrameBorderFrontColor=FrameBorderFrontColor;
  SingleFrameBorderBackColor=FrameBorderBackColor;
  SingleFrameSurfaceFrontColor=FrameSurfaceFrontColor;
  SingleFrameSurfaceBackColor=FrameSurfaceBackColor;
  SingleFrameSelectionBarFrontColor=7;
  SingleFrameSelectionBarBackColor=1;
  ListSurfaceFrontColor=0+16;
  ListSurfaceBackColor=7;
  ListPanelFrontColor=15+16;
  ListPanelBackColor=0;
  ListSelectionBarFrontColor=15;
  ListSelectionBarBackColor=1;
  MenuSurfaceFrontColor=FrameSurfaceFrontColor;
  MenuSurfaceBackColor=FrameSurfaceBackColor;
  MenuSelectionBarFrontColor=0+16;
  MenuSelectionBarBackColor=3;
  InputSurfaceFrontColor=0+16;
  InputSurfaceBackColor=7;
  ErrorFrameBorderFrontColor=15+16;
  ErrorFrameBorderBackColor=4;
  CautionFrameBorderFrontColor=0+16;
  CautionFrameBorderBackColor=6;
  SideBarFrontColor=15+16;
  SideBarBackColor=ScreenBorderBackColor;
  SideBarActiveSelectionFrontColor=15+16;
  SideBarActiveSelectionBackColor=ScreenBorderBackColor;
  SideBarInactiveSelectionFrontColor=0+16;
  SideBarInactiveSelectionBackColor=ScreenBorderBackColor;
  SideBarActiveElementFrontColor=11+16;
  SideBarActiveElementBackColor=SideBarBackColor;
  FlatFrameSurfaceFrontColor=SideBarInactiveSelectionFrontColor;
  FlatFrameSurfaceBackColor=SideBarInactiveSelectionBackColor;
  FlatFrameSelectionBarFrontColor=SideBarActiveSelectionFrontColor;
  FlatFrameSelectionBarBackColor=SideBarActiveSelectionBackColor;
  WaveDisplayPanelFrontColor=10;
  WaveDisplayPanelBackColor=0;

procedure EnableSurfaceIntensity;
procedure EnableSurfaceBlink;
procedure HorizontalRetrace;
procedure VerticalRetrace;
procedure CursorOn;
procedure CursorOff;

function DecimalValue(HexVal:string):word;
function HexadecimalValue(DecVal:word):string;
function NormalizeNumber(Value:real):string;
function ShortenNumber(Value:longint):string;
function FormatNumber(Number:string):string;

function UpperCase(TextString:string):string;
function LowerCase(TextString:string):string;
function TrimText(OldStr:string):string;

procedure ClearKeyboardBuffer;
function GetKeyboardShiftStatus(var ShiftKey,CtrlKey,AltKey:boolean):byte;

function OpenPicture(PictureFileName:string;Index:longint):byte;
procedure ClosePicture;
procedure ChangePalette(Color,RealValue,G,B:byte);
procedure RetrievePalette(Color:byte;var RealValue,G,B:byte);

procedure PushScreen;
procedure PopScreen;
procedure PrepareScreen;
procedure ShutScreen;

procedure ChangeTextOffset(X,Y:byte);
procedure ChangeTextRegion(StartX,StartY,EndX,EndY:byte);
function RequestTextOffsetX:byte;
function RequestTextOffsetY:byte;
procedure ClearTextContent;
procedure ChangeTextAppearance(OldColor,NewColor:byte);
procedure InsertText(Content:string);
procedure InsertNewText(Content:string);
procedure InsertMultipleText(Content,Count:byte);
procedure InsertNewMultipleText(Content,Count:byte);

procedure SketchFrame(StartX,StartY,EndX,EndY,BorderFrontColor,BorderBackColor,
                      SurfaceFrontColor,SurfaceBackColor:byte;Title:string);
procedure SketchSingleFrame(StartX,StartY,EndX,EndY:byte;Title:string);
procedure SketchFlatFrame(StartX,StartY,EndX,EndY:byte);
procedure SketchWindow(StartX,StartY,EndX,EndY,FrontColor,BackColor:byte);
procedure OpenScreen(Title:string);
procedure OpenFrame(Width,Height:byte;Title:string);
procedure OpenErrorFrame(Width,Height:byte;Title:string);
procedure OpenCautionFrame(Width,Height:byte;Title:string);
function OpenMenu(MenuItemArray:MenuItemArrayType;MenuItemCount:byte;
                  SurfaceFrontColor,SurfaceBackColor,
                  SelectionBarFrontColor,SelectionBarBackColor:byte):byte;
procedure SketchInput(StartX,StartY,Width:byte;InputName:string);
function AcquireStringInput(StartX,StartY,Width:byte;InputName:string;
                            var InputValue:string):boolean;
function AcquireWordInput(StartX,StartY,Width:byte;InputName:string;
                          var InputValue:word):boolean;
function AcquireRealInput(StartX,StartY,Width:byte;InputName:string;
                          var InputValue:real):boolean;

implementation

uses CRT;

const
  TextContentLimitX=80;
  TextContentLimitY=25;

type
  TextContentType=array [0..(TextContentLimitX*TextContentLimitX)-1] of word;

var
  TextContent:TextContentType absolute $b800:0;
  TextContentOffsetX,TextContentOffsetY:byte;
  TextContentLowerBoundX,TextContentLowerBoundY:byte;
  TextContentUpperBoundX,TextContentUpperBoundY:byte;
  KeyboardFlag:byte absolute $0000:$0417;
  EnhancedKeyboardSupport:boolean;
  Counter:word;
  TextString:string;
  Screen:pointer;

procedure EnableSurfaceIntensity; assembler;
asm
  mov  ax, 1003h
  sub  bl, bl
  int  10h
end;

procedure EnableSurfaceBlink; assembler;
asm
  mov  ax, 1003h
  mov  bl, 1
  int  10h
end;

procedure HorizontalRetrace; assembler;
asm
  mov  dx, 03dah
@1:
  in   al, dx
  test al, 1
  jnz  @1
@2:
  in   al, dx
  test al, 1
  jz   @2
end;

procedure VerticalRetrace; assembler;
asm
  mov  dx, 03dah
@1:
  in   al, dx
  test al, 8
  jnz  @1
@2:
  in   al, dx
  test al, 8
  jz   @2
end;

procedure CursorOn; assembler;
asm
  mov  ah, 1
  mov  cx, 0607h
  int  10h
end;

procedure CursorOff; assembler;
asm
  mov  ah, 1
  mov  cx, 2020h
  int  10h
end;

function DecimalValue(HexVal:string):word;
var
  Result:word;
  Counter:word;

begin
  Result:=0;

  for Counter:=1 to Length(HexVal) do
  begin
    HexVal[Counter]:=UpCase(HexVal[Counter]);

    if Ord(HexVal[Counter])>=Ord('A') then
      Result:=(Result shl 4)+Ord(HexVal[Counter])-Ord('A')+10
    else
      Result:=(Result shl 4)+Ord(HexVal[Counter])-Ord('0');
  end;

  DecimalValue:=Result;
end;

function HexadecimalValue(DecVal:word):string;
const
  HexadecimalTable:array [0..$f] of char='0123456789ABCDEF';

begin
  HexadecimalValue:=(HexadecimalTable[Hi(DecVal) shr 4])
                     +(HexadecimalTable[Hi(DecVal) and $f])
                     +(HexadecimalTable[Lo(DecVal) shr 4])
                     +(HexadecimalTable[Lo(DecVal) and $f]);
end;

function NormalizeNumber(Value:real):string;
const
  HighNotation:array [1..3] of char=('K','M','G');
  LowNotation:array [1..3] of char=('m','','n');

var
  NewValue:real;
  Counter:word;

begin
  for Counter:=3 downto 1 do
    if Value>=(Exp(Ln(1000)*Counter)) then
    begin
      NewValue:=Value/(Exp(Ln(1000)*Counter));
      Str(NewValue:0:2,TextString);
      NormalizeNumber:=TextString+HighNotation[Counter];
      Exit;
    end;

    if Value<1 then
      for Counter:=1 to 3 do
        if Value>=1/(Exp(Ln(1000)*Counter)) then
        begin
          NewValue:=Value*(Exp(Ln(1000)*Counter));
          Str(NewValue:0:2,TextString);
          NormalizeNumber:=TextString+LowNotation[Counter];
          Exit;
        end;

  Str(Value:0:2,TextString);
  NormalizeNumber:=TextString;
end;

function ShortenNumber(Value:longint):string;
var
  NewValue:real;

begin
  if Value>=1024*1024*1024 then
  begin
    NewValue:=Value/(1024*1024*1024);
    Str(NewValue:0:2,TextString);
    TextString:=TextString+'GB';
  end
  else
  if Value>=1024*1024 then
  begin
    NewValue:=Value/(1024*1024);
    Str(NewValue:0:2,TextString);
    TextString:=TextString+'MB';
  end
  else
  if Value>=1024 then
  begin
    NewValue:=Value/1024;
    Str(NewValue:0:2,TextString);
    TextString:=TextString+'KB';
  end
  else
  begin
    Str(Value:0,TextString);
    TextString:=TextString+' bytes'
  end;

  ShortenNumber:=TextString;
end;

function FormatNumber(Number:string):string;
var
  Weight,Loop:byte;

begin
  Weight:=1;
  Loop:=0;
  TextString:=Number;

  repeat
    if (Weight mod 3)=0 then
    begin
      Insert(',',TextString,Length(TextString)-Weight-Loop+1);
      Inc(Loop);
    end;

    Inc(Weight);
  until Weight>=(Length(TextString)-Loop);

  FormatNumber:=TextString;
end;

function UpperCase(TextString:string):string;
var
  Counter:word;

begin
  for Counter:=1 to Length(TextString) do
    if (TextString[Counter]>='a') and (TextString[Counter]<='z') then
      Dec(TextString[Counter],32);

  UpperCase:=TextString;
end;

function LowerCase(TextString:string):string;
var
  Counter:word;

begin
  for Counter:=1 to Length(TextString) do
    if (TextString[Counter]>='A') and (TextString[Counter]<='Z') then
      Inc(TextString[Counter],32);

  LowerCase:=TextString;
end;

function TrimText(OldStr:string):string;
var
  NewStr:string;
  Counter:word;

begin
  NewStr:='';

  for Counter:=1 to Length(OldStr) do
    if not (Ord(OldStr[Counter]) in [32,9,13,10]) then
      Break;

  NewStr:=NewStr+Copy(OldStr,Counter,Length(OldStr));

  for Counter:=Length(NewStr) downto 1 do
    if not (Ord(NewStr[Counter]) in [32,9,13,10]) then
      Break;

  NewStr:=Copy(NewStr,1,Counter);

  TrimText:=NewStr;
end;

procedure ClearKeyboardBuffer;
begin
  while Keypressed do
    ReadKey;
end;

function GetKeyboardShiftStatus(var ShiftKey,CtrlKey,AltKey:boolean):byte;
var
  ScanCode:byte;

begin
  if EnhancedKeyboardSupport then
  asm
    mov  ah, 10h
    int  16h
    mov  ScanCode, ah
  end
  else
  asm
    xor  ah, ah
    int  16h
    mov  ScanCode, ah
  end;

  ShiftKey:=(KeyboardFlag and 3)<>0;
  CtrlKey:=(KeyboardFlag and 4)<>0;
  AltKey:=(KeyboardFlag and 8)<>0;

  GetKeyboardShiftStatus:=ScanCode;
end;

function OpenPicture(PictureFileName:string;Index:longint):byte;
var
  PictureFile:file;
  Picture:pointer;
  Palette:pointer;

begin
  OpenPicture:=255;

  if MaxAvail<64000 then
  begin
    OpenPicture:=1;
    Exit;
  end;

  GetMem(Picture,64000);
  GetMem(Palette,768);

  {$I-}
  Assign(PictureFile,PictureFileName);
  Reset(PictureFile,1);
  {$I+}
  if IOResult<>0 then
  begin
    OpenPicture:=2;
    Exit;
  end;

  Seek(PictureFile,Index*64768);
  BlockRead(PictureFile,Palette^,768);
  BlockRead(PictureFile,Picture^,64000);

  asm
    mov  ax, 0013h
    int  10h
    mov  ax, 1012h
    xor  bx, bx
    mov  cx, 256
    les  dx, Palette
    int  10h
  end;

  Move(Picture^,Ptr($a000,$0000)^,64000);

  FreeMem(Picture,64000);
  FreeMem(Palette,768);

  Close(PictureFile);
  OpenPicture:=0;
end;

procedure ClosePicture;
begin
  TextMode(LastMode);
end;

procedure ChangePalette(Color,RealValue,G,B:byte);
begin
  Port[$03c8]:=Color;
  Port[$03c9]:=RealValue;
  Port[$03c9]:=G;
  Port[$03c9]:=B;
end;

procedure RetrievePalette(Color:byte;var RealValue,G,B:byte);
begin
  Port[$03c7]:=Color;
  RealValue:=Port[$03c9];
  G:=Port[$03c9];
  B:=Port[$03c9];
end;

procedure PushScreen;
begin
  GetMem(Screen,4000);
  Move(Ptr($b800,0)^,Screen^,4000);
end;

procedure PopScreen;
begin
  Move(Screen^,Ptr($b800,0)^,4000);
  FreeMem(Screen,4000);
end;

procedure PrepareScreen;
begin
  TextMode(co80);
  CursorOff;
  EnableSurfaceIntensity;
end;

procedure ShutScreen;
begin
  TextMode(LastMode);
  CursorOn;
end;

procedure AutoAdjustTextOffset;
begin
  if TextContentOffsetX<TextContentLowerBoundX then
    TextContentOffsetX:=TextContentLowerBoundX;
  if TextContentOffsetY<TextContentLowerBoundY then
    TextContentOffsetY:=TextContentLowerBoundY;

  if TextContentOffsetX>TextContentUpperBoundX then
  begin
    TextContentOffsetX:=TextContentLowerBoundX;
    Inc(TextContentOffsetY);
  end;

  if TextContentOffsetY>TextContentUpperBoundY then
    TextContentOffsetY:=TextContentUpperBoundY;
end;

procedure ChangeTextOffset(X,Y:byte);
begin
  TextContentOffsetX:=TextContentLowerBoundX+X-1;
  TextContentOffsetY:=TextContentLowerBoundY+Y-1;

  {ChangeTextOffset(TextContentOffsetX,TextContentOffsetY);}
  GotoXY(TextContentOffsetX,TextContentOffsetY);
end;

procedure ChangeTextRegion(StartX,StartY,EndX,EndY:byte);
begin
  TextContentLowerBoundX:=StartX;
  TextContentLowerBoundY:=StartY;
  TextContentUpperBoundX:=EndX;
  TextContentUpperBoundY:=EndY;

  TextContentOffsetX:=TextContentLowerBoundX;
  TextContentOffsetY:=TextContentLowerBoundY;
end;

function RequestTextOffsetX:byte;
begin
  RequestTextOffsetX:=TextContentOffsetX;
end;

function RequestTextOffsetY:byte;
begin
  RequestTextOffsetY:=TextContentOffsetY;
end;

procedure ClearTextContent;
var
  CounterX,CounterY:byte;

begin
  for CounterX:=TextContentLowerBoundX to TextContentUpperBoundX do
    for CounterY:=TextContentLowerBoundY to TextContentUpperBoundY do
      TextContent[(CounterY-1)*TextContentLimitX+CounterX-1]:=255+TextAttr shl 8;
end;

procedure ChangeTextAppearance(OldColor,NewColor:byte);
var
  Counter:word;

begin
  Counter:=1;

  repeat
    if (Mem[$b800:Counter] and $f0)=(OldColor shl 4) then
      Mem[$b800:Counter]:=(Mem[$b800:Counter] and $0f) or (NewColor shl 4);

    Inc(Counter,2);
  until Counter>4000;
end;

procedure InsertText(Content:string);
var
  Counter:word;

begin
  for Counter:=1 to Length(Content) do
  begin
    TextContent[(TextContentOffsetY-1)*TextContentLimitX+TextContentOffsetX-1]:=Ord(Content[Counter])+TextAttr shl 8;

    Inc(TextContentOffsetX);
    AutoAdjustTextOffset;
  end;
end;

procedure InsertNewText(Content:string);
var
  Counter:word;

begin
  for Counter:=1 to Length(Content) do
  begin
    TextContent[(TextContentOffsetY-1)*TextContentLimitX+TextContentOffsetX-1]:=Ord(Content[Counter])+TextAttr shl 8;

    Inc(TextContentOffsetX);
    AutoAdjustTextOffset;
  end;

  TextContentOffsetX:=TextContentLowerBoundX;
  Inc(TextContentOffsetY);
  AutoAdjustTextOffset;
end;

procedure InsertMultipleText(Content,Count:byte);
var
  Counter:word;

begin
  for Counter:=1 to Count do
  begin
    TextContent[(TextContentOffsetY-1)*TextContentLimitX+TextContentOffsetX-1]:=Ord(Content)+TextAttr shl 8;

    Inc(TextContentOffsetX);
    AutoAdjustTextOffset;
  end;
end;

procedure InsertNewMultipleText(Content,Count:byte);
var
  Counter:word;

begin
  for Counter:=0 to Count do
  begin
    TextContent[(TextContentOffsetY-1)*TextContentLimitX+TextContentOffsetX-1]:=Ord(Content)+TextAttr shl 8;

    Inc(TextContentOffsetX);
    AutoAdjustTextOffset;
  end;

  TextContentOffsetX:=TextContentLowerBoundX;
  Inc(TextContentOffsetY);
  AutoAdjustTextOffset;
end;

procedure SketchFrame(StartX,StartY,EndX,EndY,BorderFrontColor,BorderBackColor,
                      SurfaceFrontColor,SurfaceBackColor:byte;Title:string);
begin
  ChangeTextRegion(StartX,StartY,EndX,EndY);
  TextColor(BorderFrontColor);
  TextBackground(BorderBackColor);
  ClearTextContent;

  ChangeTextRegion(StartX+1,StartY+1,EndX-1,StartY+1);
  InsertText(Title);

  ChangeTextRegion(StartX+1,StartY+2,EndX-1,EndY-1);
  TextColor(SurfaceFrontColor);
  TextBackground(SurfaceBackColor);
  ClearTextContent;

  ChangeTextRegion(StartX+2,StartY+3,EndX-2,EndY-2);
end;

procedure SketchSingleFrame(StartX,StartY,EndX,EndY:byte;Title:string);
begin
  ChangeTextRegion(StartX,StartY,EndX,EndY);
  TextColor(SingleFrameBorderFrontColor);
  TextBackground(SingleFrameBorderBackColor);
  ClearTextContent;

  ChangeTextRegion(StartX+1,StartY,EndX-1,StartY);
  InsertText(Title);

  ChangeTextRegion(StartX+1,StartY+1,EndX-1,EndY-1);
  TextColor(SingleFrameSurfaceFrontColor);
  TextBackground(SingleFrameSurfaceBackColor);
  ClearTextContent;

  ChangeTextRegion(StartX+2,StartY+2,EndX-2,EndY-2);
end;

procedure SketchFlatFrame(StartX,StartY,EndX,EndY:byte);
begin
  ChangeTextRegion(StartX,StartY,EndX,EndY);
  TextColor(FlatFrameSurfaceFrontColor);
  TextBackground(FlatFrameSurfaceBackColor);
  ClearTextContent;

  ChangeTextRegion(StartX+1,StartY+1,EndX-1,EndY-1);
end;

procedure SketchWindow(StartX,StartY,EndX,EndY,FrontColor,BackColor:byte);
begin
  ChangeTextRegion(StartX,StartY,EndX,EndY);
  TextColor(FrontColor);
  TextBackground(BackColor);
  ClearTextContent;

  ChangeTextRegion(StartX,StartY,EndX,EndY);
end;

procedure OpenScreen(Title:string);
begin
  SketchFrame(1,1,80,25,ScreenBorderFrontColor,ScreenBorderBackColor,
              ScreenSurfaceFrontColor,ScreenSurfaceBackColor,Title);
end;

procedure OpenFrame(Width,Height:byte;Title:string);
var
  StartX,StartY,EndX,EndY:byte;

begin
  StartX:=((80-Width-2) div 2);
  EndX:=StartX+Width+2+1;
  StartY:=((25-Height-3) div 2);
  EndY:=StartY+Height+3+1;

  SketchFrame(StartX,StartY,EndX,EndY,FrameBorderFrontColor,FrameBorderBackColor,
              FrameSurfaceFrontColor,FrameSurfaceBackColor,Title);
end;

procedure OpenErrorFrame(Width,Height:byte;Title:string);
var
  StartX,StartY,EndX,EndY:byte;

begin
  StartX:=((80-Width-2) div 2);
  EndX:=StartX+Width+2+1;
  StartY:=((25-Height-3) div 2);
  EndY:=StartY+Height+3+1;

  SketchFrame(StartX,StartY,EndX,EndY,ErrorFrameBorderFrontColor,ErrorFrameBorderBackColor,
              FrameSurfaceFrontColor,FrameSurfaceBackColor,Title);
end;

procedure OpenCautionFrame(Width,Height:byte;Title:string);
var
  StartX,StartY,EndX,EndY:byte;

begin
  StartX:=((80-Width-2) div 2);
  EndX:=StartX+Width+2+1;
  StartY:=((25-Height-3) div 2);
  EndY:=StartY+Height+3+1;

  SketchFrame(StartX,StartY,EndX,EndY,CautionFrameBorderFrontColor,CautionFrameBorderBackColor,
              FrameSurfaceFrontColor,FrameSurfaceBackColor,Title);
end;

function OpenMenu(MenuItemArray:MenuItemArrayType;MenuItemCount:byte;
                  SurfaceFrontColor,SurfaceBackColor,
                  SelectionBarFrontColor,SelectionBarBackColor:byte):byte;

var
  OldIndex,NewIndex:byte;
  SeparatorWidth:byte;
  Counter:word;
  Key:char;

  procedure UpdateMenu;
  begin
    if OldIndex<>NewIndex then
    begin
      TextColor(SurfaceFrontColor);
      TextBackground(SurfaceBackColor);

      ChangeTextOffset(1,OldIndex);
      if MenuItemArray[OldIndex]='' then
        InsertMultipleText(196,SeparatorWidth)
      else
        InsertText(MenuItemArray[OldIndex]);

      TextColor(SelectionBarFrontColor);
      TextBackground(SelectionBarBackColor);

      ChangeTextOffset(1,NewIndex);
      if MenuItemArray[NewIndex]='' then
        InsertMultipleText(196,SeparatorWidth)
      else
        InsertText(MenuItemArray[NewIndex]);
    end;
  end;

begin
  OpenMenu:=255;
  SeparatorWidth:=0;

  for Counter:=1 to MenuItemCount do
    if Length(MenuItemArray[Counter])>SeparatorWidth then
      SeparatorWidth:=Length(MenuItemArray[Counter]);

  TextColor(SurfaceFrontColor);
  TextBackground(SurfaceBackColor);

  for Counter:=1 to MenuItemCount do
  begin
    ChangeTextOffset(1,Counter);

    if MenuItemArray[Counter]='' then
      InsertMultipleText(196,SeparatorWidth)
    else
      InsertText(MenuItemArray[Counter]);
  end;

  Key:=Chr(0);
  OldIndex:=2;
  NewIndex:=1;

  UpdateMenu;
  ClearKeyboardBuffer;

  repeat
    if Keypressed then
    begin
      OldIndex:=NewIndex;
      Key:=ReadKey;

      case Key of
        Chr(72):
        begin
          Dec(NewIndex);

          while ((MenuItemArray[NewIndex]='') and (NewIndex>1)) do
            Dec(NewIndex);
        end;
        Chr(80):
        begin
          Inc(NewIndex);

          while ((MenuItemArray[NewIndex]='') and (NewIndex<MenuItemCount)) do
            Inc(NewIndex);
        end;
        Chr(71):NewIndex:=1;
        Chr(79):NewIndex:=MenuItemCount;
        Chr(13):
        begin
          OpenMenu:=NewIndex;
          Exit;
        end;
      end;

      if NewIndex<1 then
        NewIndex:=MenuItemCount;
      if NewIndex>MenuItemCount then
        NewIndex:=1;

      UpdateMenu;
    end;
  until Key=Chr(27);
end;

procedure SketchInput(StartX,StartY,Width:byte;InputName:string);
begin
  TextColor(FrameSurfaceFrontColor);
  TextBackground(FrameSurfaceBackColor);

  ChangeTextOffset(StartX,StartY);
  InsertText(InputName);

  TextColor(InputSurfaceFrontColor);
  TextBackground(InputSurfaceBackColor);

  InsertMultipleText(32,Width);
end;

function AcquireStringInput(StartX,StartY,Width:byte;InputName:string;
                            var InputValue:string):boolean;
var
  Key:char;
  TextString:string;
  CursorPosition:byte;

  procedure UpdateInput;
  begin
    TextColor(InputSurfaceFrontColor);
    TextBackground(InputSurfaceBackColor);
    ChangeTextOffset(StartX+Length(InputName),StartY);
    InsertMultipleText(32,Width);
    ChangeTextOffset(StartX+Length(InputName),StartY);
    InsertText(TextString);
    ChangeTextOffset(StartX+Length(InputName)+CursorPosition-1,StartY);
  end;

begin
  SketchInput(StartX,StartY,Width,InputName);
  ChangeTextOffset(StartX+Length(InputName),StartY);

  AcquireStringInput:=False;
  TextString:=InputValue;
  Key:=Chr(0);

  CursorPosition:=Length(InputValue)+1;

  UpdateInput;

  repeat
    if Keypressed then
    begin
      asm
        xor  ax, ax
        int  16h
        mov  Key, al
      end;

      case Key of
        Chr(32)..Chr(127):
        if CursorPosition<=Width then
        begin
          TextString:=TextString+Key;
          Inc(CursorPosition);
          UpdateInput;
        end;
        Chr(8):
        if CursorPosition>1 then
        begin
          TextString:=Copy(TextString,1,Length(TextString)-1);
          Dec(CursorPosition);
          UpdateInput;
        end;
        Chr(13):
        if TextString<>'' then
        begin
          AcquireStringInput:=True;
          InputValue:=TextString;
          Exit;
        end;
      end;
    end;
  until Key=Chr(27);
end;

function AcquireWordInput(StartX,StartY,Width:byte;InputName:string;
                          var InputValue:word):boolean;
var
  Key:char;
  TextString:string;
  WordValue:word;
  ValError:integer;
  CursorPosition:byte;

  procedure UpdateInput;
  begin
    TextColor(InputSurfaceFrontColor);
    TextBackground(InputSurfaceBackColor);
    ChangeTextOffset(StartX+Length(InputName),StartY);
    InsertMultipleText(32,Width);
    ChangeTextOffset(StartX+Length(InputName),StartY);
    InsertText(TextString);
    ChangeTextOffset(StartX+Length(InputName)+CursorPosition-1,StartY);
  end;

begin
  SketchInput(StartX,StartY,Width,InputName);
  ChangeTextOffset(StartX+Length(InputName),StartY);

  AcquireWordInput:=False;
  Key:=Chr(0);

  if InputValue<>0 then
    Str(InputValue,TextString)
  else
    TextString:='';

  CursorPosition:=Length(TextString)+1;

  UpdateInput;

  repeat
    if Keypressed then
    begin
      asm
        xor  ax, ax
        int  16h
        mov  Key, al
      end;

      case Key of
        '0'..'9':
        if CursorPosition<=Width then
        begin
          TextString:=TextString+Key;
          Inc(CursorPosition);
          UpdateInput;
        end;
        Chr(8):
        if CursorPosition>1 then
        begin
          TextString:=Copy(TextString,1,Length(TextString)-1);
          Dec(CursorPosition);
          UpdateInput;
        end;
        Chr(13):
        if TextString<>'' then
        begin
          AcquireWordInput:=True;
          Val(TextString,WordValue,ValError);
          InputValue:=WordValue;
          Exit;
        end;
      end;
    end;
  until Key=Chr(27);
end;

function AcquireRealInput(StartX,StartY,Width:byte;InputName:string;
                          var InputValue:real):boolean;
var
  Key:char;
  TextString:string;
  RealValue:real;
  ValError:integer;
  CursorPosition:byte;

  procedure UpdateInput;
  begin
    TextColor(InputSurfaceFrontColor);
    TextBackground(InputSurfaceBackColor);
    ChangeTextOffset(StartX+Length(InputName),StartY);
    InsertMultipleText(32,Width);
    ChangeTextOffset(StartX+Length(InputName),StartY);
    InsertText(TextString);
    ChangeTextOffset(StartX+Length(InputName)+CursorPosition-1,StartY);
  end;

begin
  SketchInput(StartX,StartY,Width,InputName);
  ChangeTextOffset(StartX+Length(InputName),StartY);

  AcquireRealInput:=False;
  CursorPosition:=1;
  Key:=Chr(0);

  if InputValue<>0 then
    Str(InputValue:0:2,TextString)
  else
    TextString:='';

  CursorPosition:=Length(TextString)+1;

  UpdateInput;

  repeat
    if Keypressed then
    begin
      asm
        xor  ax, ax
        int  16h
        mov  Key, al
      end;

      case Key of
        '0'..'9','.':
        if CursorPosition<=Width then
        begin
          TextString:=TextString+Key;
          Inc(CursorPosition);
          UpdateInput;
        end;
        Chr(8):
        if CursorPosition>1 then
        begin
          TextString:=Copy(TextString,1,Length(TextString)-1);
          Dec(CursorPosition);
          UpdateInput;
        end;
        Chr(13):
        if TextString<>'' then
        begin
          AcquireRealInput:=True;
          Val(TextString,RealValue,ValError);
          InputValue:=RealValue;
          Exit;
        end;
      end;
    end;
  until Key=Chr(27);
end;

begin
  TextContentLowerBoundX:=1;
  TextContentLowerBoundY:=1;
  TextContentUpperBoundX:=TextContentLimitX;
  TextContentUpperBoundY:=TextContentLimitY;

  TextContentOffsetX:=TextContentLowerBoundX;
  TextContentOffsetY:=TextContentLowerBoundY;

  EnhancedKeyboardSupport:=(Byte(Ptr($40,$96)^) and 16)<>0;
end.
