;
; PCD V1.0 - a simple CD player for the DOS MSCDEX interface
;
; Use NASM (Netwide Assembler) 0.97+ to compile it.
;

org 0x100

; X and Y coordinates of player's upper left corner

PLAYER_X	equ	6
PLAYER_Y	equ	3

;colors of help text

HELP_COLOR	equ	0x03
HELP_HILITE	equ	0x0B

;default color of the player

PLAYER_COLOR	equ	0x87

;editor highlight:
EDITOR_HILITE	equ	0x6F

;offset of screen save buffer

SCR_SAVEBUF		equ	0x4000

;maximum length of textual disc information block

DISCINFO_MAXLEN	equ	0x1000

;scrolling speed of the status message, must be power of 2

SCROLL_SPEED	equ	4

; Command codes for IOCTL READ:

HEAD_LOC	equ	1
CHANNEL_INFO	equ	4
DEVICE_STATUS	equ	6
MEDIA_CHANGED	equ	9
DISK_INFO	equ	10
TRACK_INFO	equ	11
QCHANNEL_INFO	equ	12
SUBCHANNEL_INFO	equ	13
UPC_CODE	equ	14
STATUS_INFO	equ	15

; Command codes for IOCTL WRITE:

EJECT_DISK	equ	0
LOCK_DOOR	equ	1
RESET_DRIVE	equ	2
CHANNEL_WRITE	equ	3
CLOSE_TRAY	equ	5

; Device Driver Request Header

struc ReqHdr
.hdr_len	resb	1
.subunit_code	resb	1
.command	resb	1	; see possible commands above
.status		resw	1
.reserved	resb	8
.cb		resb	13
endstruc


%macro PLAYER_SETXY 2
		mov	di,[rows+(PLAYER_Y+%2)*2]
		add	di,(PLAYER_X+%1)*2
%endmacro



section .text

		cld

		mov	ax,0x1500
		int	0x2f			; check MSCDEX
		or	bx,bx			; present?
		jnz	mscdex_ok

		mov	si,msg_NoMSCDEX
		jmp	msg_quit

mscdex_ok:	mov	[drv_count],bx
		mov	bx,drv_letters
		mov	ax,0x150d		; get drive letters to [bx]
		int	0x2f
		mov	al,[drv_letters]
		cbw
		mov	[drive],ax		; we use the first drive

		call	get_status
		test	word [status],0x0010	; audio capability?
		jnz	.audio_ok
		mov	si,msg_NoAudioCap
		jmp	msg_quit

.audio_ok:	test	word [status],0x0200	; RedBook addressing?
		jnz	.redbook_ok
		mov	si,msg_NoRedBook
		jmp	msg_quit

.redbook_ok:	int	0x11			; get equipment list
		and	ax,0x10
		shr	ax,4
		mov	byte [mono],al
		mov	bx,0xb800
		or	al,al
		jz	.color
		mov	bx,0xb000
.color:		mov	[scrseg],bx

		mov	ax,0x1003
		xor	bl,bl
		int	0x10			; enable background intensity

		push	ds
		push	es

		mov	ah,0x0F
		int	0x10			; get video information
		xchg	al,ah
		cbw
		shl	ax,1
		mov	[cs:scrwidth],ax
		mov	[cs:scrpage],bh

		mov	ah,0x03
		int	0x10			; get cursor position
		mov	[cs:cursor_x],dl
		mov	[cs:cursor_y],dh

		mov	ah,0x02
		mov	dx,0xFF00
		int	0x10			; move cursor out of screen

		xor	ax,ax
		mov	es,ax
		mov	cx,[es:0x044c]		; size of screen regen buffer
		mov	[cs:scrsize],cx
		xor	ch,ch
		mov	cl,[es:0x0484]		; number of rows - 1
		inc	cx
		mov	[cs:scrheight],cx

		pop	es
		pop	ds

		; modify [scrseg] to point to active screen page

		xor	ah,ah
		mov	al,[scrpage]
		mul	word [scrsize]
		shr	ax,4
		add	[scrseg],ax

		; generate lookup table of row start addresses

		xor	ax,ax
		mov	di,rows
.fillrows:	stosw
		add	ax,[scrwidth]
		loop	.fillrows

		; the timer routine is responsible for
		; refreshing on-screen CD player data

		call	setint1c

		call	save_screen

redraw:		call	restore_screen
		call	draw_player

main_loop:
		call	get_cd_status	; get status/qchannel information
		mov	ah,1
		int	0x16
		jz	main_loop

		xor	ax,ax
		int	16h

; ------------- g : go to track -------------------------------------
		cmp	ah,0x22		; 'g' and 'G' are both accepted
		jne	.goto

		push	ds
		push	es

		mov	bh,[scrpage]

		mov	ah,0x02
		mov	dx,(PLAYER_Y+2)*0x100 + (PLAYER_X+5)
		int	0x10		; set cursor position

		xor	ax,ax
		int	0x16		; get 1st digit of track number

		mov	ah,0x0A
		mov	cx,1
		int	0x10		; print it back to screen
		sub	al,0x30
		mov	bl,10
		mul	bl
		mov	si,ax		; si = digit*10

		mov	ah,0x02
		mov	dx,(PLAYER_Y+2)*0x100 + (PLAYER_X+6)
		int	0x10

		xor	ax,ax
		int	0x16		; get 2nd digit

		mov	ah,0x0A
		mov	cx,1
		int	0x10

		push	ax
		mov	ah,0x02
		mov	dx,0xFF00
		int	0x10		; set cursor back to invisible area
		pop	ax

		sub	al,0x30
		cbw
		add	ax,si		; al = track to go

		pop	es
		pop	ds

		cmp	al,[last_track]
		ja	.q
		cmp	al,[first_track]
		jb	.q
		push	ax
		mov	bp,ioctl_stop
		call	request
		pop	ax
		call	play_track
.q:		jmp	main_loop
.goto:
; ------------- insert : eject/close tray ---------------------------
		cmp	ax,0x5200
		jne	.eject
		test	byte [status],0x01	;tray open?
		mov	dx,CLOSE_TRAY
		jnz	.ej			; yes, then close it
		mov	bp,ioctl_stop		; else stop playing
		call	request
		mov	dx,EJECT_DISK		; and eject disk
.ej:		mov	bp,ioctl_output
		call	request
		jmp	main_loop
.eject:
; ------------- 0123456789 : skip to N*10 % of track ----------------
		cmp	al,'0'
		jb	.digit
		cmp	al,'9'
		ja	.digit

		mov	bl,al
		sub	bl,0x30
		xor	bh,bh
		mov	ax,[track_len]		; track length in frames
		mov	dx,[track_len+2]
		mov	cx,10
		cmp	dx,cx
		jae	.zq
		div	cx
		mul	bx			; DX:AX = N*(length/10), N=0..9
		add	ax,[track_start]
		adc	dx,[track_start+2]	; + start of track
		call	Frames_to_RedBook
		mov	[frame_MSF],ax
		mov	[frame_MSF+2],dx
		mov	bp,ioctl_stop
		call	request
		mov	bp,ioctl_play		; playing starts at [frame_MSF]
		call	request
.zq:		jmp	main_loop
.digit:
; ------------- backspace : back to beginning of track --------------
		cmp	ax,0x0e08
		jnz	.bs
		mov	bp,ioctl_stop
		call	request
		mov	al,[track]
		call	seek_track
		jmp	main_loop
.bs:
; ------------- delete : reset CDROM drive --------------------------
		cmp	ax,0x5300
		jnz	.delete
		mov	byte [no_disc],1
		mov	bp,ioctl_output
		mov	dx,RESET_DRIVE
		call	request
		jmp	main_loop
.delete:
; ------------- enter : play ----------------------------------------
		cmp	ax,0x1c0d
		jnz	.enter
		mov	al,[track]
		call	play_track
		jmp	main_loop
.enter:
; ------------- space : pause ---------------------------------------
		cmp	al,0x20
		jnz	.space
		jmp	pause_resume
.space:
; ------------- right : next track ----------------------------------
		cmp	ax,0x4d00
		jz	.n
.right:
; ------------- up : next track -------------------------------------
		cmp	ax,0x4800
		jnz	.up
.n:		jmp	next_track
.up:
; ------------- down : previous track -------------------------------
		cmp	ax,0x5000
		jz	.p
.down:
; ------------- left : previous track -------------------------------
		cmp	ax,0x4b00
		jnz	.left
.p:		jmp	prev_track
.left:
; ------------- pgup : next track -----------------------------------
		cmp	ax,0x4900
		jz	.n
.pgup:
; ------------- pgdn : previous track -------------------------------
		cmp	ax,0x5100
		jz	.p
.pgdn:
; ------------- home : go to beginning of track ---------------------
		cmp	ax,0x4700
		jnz	.home
		jmp	track_home
.home:
; ------------- ctrl+home : first track -----------------------------
		cmp	ax,0x7700
		jnz	.ctrlhome
		jmp	disc_home
.ctrlhome:
; ------------- F1 : help -------------------------------------------
		cmp	ax,0x3b00
		jnz	.help
		call	show_help
		jmp	redraw
.help:
; ------------- F4 : enter edit mode --------------------------------
		cmp	ax,0x3e00
		jnz	.edit
		jmp	edit
.edit:
; ------------- Shift+F4 : edit CDDB dir ----------------------------
		cmp	ax,0x5700
		jnz	.cddb
		call	edit_cddb_dir
		mov	byte [no_disc],1
		jmp	redraw
.cddb:
; ------------- ESC : quit player -----------------------------------
		cmp	ax,0x011b
		jz	quit_player
; ------------- F10 : quit player -----------------------------------
		cmp	ax,0x4400
		jz	quit_player
		jmp	main_loop


quit_player:	call	restoreint1c
		call	restore_screen
		mov	ah,0x02
		mov	bh,[scrpage]
		mov	dh,[cursor_y]
		mov	dl,[cursor_x]
		int	0x10
		mov	ax,0x4c00
		int	0x21



get_cd_status:	call	get_status

		mov	ax,[req_hdr+ReqHdr.status]
		and	ax,0x0200
		shr	ax,9
.playing:	mov	[playing],al

		test	word [status],0x0801
		jz	.disc_ok
		cmp	byte [no_disc],1
		je	.2
		mov	byte [no_disc],1
		mov	byte [cddb_info],0
		jmp	.0
.disc_ok:
		cmp	byte [no_disc],1		; disc inserted?
		jne	.1
		mov	byte [no_disc],0
		call	get_track_info
		call	get_qchannel
.0:		
		cmp	byte [drive_not_ready],1
		je	.1
		call	draw_player

.1:		cmp	byte [playing],0
		jz	.2
		call	get_qchannel

.2:		retn


pause_resume:	mov	bp,ioctl_resume
		cmp	byte [paused],1
		je	.1
		mov	bp,ioctl_stop
.1:		call	request
		jmp	main_loop

next_track:	mov	al,[track]
		inc	al
		cmp	al,[last_track]
		jbe	disc_home.1
disc_home:	mov	al,[first_track]
.1:		push	ax
		mov	bp,ioctl_stop
		call	request
		pop	ax
		call	play_track
		jmp	main_loop

track_home:	mov	al,[track]
		jmp	disc_home.1

prev_track:	mov	al,[track]
		dec	al
		cmp	al,[first_track]
		jae	.1
		mov	al,[last_track]
.1:		jmp	disc_home.1

;
; call device driver request routine with:
;
; BP = address of ioctl routine
; DX = command code (see defines at the top)
;

request:	mov	bx,req_hdr

		; initialize request header

		mov	di,bx
		mov	cx,ReqHdr_size
		xor	al,al
		rep	stosb
		mov	byte [bx+ReqHdr.hdr_len],13
		jmp	bp


ioctl_input:	add	byte [bx+ReqHdr.hdr_len],13
		mov	byte [bx+ReqHdr.command],3
		mov	word [bx+ReqHdr.cb+1],req_buf
		mov	word [bx+ReqHdr.cb+3],es
		mov	di,dx
		mov	al,[input_sizes+di]
		cbw
		mov	word [bx+ReqHdr.cb+5],ax
		mov	byte [req_buf],dl

		; setting input parameters

		cmp	dx,HEAD_LOC
		jnz	.a1
		mov	byte [req_buf+1],1
		jmp	.doit
.a1:		cmp	dx,TRACK_INFO
		jnz	.a2
		mov	al,[track]
		mov	byte [req_buf+1],al
		jmp	.doit
.a2:

.doit:		push	dx
		call	exec_request
		pop	dx

		; transferring output parameters

		cmp	dx,HEAD_LOC
		jnz	.b1

		mov	ax,[req_buf+2]
		mov	[frame_MSF],ax
		mov	ax,[req_buf+4]
		mov	[frame_MSF+2],ax
		retn

.b1:		cmp	dx,CHANNEL_INFO
		jnz	.b2

		mov	al,[req_buf+2]
		mov	[volume],al
		mov	al,[req_buf+4]
		mov	[volume+1],al
		mov	al,[req_buf+6]
		mov	[volume+2],al
		mov	al,[req_buf+8]
		mov	[volume+3],al
		retn

.b2:		cmp	dx,DEVICE_STATUS
		jnz	.b3

		mov	ax,[req_buf+1]
		mov	[status],ax
		retn

.b3:		cmp	dx,MEDIA_CHANGED
		jnz	.b4

		mov	al,[req_buf+1]
		mov	[media_changed],al
		retn

.b4:		cmp	dx,DISK_INFO
		jnz	.b5

		mov	al,[req_buf+1]
		mov	[first_track],al
		mov	al,[req_buf+2]
		mov	[last_track],al
		mov	ax,[req_buf+3]
		mov	[lead_out],ax
		mov	ax,[req_buf+5]
		mov	[lead_out+2],ax
		retn

.b5:		cmp	dx,TRACK_INFO
		jnz	.b6

		mov	byte [drive_not_ready],0
		test	byte [req_hdr+ReqHdr.status],0xFF
		jz	.b5a
		mov	byte [drive_not_ready],1
		mov	byte [no_disc],1
		retn
.b5a:		mov	ax,[req_buf+2]
		mov	[frame_MSF],ax
		mov	ax,[req_buf+4]
		mov	[frame_MSF+2],ax
		mov	al,[req_buf+6]
		mov	[track_ctrl_info],al
		retn

.b6:		cmp	dx,QCHANNEL_INFO
		jnz	.b7

		mov	al,[req_buf+2]
		call	BCD_to_Byte
		mov	[track],al

		xor	al,al
		mov	[tracktime+3],al
		mov	al,[req_buf+4]
		mov	[tracktime+2],al
		mov	al,[req_buf+5]
		mov	[tracktime+1],al
		mov	al,[req_buf+6]
		mov	[tracktime],al
		mov	byte [disktime+3],0
		mov	al,[req_buf+8]
		mov	[disktime+2],al
		mov	al,[req_buf+9]
		mov	[disktime+1],al
		mov	al,[req_buf+10]
		mov	[disktime],al
		retn

.b7:		cmp	dx,UPC_CODE
		jnz	.b8

		lea	si,[req_buf+2]
		mov	di,upc
		mov	cx,8
		rep	movsb
		retn

.b8:		cmp	dx,STATUS_INFO
		jnz	.b9

		mov	al,[req_buf+1]
		and	al,1
		mov	[paused],al
.b9:		retn


ioctl_output:	add	byte [bx+ReqHdr.hdr_len],13
		mov	byte [bx+ReqHdr.command],12
		mov	word [bx+ReqHdr.cb+1],req_buf
		mov	word [bx+ReqHdr.cb+3],es
		mov	di,dx
		mov	al,[output_sizes+di]
		cbw
		mov	word [bx+ReqHdr.cb+5],ax
		mov	byte [req_buf],dl
		jmp	exec_request

ioctl_seek:	add	byte [bx+ReqHdr.hdr_len],11
		mov	byte [bx+ReqHdr.command],131
		mov	byte [bx+ReqHdr.cb],1
		mov	ax,[frame_MSF]
		mov	word [bx+ReqHdr.cb+7],ax
		mov	ax,[frame_MSF+2]
		mov	word [bx+ReqHdr.cb+9],ax
		jmp	exec_request

ioctl_play:	add	byte [bx+ReqHdr.hdr_len],9
		mov	byte [bx+ReqHdr.command],132
		mov	byte [bx+ReqHdr.cb],1
		mov	ax,[frame_MSF]
		mov	dx,[frame_MSF+2]
		mov	word [bx+ReqHdr.cb+1],ax	; play from here
		mov	word [bx+ReqHdr.cb+3],dx
		call	RedBook_to_HSG
		mov	[HSG],ax
		mov	[HSG+2],dx
		mov	ax,[lead_out]
		mov	dx,[lead_out+2]
		call	RedBook_to_HSG
		sub	ax,[HSG]
		sbb	dx,[HSG+2]
		mov	word [bx+ReqHdr.cb+5],ax	; play dx:ax sectors
		mov	word [bx+ReqHdr.cb+7],dx
		jmp	exec_request

ioctl_stop:	mov	byte [bx+ReqHdr.command],133
		jmp	exec_request

ioctl_resume:	mov	byte [bx+ReqHdr.command],136
		jmp	exec_request

exec_request:	mov	cx,[drive]
		mov	ax,0x1510
		int	0x2f

		; wait for completion

.wait:		test	word [req_hdr+ReqHdr.status],0x81FF
		jz	.wait

.ok:		retn


; Convert a BCD value to byte
;
; In:  AL = packed BCD value (4 bits/digit)
; Out: AL = converted to byte

BCD_to_Byte:	mov	bh,al
		and	bh,0x0F
		shr	al,4
		mov	bl,10
		mul	bl
		add	al,bh
		retn

;
; The following routines convert between three
; different representations of frame offsets on CD:
;
; High Sierra (HSG)
; RedBook (MSF = Minutes/Seconds/Frames)
; Frames (M*60*75+S*75+F)
; Elapsed seconds (Frames/75)
;
; HSG, RedBook and Frame values are 2 bytes (DX:AX)
; Elapsed seconds are transferred in the AX register
;

RedBook_to_HSG:	call	RedBook_to_Frames
		sub	ax,150
		sbb	dx,0
		retn

Frames_to_SEC:
		call	Frames_to_RedBook

RedBook_to_SEC:	push	bx
		xchg	dx,ax
		xor	ah,ah
		mov	bl,60
		mul	bl
		add	al,dh
		adc	ah,0
		pop	bx
		retn

SEC_to_RedBook:	
		push	bx
		mov	bl,60
		cmp	ah,bl
		jae	.zero
		div	bl
.zero:		mov	dl,al
		xor	al,al
		xor	dh,dh
		pop	bx
		retn

RedBook_to_Frames:
		push	bx
		mov	bl,75
		push	ax
		mov	al,ah
		mul	bl
		push	ax
		mov	ax,dx
		mov	bx,75*60
		mul	bx		; DX:AX=minutes*75*60
		pop	bx
		add	ax,bx
		adc	dx,0		; + seconds*75
		pop	bx
		xor	bh,bh
		add	ax,bx
		adc	dx,0		; + frames
		pop	bx
		retn

Frames_to_RedBook:
		push	bx
		mov	bx,75
		cmp	dx,bx
		jae	.z1
		div	bx		; divide by 75
.z1:		push	dx
		mov	bl,60
		cmp	ah,bl
		jae	.z2
		div	bl		; AH = # of seconds
.z2:		mov	dl,al		; DL = # of minutes
		pop	bx
		mov	al,bl		; AL = # of extra frames
		xor	dh,dh
		pop	bx
		retn

;
; String routines
;

;
; scan_string	scans memory from DS:SI up to DS:SI+CX
;		looking for the DX long string pointed to by ES:DI
;		If string found, sets ZF to 1, otherwise ZF=0.
;

scan_string:	mov	al,[es:di]
.0:		cmp	[si],al
		jz	.1
.n:		inc	si
		loop	.0
		cmp	cx,si
		retn
.1:		xchg	cx,dx
		push	cx
		push	si
		push	di
		rep	cmpsb
		pop	di
		pop	si
		pop	cx
		xchg	cx,dx
		jnz	.n
		retn

;
; copy_string	copies bytes from DS:SI to ES:DI up to the
;		first character equal to AL, 0x0D or 0x0A.
;		Instead of copying this last character, it writes a 0x00.
;		If the last character was equal to AL, CF=0.
;		If it was 0x0D or 0x0A, CF=1.

copy_string:	cmp	[si],al
		jz	.q1
		cmp	byte [si],13
		jz	.q2
		cmp	byte [si],10
		jz	.q2
		movsb
		jmp	copy_string
.q1:		inc	si		; if it was AL, step it over
		xor	al,al
		stosb
		clc
		retn
.q2:		xor	al,al
		stosb
		stc
		retn

;
; skip_spaces	advances SI until it points to a non-whitespace character
;		whitespace characters are 0x20 (space) and 0x09 (tab)
;

skip_spaces0:	inc	si
skip_spaces:	cmp	byte [si],' '
		jz	skip_spaces0
		cmp	byte [si],9
		jz	skip_spaces0
		retn


; Convert the CDDB disc ID to an ASCIIZ filename

set_cddb_fn:
		mov	di,[cddb_filename]
		mov	cx,1
		mov	al,[cddb_id+3]
		call	write_byte_h
		mov	al,[cddb_id+2]
		call	write_byte_h
		mov	al,[cddb_id+1]
		call	write_byte_h
		mov	al,[cddb_id]
		call	write_byte_h
		xor	al,al
		stosb
		retn


; Retrieve CDDB info from file

get_cddb_info:	mov	byte [cddb_info],0

		; initialize disc data area

		mov	di,[disc_data]
		mov	cx,DISCINFO_MAXLEN
		xor	al,al
		rep	stosb

		; generate filename

		call	set_cddb_fn

		; open file

		mov	dx,cddb_path
		mov	ax,0x3d00
		int	0x21
		jnb	.j
		jmp	.noinfo

		; read maximum possible amount of bytes

.j:		mov	bx,ax
		mov	cx,DISCINFO_MAXLEN
		mov	dx,[disc_data]
		mov	ah,0x3F
		int	0x21
		mov	[cddb_filesize],ax	; DX is irrelevant (=0)

		; close file

		mov	ah,0x3e
		int	0x21

		; find '# xmcd' identifier

		mov	si,[disc_data]
		mov	cx,[cddb_filesize]
		mov	di,cddb_textid
		mov	dx,6
		call	scan_string
		jnz	.noinfo

		; find 'DTITLE=' (artist name and disc title)

		mov	di,cddb_DTITLE
		mov	dx,7
		call	scan_string
		jnz	.noinfo
		add	si,dx

		; copy artist name (till '/')

		mov	al,'/'
		mov	di,[disc_data]
		call	copy_string

		; remove trailing spaces from artist name

.no_spaces:	cmp	byte [di-2],' '
		jnz	.nospace
		dec	di
		jmp	.no_spaces

		; skip spaces before disc title

.nospace:	mov	byte [di-1],0
		call	skip_spaces

		; copy disc title (till end of line)

		mov	al,0x0A			; Line Feed
		call	copy_string

		; copy track titles

		mov	bp,di
		mov	bl,[first_track]

.scan_titles:	mov	di,cddb_TTITLE
		mov	dx,6
		call	scan_string
		jnz	.noinfo
		add	si,dx
		mov	di,cddb_equalsign
		mov	dx,1
		call	scan_string
		jnz	.noinfo
		add	si,dx

		mov	al,0x0A
		mov	di,bp
		call	copy_string
		mov	bp,di

		inc	bl
		cmp	bl,[last_track]
		jbe	.scan_titles

		mov	byte [cddb_info],1

.noinfo:	retn






; buffer for 2-digit decimal conversion

wbd_buf		db	'  '

;
; write byte decimal
;
; In:  AL = byte to write
; Out: digit(s) at wbd_buf
;      CX = number of digits in result
;

wbd:		mov	cx,2
		mov	bl,10
		xor	ah,ah
		div	bl
		add	ax,0x3030
		cmp	al,0x30
		jnz	.1
		xchg	al,ah
		dec	cx
.1		mov	word [wbd_buf],ax
		retn
		




; Save in-memory CDDB disc information to a file

save_cddb_info:	call	set_cddb_fn

		; create file

		mov	ah,0x3c
		xor	cx,cx
		mov	dx,cddb_path
		int	0x21
		jnb	.j
		jmp	.error
.j:
		mov	bx,ax

		; write '# xmcd' CDDB identifier

		mov	ah,0x40
		mov	dx,cddb_textid
		mov	cx,8
		int	0x21

		; write 'DISCID=xxxxxxxx'

		mov	dx,cddb_DISCID
		mov	cx,7
		mov	ah,0x40
		int	0x21
		mov	dx,[cddb_filename]
		mov	cx,8
		mov	ah,0x40
		int	0x21
		mov	dx,cddb_LF
		mov	cx,1
		mov	ah,0x40
		int	0x21

		; write 'DTITLE=artist / disc title'

		mov	dx,cddb_DTITLE
		mov	cx,7
		mov	ah,0x40
		int	0x21

		push	bx
		mov	bx,-1
		call	find_title	; find artist name
		mov	dx,si
		call	strlen
		mov	cx,bx
		pop	bx
		mov	ah,0x40
		int	0x21

		mov	dx,cddb_per	; ' / '
		mov	cx,3
		mov	ah,0x40
		int	0x21

		push	bx
		xor	bx,bx
		call	find_title	; find disc title
		mov	dx,si
		call	strlen
		mov	cx,bx
		pop	bx
		mov	ah,0x40
		int	0x21

		mov	dx,cddb_LF
		mov	cx,1
		mov	ah,0x40
		int	0x21

		; write 'TTITLEn=title of track n'

		mov	si,1
.w:		mov	dx,cddb_TTITLE
		mov	cx,6
		mov	ah,0x40
		int	0x21

		push	bx
		mov	ax,si
		dec	ax		; track numbers start from 0 in CDDB
		call	wbd
		mov	dx,wbd_buf
		mov	ah,0x40
		pop	bx
		int	0x21

		mov	dx,cddb_equalsign
		mov	cx,1
		mov	ah,0x40
		int	0x21

		push	si
		push	bx
		mov	bx,si
		call	find_title
		mov	dx,si
		call	strlen
		mov	cx,bx
		pop	bx
		mov	ah,0x40
		int	0x21
		mov	dx,cddb_LF
		mov	cx,1
		mov	ah,0x40
		int	0x21
		pop	si

		inc	si
		push	bx
		mov	bx,si
		cmp	bl,[last_track]
		pop	bx
		jbe	.w

		; write 'EXTD='

		mov	dx,cddb_EXTD
		mov	cx,4
		mov	ah,0x40
		int	0x21
		mov	dx,cddb_equalsign
		mov	cx,2
		mov	ah,0x40
		int	0x21

		; write 'EXTTn='

		mov	si,1
.w1:		mov	dx,cddb_EXTT
		mov	cx,4
		mov	ah,0x40
		int	0x21
		push	bx
		mov	ax,si
		dec	ax
		call	wbd
		mov	dx,wbd_buf
		mov	ah,0x40
		pop	bx
		int	0x21
		mov	dx,cddb_equalsign
		mov	cx,2
		mov	ah,0x40
		int	0x21
		inc	si
		push	bx
		mov	bx,si
		cmp	bl,[last_track]
		pop	bx
		jbe	.w1

		; write 'PLAYORDER='

		mov	dx,cddb_PLAYORDER
		mov	cx,11
		mov	ah,0x40
		int	0x21

		; close file

		mov	ah,0x3e
		int	0x21

.error:		mov	byte [cddb_info],1
		retn




; get various state information about CD-ROM

get_status:	mov	bp,ioctl_input
		mov	dx,DEVICE_STATUS
		call	request
		mov	bp,ioctl_input
		mov	dx,STATUS_INFO
		jmp	request

get_qchannel:
		mov	bp,ioctl_input
		mov	dx,QCHANNEL_INFO
		jmp	request




; get starting frame numbers of tracks and calculate CDDB identifier

get_track_info:
		mov	bp,ioctl_input
		mov	dx,DISK_INFO
		call	request

		mov	al,[first_track]
		mov	di,track_info
		xor	bh,bh
		mov	bl,al
		shl	bx,2
		add	di,bx
		push	di

		mov	word [cddb_sum],0		; this is a kind of accumulator

.1:		mov	[track],al
		mov	bp,ioctl_input
		mov	dx,TRACK_INFO
		push	di
		call	request
		pop	di
		mov	ax,[frame_MSF]
		mov	dx,[frame_MSF+2]
		mov	[di],ax
		mov	[di+2],dx
		call	RedBook_to_SEC

		mov	bx,10
.sum:		or	ax,ax
		jz	.2
		xor	dx,dx
		div	bx
		add	[cddb_sum],dx
		jmp	.sum
.2:
		add	di,4
		mov	al,[track]
		inc	al
		cmp	al,[last_track]
		jbe	.1

		mov	ax,[lead_out]
		mov	dx,[lead_out+2]
		mov	[di],ax
		mov	[di+2],dx
		add	di,4

		mov	[disc_data],di	; disc_data comes after track_info
		call	RedBook_to_SEC
		mov	[SEC_buf],ax
		pop	di		; DI -> frame offset of first track
		mov	ax,[di]
		mov	dx,[di+2]
		call	RedBook_to_SEC
		mov	[first_start],ax
		sub	[SEC_buf],ax
		mov	ax,[SEC_buf]
		mov	[totaltime],ax	; total disc time in seconds

		mov	dh,byte [cddb_sum]	; DH = sum calculated above
		mov	dl,ah			; DL = totaltime HI
		mov	ah,al			; AH = totaltime LO
		mov	al,[last_track]		; AL = number of tracks
		mov	word [cddb_id],ax
		mov	word [cddb_id+2],dx

		; get disc information from file

		call	get_cddb_info

		; seek to first track

		mov	al,[first_track]

; call seek_track with track number in AL

seek_track:	mov	bp,ioctl_seek
		call	seek_it
		xor	ax,ax
		mov	[tracktime],ax
		mov	[tracktime+2],ax
		mov	ax,[frame_MSF]
		mov	dx,[frame_MSF+2]
		mov	[disktime],ax
		mov	[disktime+2],dx
		retn

; call play_track with track number in AL

play_track:	mov	bp,ioctl_play
seek_it:	mov	[track],al
		xor	bx,bx
		mov	bl,al
		shl	bx,2
		mov	ax,[track_info+bx]
		mov	dx,[track_info+bx+2]
		mov	[frame_MSF],ax
		mov	[frame_MSF+2],dx
		jmp	request



; set up INT 1C vector (timer interrupt)

setint1c:	push	es
		mov	ax,0x351c
		int	0x21
		mov	[old1c],bx
		mov	[old1c+2],es
		pop	es
		mov	ax,0x251c
		mov	dx,int1c
		int	0x21
		retn


; this is the new handler routine

int1c:		pushf
		push	ax
		push	bx
		push	cx
		push	dx
		push	si
		push	di
		push	ds
		push	es

		mov	ax,[cs:ticks]
		inc	ax
		test	ax,SCROLL_SPEED-1
		jnz	.q
		add	word [cs:click],2
		cmp	word [cs:click],12*2
		jne	.q
		mov	word [cs:click],0
.q:		mov	[cs:ticks],ax
		cmp	byte [cs:help_mode],0	; help mode?
		jnz	.1			; if so, don't update player

		push	cs
		pop	ds
		mov	es,[scrseg]

		call	update_player
		call	update_status

.1:		pop	es
		pop	ds
		pop	di
		pop	si
		pop	dx
		pop	cx
		pop	bx
		pop	ax
		popf
		jmp	far [cs:old1c]


; restore default INT 1c handler

restoreint1c:	mov	ax,0x251c
		mov	dx,[old1c]
		push	ds
		mov	ds,[old1c+2]
		int	0x21
		pop	ds
		retn




; draw player to screen

draw_player:
		push	es
		mov	es,[scrseg]

		; draw upper part

		PLAYER_SETXY 0,0
		mov	si,line1
		call	draw_lines
		mov	si,color_info
		call	set_colors

.1:		cmp	byte [cddb_info],0
		jz	.clr_info

		; if there is disc info, draw lower half, too

		PLAYER_SETXY 0,4
		mov	si,info1
		call	draw_lines
		mov	si,color_info2
		call	set_colors

.q:		pop	es
		retn

		; no disc info -> copy original screen contents to lower half

.clr_info:	PLAYER_SETXY 33,4
		lea	si,[SCR_SAVEBUF+di]
		mov	cx,[scrsize]
		sub	cx,di
		rep	movsb
		jmp	.q

;
; draw_lines copies text from DS:SI to ES:DI (screen) with PLAYER_COLOR
; text data may contain the following control characters:
;
;	0x00		end of text
;	0x01-0x1F	repeat next character N times (N=value of control byte)
;	0xFF		end of line, advance to next
;

draw_lines:	mov	ah,PLAYER_COLOR
		xor	ch,ch
.nextline:	mov	bx,di
.nextchar:	lodsb
		or	al,al
		jz	.q
		test	al,0xE0
		jnz	.rep
		and	al,0x1F
		mov	cl,al
		lodsb
		rep	stosw
		jmp	.nextchar
.rep:		cmp	al,0xFF
		jnz	.LF
		mov	di,bx
		add	di,[scrwidth]
		jmp	.nextline
.LF:		stosw
		jmp	.nextchar
.q:		retn




; update player information on screen

update_player:	

		; write number of current track

		PLAYER_SETXY 5,2
		mov	al,[track]
		call	write_byte_dec
		add	di,2

		; write number of tracks

		mov	al,[last_track]
		call	write_byte_dec

		; write length of track in seconds

		xor	bh,bh
		mov	bl,[track]
		shl	bx,2
		mov	ax,[track_info+bx]
		mov	dx,[track_info+bx+2]
		call	RedBook_to_Frames
		mov	[track_start],ax
		mov	[track_start+2],dx
		mov	ax,[track_info+bx+4]
		mov	dx,[track_info+bx+6]
		call	RedBook_to_Frames
		sub	ax,[track_start]
		sbb	dx,[track_start+2]
		mov	[track_len],ax
		mov	[track_len+2],dx
		call	Frames_to_SEC
		push	ax
		PLAYER_SETXY 2,3
		call	write_time

		; write position in track (percentage)

		mov	ax,[tracktime]
		mov	dx,[tracktime+2]
		call	RedBook_to_SEC
		mov	cx,100
		mul	cx
		pop	cx
		or	cx,cx
		jz	.zero
		cmp	dx,cx
		jae	.zero
		div	cx
.zero:
		PLAYER_SETXY 8,3
		push	ax
		call	write_percent
		pop	ax
		cmp	byte [cddb_info],0
		jz	.p

		; if we have disc info, print percentage
		; to lower right corner, too

		PLAYER_SETXY 57,9
		call	write_percent

		; print time elapsed

.p:		PLAYER_SETXY 26,1
		mov	ax,[disktime]
		mov	dx,[disktime+2]
		call	RedBook_to_SEC
		sub	ax,[first_start]
		mov	[SEC_buf],ax
		call	write_time

		; print total time

		PLAYER_SETXY 26,3
		mov	ax,[totaltime]
		call	write_time

		; print time remaining

		PLAYER_SETXY 26,2
		mov	ax,[totaltime]
		sub	ax,[SEC_buf]
		call	write_time

		cmp	byte [cddb_info],0
		jz	.noinfo
		cmp	byte [edit_mode],0
		jnz	.noinfo

		; if we have disc info and we are not in edit mode,
		; print disc and track information, too

		mov	al,[track]
		call	write_info

.noinfo:	retn



; update status display (playing, paused, stopped, etc.)

update_status:

		PLAYER_SETXY 1,1
		mov	cx,12
		mov	bx,11*2
		sub	bx,[click]

		mov	al,byte [no_disc]
		add	al,byte [drive_not_ready]
		or	al,al
		jz	.nodisc
		mov	si,lb_NoDisc
		jmp	.0
.nodisc:
		cmp	byte [paused],1
		jne	.paused
		mov	si,lb_Paused
		jmp	.0
.paused:
		cmp	byte [playing],1
		jne	.playing

		mov	si,lb_Playing
		jmp	.0
.playing:
		mov	si,lb_Stopped

.0:		lodsb
		cmp	al,''
		jne	.00
		dec	si
		test	bx,2
		jz	.00
		mov	al,'-'
.00:		mov	[es:di+bx],al
		add	bx,2
		cmp	bx,12*2
		jne	.1
		xor	bx,bx
.1:		loop	.0

		PLAYER_SETXY 2,8

		cmp	byte [cddb_info],1
		jne	.q
		cmp	byte [playing],1
		jne	.q

		mov	bx,[ticks]
		and	bx,7
		mov	al,[playchars+bx]
		stosb
		add	bx,2
		and	bx,7
		mov	al,[playchars+bx]
		add	di,5
		stosb

.q:		retn


; write CDDB disc and track information
;
; In:  AL = track number (the editor uses this routine, too)

write_info:	push	ax

		; write artist name

		PLAYER_SETXY 2,5
		mov	bx,-1
		call	find_title
		mov	cx,60
		call	write_string

		; write number of tracks

		PLAYER_SETXY 3,6
		mov	al,[last_track]
		call	write_byte_dec
		add	di,4

		; write disc title

		xor	bx,bx
		call	find_title
		mov	cx,55
		call	write_string

		; write number of current track

		PLAYER_SETXY 3,8
		pop	ax
		push	ax
		call	write_byte_dec
		add	di,4

		; write track name

		pop	bx
		call	find_title
		mov	cx,55
		jmp	write_string

;
; find_title returns pointer to specified disc info in SI
;
; In:  BX = info to return (-1: artist, 0: disc title, 1-n: track name)
;

find_title:	push	bx
		mov	si,[disc_data]
.scanzero:	dec	bl
		cmp	bl,-2
		jz	.q
.scan0:		lodsb
		or	al,al
		jnz	.scan0
		jz	.scanzero
.q:		pop	bx
		retn



; save_screen copies the contents of the screen to a memory buffer

save_screen:	xor	si,si
		mov	di,SCR_SAVEBUF
		mov	cx,[scrsize]
		push	ds
		mov	ds,[scrseg]
		rep	movsb
		pop	ds
		retn

; restore_screen copies it back to screen

restore_screen:	mov	si,SCR_SAVEBUF
		xor	di,di
		mov	cx,[scrsize]
		push	es
		mov	es,[scrseg]
		rep	movsb
		pop	es
		retn


;
; set_colors sets screen colors based on a coloring specification table
;
; In:  SI = address of color specs
;
; Format of a color spec record:
;
;	X,Y,repeat count,color (all bytes)
;
; If X=-1, coloring ends
;

set_colors:	push	es
		mov	es,[scrseg]
.0:		lodsb
		cmp	al,-1
		jz	.q
		xor	bh,bh
		mov	bl,al
		add	bl,PLAYER_X
		shl	bx,1
		lodsb
		add	al,PLAYER_Y
		cbw
		mov	cx,[scrwidth]
		mul	cx
		add	bx,ax
		lodsb
		mov	cl,al
		xor	ch,ch
		lodsb
.1:		mov	[es:bx+1],al
		add	bx,2
		loop	.1
		jmp	.0
.q:		pop	es
		retn





; write CX dots to ES:[DI]

write_dots:	mov	al,''
.0:		stosb
		inc	di
		loop	.0
		retn

; write AL as percentage

write_percent:	mov	bl,[no_disc]
		add	bl,[drive_not_ready]
		or	bl,bl
		mov	cx,3
		jnz	write_dots
		mov	bl,' '
		cmp	al,100
		jb	.1
		mov	bl,'1'
		xor	al,al
.1:		mov	[es:di],bl
		add	di,2

; write AL in decimal

write_byte_dec:	mov	bl,[no_disc]
		add	bl,[drive_not_ready]
		or	bl,bl
		mov	cx,2
		jnz	write_dots
		mov	bl,10
		xor	ah,ah
		div	bl
		add	ax,0x3030
		mov	[es:di],al
		mov	[es:di+2],ah
		add	di,4
		retn

; write AL in hex

write_byte_hex:	mov	cx,2
write_byte_h:	mov	ah,al
		shr	al,4
		call	.w
		mov	al,ah
		and	al,0x0F
.w:		cmp	al,0xa
		jb	.1
		add	al,0x27
.1:		add	al,0x30
		mov	[es:di],al
		add	di,cx
		retn

; write AL in binary

write_byte_bin:	mov	ah,al
		mov	bl,0x80
.1:		test	ah,bl
		mov	al,0x30
		jz	.0
		inc	al
.0:		stosb
		inc	di
		shr	bl,1
		or	bl,bl
		jnz	.1
		retn

; write at most CX bytes from DS:SI to screen
; if string length is less than CX, fills extra room with spaces

write_string:	lodsb
		or	al,al
		jz	.1
		stosb
		inc	di
		loop	write_string
.1:		jcxz	.q
		mov	al,' '
.2:		stosb
		inc	di
		loop	.2
.q:		retn

; write AX (seconds) in 'MM:SS' format

write_time:	mov	cl,60
		cmp	ah,cl
		jae	.zero
		div	cl
.zero:		push	ax
		call	write_byte_dec
		mov	al,0x3a
		stosb
		inc	di
		pop	ax
		mov	al,ah
		jmp	write_byte_dec




; help

show_help:	push	es
		mov	es,[scrseg]
		mov	byte [help_mode],1

		xor	di,di
		mov	ax,HELP_COLOR*0x100+0x20
		mov	cx,[scrsize]
		shr	cx,1
		rep	stosw

		mov	si,help
		mov	bp,[scrwidth]
		shr	bp,1
		sub	bp,10
		mov	cx,[scrheight]
		sub	cx,25
		shr	cx,1
		jcxz	.nextline
.bp:		add	bp,[scrwidth]
		loop	.bp

.nextline:	mov	di,bp
		mov	bx,-2
		mov	ah,HELP_HILITE
.nextchar:	lodsb
		or	al,al
		jz	.q
		cmp	al,10
		jnz	.LF
		add	bp,[scrwidth]
		jmp	.nextline
.LF:		cmp	al,9
		jnz	.TAB
		mov	di,bp
		add	di,4
		mov	ah,HELP_COLOR
		neg	bx
		jmp	.nextchar
.TAB:		cmp	al,0xF0
		jb	.color
		and	al,0x0F
		mov	ah,al
		jmp	.nextchar
.color:		mov	[es:di],ax
		add	di,bx
		jmp	.nextchar

.q:		pop	es
		xor	ax,ax
		int	0x16
		mov	byte [help_mode],0
		jmp	restore_screen


; --- editor ---


; strlen returns the length of string at DS:SI in BX

strlen:		xor	bx,bx
.1:		cmp	byte [si+bx],0
		jz	.q
		inc	bx
		jmp	.1
.q:		retn


;
; initialize editor
;
; In:	SI -> ASCIIZ string to edit
;	BX = end of source string area
;	CX = maximum length of result
;	DI = edit buffer offset in screen seg
;

editor_init:	mov	[edit_from],si
		mov	[edit_from_end],bx
		mov	[edit_max_len],cx
		mov	[edit_offset],di

		call	strlen
		mov	[edit_orig_len],bx

		mov	word [edit_pos],0

		; fill edit buffer with spaces

		mov	di,edit_buf
		mov	al,' '
		mov	cx,128
		rep	stosb

		; copy string to buffer

		mov	di,edit_buf
		xor	al,al
		call	copy_string
		mov	byte [di-1],' '		; replace closing 0x00

editor_draw:	push	es
		mov	es,[scrseg]

		mov	di,[edit_offset]
		mov	cx,[edit_max_len]
		mov	si,edit_buf
.1:		lodsb
		stosb
		inc	di
		loop	.1

		; set edit cursor

		mov	di,[edit_pos]
		shl	di,1
		add	di,[edit_offset]
		mov	byte [es:di+1],EDITOR_HILITE

		pop	es
		retn


;
; handle an editor event
;
; In:	AX = keycode (as BIOS INT 0x16/AH=0 returns it)
;

editor_handleevent:
		mov	bx,[edit_pos]

		cmp	ax,0x0e08	; backspace
		je	.bs
		cmp	ax,0x4d00	; right
		je	.right
		cmp	ax,0x4b00	; left
		je	.left
		cmp	ax,0x4700	; home
		je	.home
		cmp	ax,0x4f00	; end
		je	.end
		cmp	ax,0x0e7f	; ctrl-backspace
		je	.ctrlbs_j
		cmp	ax,0x7300	; ctrl-left
		je	.ctrll_j
		cmp	ax,0x7400	; ctrl-right
		je	.ctrlr_j

		or	al,al
		jz	.q

.w:		cmp	bx,[edit_max_len]
		je	.q

		mov	si,edit_buf-1
		mov	cx,[edit_max_len]
		add	si,cx
		cmp	byte [si],' '
		jne	.q
		sub	cx,bx
		dec	cx
		jcxz	.w1

		mov	di,si
		dec	si
		std
		rep	movsb
		cld

.w1:		mov	byte [edit_buf+bx],al
		jmp	.right

.ctrlbs_j:	jmp	.ctrlbs
.ctrll_j:	jmp	.ctrll
.ctrlr_j:	jmp	.ctrlr

.bs:		or	bx,bx
		jz	.q

		mov	si,edit_buf-1
		mov	di,si
		mov	cx,[edit_max_len]
		add	di,cx
		cmp	bx,cx
		je	.bs1

		sub	cx,bx
		add	si,bx
		mov	di,si
		inc	si
		rep	movsb
.bs1:		mov	al,' '
		stosb

.left:		or	bx,bx
		jz	.q
		dec	bx
		jmp	.q

.right:		cmp	bx,[edit_max_len]
		je	.q
		inc	bx
.q:		mov	[edit_pos],bx
		retn

.home:		xor	bx,bx
		jmp	.q

.end:		mov	bx,[edit_max_len]
.end0:		cmp	byte [edit_buf+bx-1],' '
		jne	.q
		dec	bx
		jnz	.end0
		jmp	.q

.ctrlbs:	or	bx,bx
		jz	.q
		cmp	byte [edit_buf+bx-1],' '
		jne	.ctrlbs1
		call	.bs
		jmp	.ctrlbs
.ctrlbs1:	or	bx,bx
		jz	.q
		cmp	byte [edit_buf+bx-1],' '
		je	.q
		call	.bs
		jmp	.ctrlbs1

.ctrll:		or	bx,bx
		jz	.q
		cmp	byte [edit_buf+bx-1],' '
		jne	.ctrll1
		call	.left
		jmp	.ctrll
.ctrll1:	or	bx,bx
		jz	.q
		cmp	byte [edit_buf+bx-1],' '
		je	.q
		call	.left
		jmp	.ctrll1

.ctrlr:		cmp	bx,[edit_max_len]
		je	.q
		cmp	byte [edit_buf+bx-1],' '
		jne	.ctrlr1
		call	.right
		jmp	.ctrlr
.ctrlr1:	cmp	bx,[edit_max_len]
		je	.q
		cmp	byte [edit_buf+bx-1],' '
		je	.q
		call	.right
		jmp	.ctrlr1

; editor destructor

editor_done:	push	bx

		; find end of string in edit_buf

		mov	bx,127
.scan_end:	cmp	byte [edit_buf+bx],' '
		jne	.1
		sub	bx,1
		jnb	.scan_end
.1:		inc	bx

		; append a closing 0x00

		mov	byte [edit_buf+bx],0

		cmp	bx,[edit_orig_len]
		jl	.left
		je	.copy

		; if new string is longer than original then
		; we make place for the extra characters

.right:		std
		mov	cx,[edit_from_end]
		mov	di,cx
		dec	di
		mov	si,di
		mov	ax,bx
		sub	ax,[edit_orig_len]	; AX = difference of lengths
		sub	si,ax
		sub	cx,[edit_from]
		sub	cx,[edit_orig_len]
		sub	cx,ax
		dec	cx			; don't copy zero byte at end
		rep	movsb
		cld
		jmp	.copy

		; if new string is shorter than original then
		; we move the remaining data downwards

.left:		mov	cx,[edit_from_end]
		mov	di,[edit_from]
		mov	si,di
		sub	cx,di
		mov	ax,[edit_orig_len]
		inc	ax			; plus zero byte at end
		add	si,ax
		add	di,bx
		inc	di
		sub	cx,ax			; don't copy zero byte at end
		rep	movsb

		; copy the edit buffer to original location

.copy:		mov	si,edit_buf
		mov	di,[edit_from]
		mov	cx,bx
		rep	movsb
		movsb

		pop	bx
		retn


; enter edit mode

edit:		
		; copy original data to save buffer

		mov	si,[disc_data]
		mov	di,si
		mov	cx,DISCINFO_MAXLEN
		add	di,cx
		rep	movsb

		mov	byte [cddb_info],1
		call	draw_player

		mov	dx,1

		mov	byte [edit_mode],1

		; start with artist name

.bx:		mov	bx,-1

.0:		push	bx
		push	dx
		push	es
		mov	es,[scrseg]
		mov	ax,dx
		call	write_info
		pop	es
		pop	dx
		pop	bx

		or	bx,bx
		js	.00
		jz	.00
		mov	bx,dx

		; set cursor position and CX depending on value in BX
		
.00:		PLAYER_SETXY 7,6
		mov	cx,55
		or	bx,bx
		jns	.1
		PLAYER_SETXY 2,5
		add	cl,5
		jmp	edit1
.1:		jz	edit1
		PLAYER_SETXY 7,8

edit1:		push	bx
		push	dx

		call	find_title
		mov	bx,[disc_data]
		add	bx,DISCINFO_MAXLEN
		call	editor_init

		pop	dx
		pop	bx

		; main loop

.input:		push	bx
		push	dx
		call	get_cd_status
		pop	dx
		pop	bx

		; if disc ejected then cancel edit mode

		cmp	byte [no_disc],1
		je	.escape

		mov	ah,1
		int	0x16
		jz	.input
		xor	ax,ax
		int	0x16

		; set colors for info area

		push	ax
		push	bx
		push	dx
		mov	si,color_info2
		call	set_colors
		pop	dx
		pop	bx
		pop	ax

		cmp	ax,0x011b	; ESC
		je	.escape
		cmp	ax,0x3c00	; F2
		je	.save
		cmp	ax,0x1c0d	; Enter
		je	.down
		cmp	ax,0x5000	; Down
		je	.down
		cmp	ax,0x4800	; Up
		je	.up
		cmp	ax,0x4e2b	; Gray +
		je	.grayplus
		cmp	ax,0x4a2d	; Gray -
		jne	.else
		jmp	.grayminus

		; something else, pass to the editor
.else:
		push	bx
		push	dx
		call	editor_handleevent
		call	editor_draw
		pop	dx
		pop	bx
		jmp	.input

		; exit edit mode, no saving

.escape:
		; restore original disc data

		mov	si,[disc_data]
		mov	di,si
		mov	cx,DISCINFO_MAXLEN
		add	si,cx
		rep	movsb

		; if there is no disc info, set cddb_info to 0

		mov	bx,-1
.esc1:		push	bx
		call	find_title
		pop	bx
		cmp	byte [si],0	; empty string?
		jnz	.esc2
		inc	bx
		cmp	bl,[last_track]
		jle	.esc1
		mov	byte [cddb_info],0
.esc2:		jmp	.qqq

		; exit and save

.save:		call	editor_done
		call	save_cddb_info

.qqq:		mov	byte [edit_mode],0
		jmp	redraw

		; move down

.down:		call	editor_done
		inc	bx
		cmp	bx,1
		jle	.gp
		jmp	edit.bx

		; move up

.up:		call	editor_done
		cmp	bx,1
		jle	.up1
		mov	bx,1
.up1:		dec	bx
		cmp	bx,-2
		jne	.up2
		mov	bx,1
.up2:		jmp	edit.0

		; next track

.grayplus:	call	editor_done
		inc	dx
		cmp	dl,[last_track]
		jbe	.gp0
		mov	dx,1
.gp0:		mov	bx,dx
.gp:		jmp	edit.0

		; previous track

.grayminus:	call	editor_done
		dec	dx
		jnz	.gp0
		mov	dl,[last_track]
		jmp	.gp0



edit_cddb_dir:	push	es
		mov	es,[scrseg]

		PLAYER_SETXY 0,11
		mov	si,cddbdir
		call	draw_lines
		mov	si,color_info3
		call	set_colors

		pop	es

		mov	bx,[cddb_filename]
		mov	byte [bx],0

		PLAYER_SETXY 2,12
		mov	si,cddb_path
		mov	cx,64
		mov	bx,si
		add	bx,cx
		call	editor_init

.input:		call	get_cd_status

		mov	ah,1
		int	0x16
		jz	.input
		xor	ax,ax
		int	0x16

		push	ax
		mov	si,color_info3
		call	set_colors
		pop	ax

		cmp	ax,0x011b	; ESC
		je	.escape
		cmp	ax,0x1c0d	; Enter
		je	.save

		; something else, pass to editor

		call	editor_handleevent
		call	editor_draw
		jmp	.input

.save:		call	editor_done
		mov	si,cddb_path
		call	strlen
		add	bx,cddb_path
		cmp	byte [bx-1],'\'
		je	.slash_ok

		mov	word [bx],'\'
		inc	bx

.slash_ok:	mov	[cddb_filename],bx

		call	get_executable_name
		jnz	.escape
		mov	ax,0x3d02
		mov	dx,executable_name
		int	0x21
		jb	.escape
		mov	bx,ax

		mov	ax,0x4200
		xor	cx,cx
		mov	dx,cddb_path_start-0x100
		int	0x21
		jb	.close

		mov	ah,0x40
		mov	dx,cddb_path_start
		mov	cx,cddb_path_end-cddb_path_start
		int	0x21

.close:		mov	ah,0x3e
		int	0x21
.escape:
		retn



zerozero:	db	0,0

get_executable_name:

		push	ds
		mov	ds,[0x2c]	; segment of environment

		xor	si,si
		mov	di,zerozero
		mov	cx,0x8000
		mov	dx,2
		call	scan_string
		jnz	.q
		add	si,dx
		add	si,2
		mov	di,executable_name
		xor	al,al
		call	copy_string
		cmp	ax,ax

.q:		pop	ds
		retn


; write message at DS:SI + CRLF then quit

msg_quit:	mov	ah,2
.0:		lodsb
		or	al,al
		jz	.1
		mov	dl,al
		int	0x21
		jmp	.0

.1:		mov	dl,13
		int	0x21
		mov	dl,10
		int	0x21
		mov	ax,0x4c00
		int	0x21


section .data

msg_NoMSCDEX:	db	'Install MSCDEX first.',0
msg_NoAudioCap:	db	'This drive cannot play audio discs.',0
msg_NoRedBook:	db	'This drive does not support Red Book addressing.',0

lb_Playing:	db	'playing'
lb_Stopped:	db	'stopped'
lb_Paused:	db	'paused'
lb_NoDisc:	db	'no disc'

playchars	db	''

; length: 33

line1		db	' Status ',7,' CD ',7,'Ŀ',0xFF
line2		db	'',12,'  Elapsed',10,' ',0xFF
line3		db	'',3,' /',3,' Ĵ Remaining',8,' ',0xFF
line4		db	'',10,' %  Total',12,' ',0xFF
line5		db	'',12,'',4,'[F1: Help]',4,'',0

; length: 64

info1		db	'',12,'',4,'[F1: Help]',4,'',30,'Ŀ',0xFF
info2		db	'',31,' ',31,' ',0xFF
info3		db	'   ',31,' ',26,' ',0xFF
info4		db	'',31,'',31,'Ĵ',0xFF
info5		db	'   ',31,' ',26,' ',0xFF
info6		db	'',30,'',25,'[',3,' %]',0

cddbdir		db	' Enter directory to retrieve CDDB data from: ',19,'Ŀ',0xFF
		db	'',31,' ',31,' ',4,' ','',0xFF
		db	'',31,'',31,'',4,'','',0

help		db	'retnE',9,'Play track',10
		db	'ecapskcaB',9,'Stop playing',10
		db	'ecapS',9,'Pause/Continue',10
		db	10
		db	'nDgP | nwoD | thgiR',9,'Next track',10
		db	'pUgP | pU | tfeL',9,'Previous track',10
		db	'emoH',9,'Skip to beginning of track',10
		db	'emoH+lrtC',9,'Skip to first track on disc',10
		db	'G',9,'Goto track (enter number)',10
		db	'9..0',9,'Skip to Nth percent of track',10
		db	10
		db	'tresnI',9,'Open/close tray',10
		db	'eteleD',9,'Reset CD-ROM',10
		db	10
		db	'1F',9,'Help',10
		db	'4F',9,'Edit disc info',10
		db	'4F+tfihS',9,'Set CDDB data location',10
		db	'csE | 01F',9,'Quit',10
		db	10
		db	0xF2,':edom tide nI',10
		db	'+ yarG',9,'Next track',10
		db	'- yarG',9,'Previous track',10
		db	'2F',9,'Save',10
		db	'csE',9,'Quit without saving',0

color_info:	db	4,0,6,0x8B
		db	22,0,2,0x8B
		db	5,2,5,0x8E
		db	1,1,12,0x8A
		db	1,3,12,0x8F
		db	15,3,16,0x83
		db	18,4,10,0x8C
		db	-1

color_info2:	db	18,4,10,0x8C
		db	2,5,61,0x82
		db	2,6,4,0x83
		db	7,6,56,0x82
		db	2,8,4,0x8F
		db	7,8,56,0x8A
		db	57,9,4,0x8F
		db	-1

color_info3:	db	4,11,43,0x8B
		db	2,12,65,0x8A
		db	-1

input_sizes:	db	0,6,0,0,9,0,5,0,0,2,7,7,11,13,11,11
output_sizes:	db	1,2,1,9,0,1

no_disc:	db	1
cddb_info	db	0
ticks		dw	0
click		dw	0

help_mode	db	0
edit_mode	db	0

cddb_textid	db	'# xmcd',10
cddb_LF		db	10
cddb_DISCID	db	'DISCID='
cddb_DTITLE	db	'DTITLE'
cddb_equalsign	db	'=',10
cddb_per	db	' / '
cddb_TTITLE	db	'TTITLE'
cddb_EXTD	db	'EXTD'
cddb_EXTT	db	'EXTT'
cddb_PLAYORDER	db	'PLAYORDER=',10

executable_name	times 128 db 0

cddb_path_start:
cddb_filename	dw	cddb_fn
cddb_path	db	'.\'
cddb_fn:	times cddb_path-$+128 db 0
cddb_path_end:

filler		times 2583 db 0

section .bss


old1c		resd	1

mono		resb	1
scrseg		resw	1
scrpage		resb	1
scrwidth	resw	1
scrheight	resw	1
scrsize		resw	1
cursor_x	resb	1
cursor_y	resb	1

drv_count	resw	1
drv_letters	resb	26
drive		resw	1
drive_not_ready	resb	1

media_changed	resb	1
paused		resb	1
playing		resb	1

first_track	resb	1
last_track	resb	1
lead_out	resd	1
track_ctrl_info	resb	1

track_start	resd	1	; in frames
track_len	resd	1	; in frames

track		resb	1
volume		resb	4
status		resd	1
frame_MSF	resd	1

tracktime	resd	1
disktime	resd	1

totaltime	resw	1
first_start	resw	1

upc		resb	8

req_hdr:	resb	ReqHdr_size
req_buf:	resb	13

HSG:		resd	1
SEC_buf:	resw	1

cddb_id		resb	8
cddb_sum	resw	1
cddb_filesize	resw	1

rows		resw	256

edit_buf	resb	128
edit_from	resw	1
edit_from_end	resw	1
edit_orig_len	resw	1
edit_max_len	resw	1
edit_offset	resw	1
edit_pos	resw	1

disc_data	resw	1

track_info:
