	/**************************************************/
	/*     An example of EMU8000 init and control     */
	/*       Version 1.0   (c) Grinus/ToM, 1996       */
	/*         Written for Borland C++ 3.1	          */
	/**************************************************/

/*
  This little proggie shows the direct control of EMU8000 wave-table
  synthesizer found on Sound Blaster AWE32 compatible sound cards.

  The following steps are taken:
    1. A complete initialization of the EMU chip
    2. A sample is sent to the on-board sample memory
    3. A note is started
    4. Various parameters of the sounding note are modified
    5. The note is released
    6. The EMU is reset to the default state (to provide FM effects)

  To hear the note, a reasonable setting of SB mixer is expected.
*/


#include <stdio.h>
#include <stdlib.h>

#include "awe.h"

// The definition of a sample (16-bit signed PCM sample)
#define FILENAME "STRINGS.RAW"
#define LOOPS     5112
#define LOOPE    15493
#define LENGTH   (LOOPE+ANTICLICK)

// The basic sample rate 22050Hz (-5 octaves from 176400Hz)
#define	AWEPITCH  (0x10000L -5 * 0x1000)
#define	LINPITCH  (0x10000L >> 5)

// --------------------------------------------------------------------- //

extern WORD Port620;	// EMU base port

// --------------------------------------------------------------------- //

WORD  AweRAMsize;	// [kB]
WORD *buf = NULL;	// ptr to a buffer for sample data
FILE *f   = NULL;	// for the sample file

// --------------------------------------------------------------------- //

void Err(char *s)	// abort with an error message
{
 if (f)			// a file is open?
    fclose(f);

 if (buf)		// a memory is allocated?
    free(buf);

 printf("ERROR: %s !\n", s);
 exit(1);
}


void ReadSample()
{
 // Allocate buf[]
 if (!(buf = (WORD *)malloc(LENGTH*2)))
    Err("Can't allocate a buffer");

 // Load the sample to buf[]
 if (!(f=fopen(FILENAME, "rb")))
    Err("Can't open the file " FILENAME );
 if (fread(buf, 2, LENGTH, f) != LENGTH)
    Err("Can't read the sample file");
 fclose(f); f = NULL;
}


// --------------------------  M A I N  ----------------------------- //

int main()
{
int   gChan = 0;	// let's use the first of 30 available channels

 puts("\nThe example of EMU8000 control.  Copyright (c) Grinus/ToM, 1996.\n");

 if (!AweDetect() )		// find EMU8000 chip
    Err("EMU8000 not found");

 if (!(AweRAMsize = AweInitHw())) // error or no sample RAM ?
    Err("EMU8000 init error");

 atexit( AweTerminate );	// setup the EMU clean-up routine

 // the start of main code

 // Report the configuration
 printf("EMU8000 found at port 0x%X with %d KB of sample RAM.\n",
			 Port620, AweRAMsize );

 // Set the EMU8000 internal equalizer
 AweTrebleBass(5, 5);		// 0dB, 0dB

 // Set the reverb and chorus types
 AweReverbType(0);		// Room 1
 AweChorusType(3);		// Chorus 4

 // Read a sample file
 ReadSample();

 // Send the sample to the sample RAM
 AweEnableRam(0);
 AweWrD(AWE_WrAdrD, TOPRAM >> 16, TOPRAM & 0xFFFF );	// set address
 AweWrBlock(buf, LENGTH);				// send buf[]
 AweDisableRam();

 // Start a note
 AweNoteOn( gChan,
	    6 * 16 / 6,		// Volume = -6 dB
	    0,			// Panning = left
	    25 * 256 / 100,	// Reverb = 25%
	    4 * 256 / 100,	// Chorus = 4%
	    AWEPITCH,		// AwePitch
	    LINPITCH,		// AweLinPitch
	    TOPRAM,  		// Starting addr in the sample memory
	    TOPRAM+LOOPS,	// LoopS
	    TOPRAM+LOOPE	// LoopE
	  );
 puts("A note started.");

 // Wait for 2 sec
 AweWait(44100); AweWait(44100);

 // Change the volume
 AweWrW( AWE_FC_Vol |gChan, 0xFF00 |   		// FilterCutOff = max
			    15 * 16 / 6 );	// Volume = -15 dB
 puts("The volume decreased.");

 // Wait for 2 sec
 AweWait(44100); AweWait(44100);

 // Make a pitch & panning slide
 puts("The pitch and panning slide started.");
 for (int i=0; i<64; i++) {
    BYTE pan = 4 * i;			// slide from very left to right
    WORD pitch = AWEPITCH + 64 * i;	// slide-up by one octave
    DWORD l;				// temp. storage

    // Change the pitch
    AweWrW( AWE_Pitch |gChan, pitch );		// new Pitch

    // Change the panning
    l = AweRdD( AWE_DP_Rev_Pan |gChan);
    AweWrD( AWE_DP_Rev_Pan |gChan, HWORD(l),    // old DestPitch
			   ((WORD)l & 0xFF00)	// old Reverb
			   | pan );		// new RightPan
    l = AweRdD( AWE_Pan_Loops |gChan );
    AweWrD( AWE_Pan_Loops |gChan,
			   ~(WORD)pan <<8	// new LeftPan
			   | (BYTE)HWORD(l), (WORD)l );// old LoopS
    // Wait for 1/64 sec
    AweWait(44100/64);
    }

 // Wait for 1 sec
 AweWait(44100);

 // Release a note in gChan
 NoteOff(gChan);
 puts("The note released.");

 return 0;
}
