// A-B test audio comparison program
// (C) 2000, Mike Ingle (inglem@dslinks.com or mikeingle@delphi.com)
// This program is in the public domain.
// To compile: cl /O2 abtest.c winmm.lib


#include <windows.h>
#include <stdio.h>
#include <mmsystem.h>

#define MAXINPUTFILES 26
#define DEFAULT_BUFSIZE (1024*1024)
#define READ_HEADER_SIZE 16384
#define ESCKEY 27

#define HELPSTRING "\
ABTEST V1.0 (C) 2000, Mike Ingle \
(inglem@dslinks.com or mikeingle@delphi.com)\n\
\nUsage: %s wavfile1 wavfile2 [-ib inbufsize]\
 [-ob outbufsize]\n\
  A-Z or 1-9 to select input file during playback\n\
  [ Back 10 seconds        (no shift key)\n\
  ] Forward 10 seconds     (no shift key)\n\
  { Back 30 seconds        (with shift key)\n\
  } Forward 30 seconds     (with shift key)\n\
  . Pause/resume\n\
  ESC exit\n"


typedef struct {
  LPTSTR filepath; // Points to argv[] string
  HANDLE fh;
  LPBYTE rdbuf; // These two will be
  LPBYTE wrbuf; // swapped at each buffer change
  unsigned bitsPerSample;
  unsigned samplesPerSecond;
  unsigned numChannels;
  unsigned dataStart; // Offset in file of actual PCM data
  unsigned dataSize; // Size in bytes of actual PCM data
  unsigned dataRead; // Number of bytes already read
  } perFileInfo;

perFileInfo inputFiles[MAXINPUTFILES];
unsigned numInputFiles;
unsigned inputBufSize;
int largestFileNum;
unsigned totalDataSize;
WAVEFORMATEX wavFormat;
HANDLE wavPort;
HANDLE nextPlayBufferEvent;
HANDLE nextInputBufferEvent;
unsigned playBufSize=0;
WAVEHDR whdr1,whdr2;
WAVEHDR* playBuf1=&whdr1;
WAVEHDR* playBuf2=&whdr2;
BOOL fileReaderExit=FALSE;
unsigned bytesPlayed;
char globalUserCommand;
unsigned fileToPlay;


////////////////////////////////////////////////////////////////////////////
// Display the error from GetLastError
void show_io_error(char* fmt,char* pre1,char* pre2,char* post1,char* post2)
{
char errbuf[256];
int eol=0;
FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM,
  NULL,GetLastError(),MAKELANGID(LANG_NEUTRAL,SUBLANG_DEFAULT),
  errbuf,sizeof(errbuf),NULL);
eol=strlen(errbuf)-1;
while((eol>0)&&((errbuf[eol]=='\r')||
      (errbuf[eol]=='\n')||(errbuf[eol]=='.'))) {
  errbuf[eol]=0; eol--;
  }
printf(fmt,pre1,pre2,errbuf,post1,post2);
}


////////////////////////////////////////////////////////////////////////////
// Read next buffer from all files assuming they are not done
// Returns FALSE when all files at EOF
BOOL readNextFileBuffer()
{
int i;
unsigned bytesToRead;
unsigned bytesLeftInFile;
unsigned bytesRead;
int numFilesRead=0;

for(i=0;i<numInputFiles;i++) {
  bytesToRead=inputBufSize;
  bytesLeftInFile=inputFiles[i].dataSize-inputFiles[i].dataRead;
  if(bytesLeftInFile==0) continue;
  numFilesRead++;
  if(bytesLeftInFile<bytesToRead) {
    bytesToRead=bytesLeftInFile;
    memset(inputFiles[i].rdbuf+bytesToRead,
      wavFormat.wBitsPerSample==16?0:128,inputBufSize-bytesToRead);
    }
  ReadFile(inputFiles[i].fh,inputFiles[i].rdbuf,bytesToRead,&bytesRead,NULL);
  inputFiles[i].dataRead+=bytesToRead;
  }
if(numFilesRead>0) return(TRUE); else return(FALSE);
}


////////////////////////////////////////////////////////////////////////////
// File reader thread
DWORD WINAPI fileReaderThread(LPVOID data)
{
BOOL notdone;
do {
  WaitForSingleObject(nextInputBufferEvent,INFINITE);
  if(fileReaderExit) break;
  notdone=readNextFileBuffer();
  } while(notdone);

return(0);
}


////////////////////////////////////////////////////////////////////////////
// Swap the two input buffers
void flipInputBuffers()
{
int i;
LPBYTE bp;
for(i=0;i<numInputFiles;i++) { // Swap buffers
  bp=inputFiles[i].rdbuf;
  inputFiles[i].rdbuf=inputFiles[i].wrbuf;
  inputFiles[i].wrbuf=bp;
  }
}


////////////////////////////////////////////////////////////////////////////
// Playback function
void playbackThread()
{
unsigned newFileToPlay;
unsigned bufPos;
unsigned hr,min,sec,tenth;
WAVEHDR* whp;
int ch;

bufPos=0;
memcpy(playBuf1->lpData,inputFiles[fileToPlay].wrbuf+bufPos,playBufSize);
bufPos+=playBufSize; bytesPlayed+=playBufSize;
memcpy(playBuf2->lpData,inputFiles[fileToPlay].wrbuf+bufPos,playBufSize);
bufPos+=playBufSize; bytesPlayed+=playBufSize;
ResetEvent(nextPlayBufferEvent); // Event already set by system
waveOutWrite(wavPort,playBuf1,sizeof(WAVEHDR));
waveOutWrite(wavPort,playBuf2,sizeof(WAVEHDR));

globalUserCommand=0;
while(bytesPlayed<totalDataSize) {
  WaitForSingleObject(nextPlayBufferEvent,INFINITE);

  // Handle keystroke input
  ch=0;
  if(_kbhit()) {
    ch=_getch();
    if(ch>='a'&&ch<='z') ch-=32;
    if((ch>='A'&&ch<='Z')) newFileToPlay=ch-'A'; 
    else if((ch>='1'&&ch<='9')) newFileToPlay=ch-'1'; 
    if(newFileToPlay<numInputFiles) fileToPlay=newFileToPlay;
    }

  // Queue up next buffer
  memcpy(playBuf1->lpData,inputFiles[fileToPlay].wrbuf+bufPos,playBufSize);

  bufPos+=playBufSize; bytesPlayed+=playBufSize;
  waveOutWrite(wavPort,playBuf1,sizeof(WAVEHDR));
  whp=playBuf1; playBuf1=playBuf2; playBuf2=whp;
  if(bufPos>=inputBufSize) { // Reload input buffer
    flipInputBuffers();    
    bufPos=0;
    SetEvent(nextInputBufferEvent);
    }

  // Exit if user command requires it 
    if((ch==ESCKEY)||(ch=='.')||(ch=='[')||(ch==']')||(ch=='{')||(ch=='}')) { 
      fileReaderExit=TRUE;
      globalUserCommand=ch;
      break;
      }

  // Display counter
  tenth=bytesPlayed/(wavFormat.nAvgBytesPerSec/10);
  sec=tenth/10; tenth%=10;
  min=sec/60; sec%=60;
  hr=min/60; min%=60;
  printf("\r%02u:%02u:%02u.%01u %c ",hr,min,sec,tenth,(char)('A'+fileToPlay));

  }
WaitForSingleObject(nextPlayBufferEvent,INFINITE);
WaitForSingleObject(nextPlayBufferEvent,INFINITE);
}


////////////////////////////////////////////////////////////////////////////
// Program entry point
int main(int argc,char* argv[])
{
unsigned i;
DWORD bytesRead;
BOOL errorBailout;
LPBYTE bp;
unsigned bpc;
unsigned chunkLen;
DWORD dw;
MMRESULT errcode;
HANDLE playbackThreadHandle;
HANDLE fileReaderThreadHandle;
unsigned byteOffset;

// Show usage help
if(argc<2) {
  printf(HELPSTRING,argv[0]);
  errorBailout=TRUE; goto abort_no_action;
  }

// Parse argv input
inputBufSize=DEFAULT_BUFSIZE; numInputFiles=0;
errorBailout=FALSE;
for(i=1;i<argc;i++) {
  if(stricmp(argv[i],"-ib")==0) { // Got input buffer size
    i++; 
    if(i<argc) {
      sscanf(argv[i],"%u",&inputBufSize);
      }
    }
  else if(stricmp(argv[i],"-ob")==0) { // Got output buffer size
    i++; 
    if(i<argc) {
      sscanf(argv[i],"%u",&playBufSize);
      }
    }
  else if(numInputFiles<MAXINPUTFILES) { // Input file name
    inputFiles[numInputFiles].filepath=argv[i];
    numInputFiles++;
    }
  }

// Open all input files
for(i=0;i<numInputFiles;i++) {
  inputFiles[i].fh=CreateFile(inputFiles[i].filepath,
        GENERIC_READ,FILE_SHARE_READ,NULL,OPEN_EXISTING,0,NULL);
  if(inputFiles[i].fh==INVALID_HANDLE_VALUE) { // Open failed
    show_io_error("Opening %s%s%s%s%s\n",inputFiles[i].filepath,": ","","");
    errorBailout=TRUE; 
    }
  }
if(errorBailout) goto abort_close_files_only;

// Figure out input buffer size
if(inputBufSize<=(256*1024)) inputBufSize=(256*1024);
else if(inputBufSize<=(512*1024)) inputBufSize=(512*1024);
else if(inputBufSize<=(1024*1024)) inputBufSize=(1024*1024);

// Allocate all input buffers
for(i=0;i<numInputFiles;i++) {
  inputFiles[i].rdbuf=(LPBYTE)calloc(1,inputBufSize);
  inputFiles[i].wrbuf=(LPBYTE)calloc(1,inputBufSize);
  if((inputFiles[i].rdbuf==NULL)||(inputFiles[i].wrbuf==NULL)) {
    printf("Memory out error allocating input buffers.\n");
    errorBailout=TRUE; break;
    }
  }
if(errorBailout) goto abort_close_files_only;

// Parse the WAVE file headers
totalDataSize=0;
for(i=0;i<numInputFiles;i++) {
  bp=inputFiles[i].rdbuf;
  ReadFile(inputFiles[i].fh,bp,READ_HEADER_SIZE,&bytesRead,NULL);
  if(bp[0]!='R'||bp[1]!='I'||bp[2]!='F'||bp[3]!='F') {
    printf("File %s has no RIFF header.\n",inputFiles[i].filepath);
    errorBailout=TRUE; continue;
    }
  if(bp[8]!='W'||bp[9]!='A'||bp[10]!='V'||bp[11]!='E') {
    printf("File %s has no WAVE header.\n",inputFiles[i].filepath);
    errorBailout=TRUE; continue;
    }
  // Find the fmt tag
  for(bpc=12;bpc<READ_HEADER_SIZE;bpc+=8+chunkLen) {
    chunkLen=bp[bpc+7]; chunkLen<<=8; chunkLen|=bp[bpc+6]; chunkLen<<=8;
    chunkLen|=bp[bpc+5]; chunkLen<<=8; chunkLen|=bp[bpc+4];
    if(bp[bpc]=='f'&&bp[bpc+1]=='m'&&bp[bpc+2]=='t') break;
    }
  if(bpc>=READ_HEADER_SIZE) {
    printf("File %s has no fmt header.\n",inputFiles[i].filepath);
    errorBailout=TRUE; continue;
    }
  // Parse the fmt data
  if(bp[bpc+8]!='\001'||bp[bpc+9]!='\000') {
    printf("File %s is not in PCM format.\n",inputFiles[i].filepath);
    errorBailout=TRUE; continue;
    }
  inputFiles[i].numChannels=bp[bpc+10];
  dw=bp[bpc+15]; dw<<=8; dw|=bp[bpc+14]; dw<<=8; dw|=bp[bpc+13];
  dw<<=8; dw|=bp[bpc+12]; inputFiles[i].samplesPerSecond=dw;
  inputFiles[i].bitsPerSample=bp[bpc+22];
  // Find the data tag and set the file pointer to the start of data
  for(bpc+=8+chunkLen;bpc<READ_HEADER_SIZE;bpc+=8+chunkLen) {
    chunkLen=bp[bpc+7]; chunkLen<<=8; chunkLen|=bp[bpc+6]; chunkLen<<=8;
    chunkLen|=bp[bpc+5]; chunkLen<<=8; chunkLen|=bp[bpc+4];
    if(bp[bpc]=='d'&&bp[bpc+1]=='a'&&bp[bpc+2]=='t'&&bp[bpc+3]=='a') break;
    }
  if(bpc>=READ_HEADER_SIZE) {
    printf("File %s has no data header.\n",inputFiles[i].filepath);
    errorBailout=TRUE; continue;
    }
  if(chunkLen>totalDataSize) { // Find largest file
    totalDataSize=chunkLen;
    largestFileNum=i;
    }
  inputFiles[i].dataSize=chunkLen; inputFiles[i].dataStart=bpc+8;
  inputFiles[i].dataRead=0;
  SetFilePointer(inputFiles[i].fh,inputFiles[i].dataStart,0,FILE_BEGIN);
  }
if(errorBailout) goto abort_close_files_only;

// Display format, and check to make sure all files are same format
printf("WAV format is %i channel%s, %i Hz, %i bits per sample\n",
  inputFiles[0].numChannels,inputFiles[0].numChannels==1?"":"s",
  inputFiles[0].samplesPerSecond,inputFiles[0].bitsPerSample);
for(i=1;i<numInputFiles;i++) {
  if(inputFiles[i].numChannels!=inputFiles[0].numChannels ||
     inputFiles[i].samplesPerSecond!=inputFiles[0].samplesPerSecond ||
     inputFiles[i].bitsPerSample!=inputFiles[0].bitsPerSample) {
    printf("File %s has a different format and will not sound right.\n\
 (%i channel%s, %i Hz, %i bits per sample)\n",inputFiles[i].filepath,
    inputFiles[i].numChannels,inputFiles[i].numChannels==1?"":"s",
    inputFiles[i].samplesPerSecond,inputFiles[i].bitsPerSample);
    }
  }

// Configure the WAV output device
wavFormat.wFormatTag=WAVE_FORMAT_PCM;
wavFormat.nChannels=inputFiles[0].numChannels;
wavFormat.nSamplesPerSec=inputFiles[0].samplesPerSecond;
wavFormat.wBitsPerSample=inputFiles[0].bitsPerSample;
wavFormat.nAvgBytesPerSec=wavFormat.nSamplesPerSec * wavFormat.nChannels;
if(wavFormat.wBitsPerSample==16) wavFormat.nAvgBytesPerSec*=2;
wavFormat.nBlockAlign=(wavFormat.wBitsPerSample*wavFormat.nChannels)/8;

// Figure out the play buffer size
if(playBufSize>0) {
  if(playBufSize<=4096) playBufSize=4096;
  else if(playBufSize<=8192) playBufSize=8192;
  else if(playBufSize<=16384) playBufSize=16384;
  else if(playBufSize<=32768) playBufSize=32768;
  else playBufSize=65536;
  }
else if(wavFormat.nAvgBytesPerSec<=22050) playBufSize=8192;
else if(wavFormat.nAvgBytesPerSec<=44100) playBufSize=16384;
else if(wavFormat.nAvgBytesPerSec<=88200) playBufSize=32768;
else playBufSize=65536;
printf("Input buffer %u bytes, output buffer %u bytes\n",
       inputBufSize,playBufSize);

// Allocate the play buffers
playBuf1->lpData=calloc(1,playBufSize);
playBuf1->dwBufferLength=playBufSize; playBuf1->dwFlags=0;
playBuf2->lpData=calloc(1,playBufSize);
playBuf2->dwBufferLength=playBufSize; playBuf2->dwFlags=0;
if((playBuf1->lpData==NULL)||(playBuf2->lpData==NULL)) {
  printf("Memory out allocating playback buffers.\n");
  goto abort_close_files_only; 
  }
nextPlayBufferEvent=CreateEvent(NULL,FALSE,FALSE,NULL);
nextInputBufferEvent=CreateEvent(NULL,FALSE,FALSE,NULL);
errcode=waveOutOpen(&wavPort,WAVE_MAPPER,&wavFormat,
  (DWORD)nextPlayBufferEvent,0,CALLBACK_EVENT);
if(errcode!=MMSYSERR_NOERROR) {
  show_io_error("%s%s%s%s%s","Error opening WAVE device: ","","","\n");
  goto abort_close_files_only;
  } 
errcode=waveOutPrepareHeader(wavPort,playBuf1,sizeof(WAVEHDR));
if(errcode==MMSYSERR_NOERROR)
  errcode=waveOutPrepareHeader(wavPort,playBuf2,sizeof(WAVEHDR));
if(errcode!=MMSYSERR_NOERROR) {
  show_io_error("%s%s%s%s%s","Error preparing WAV buffers: ","","","\n");
  goto abort_close_files_only;
  }

// Set to start at beginning of file
bytesPlayed=0; fileToPlay=0;

for(;;) {
  // Load the two buffers with audio data
  readNextFileBuffer();
  flipInputBuffers();
  readNextFileBuffer();

  // Start the file reader thread
  fileReaderExit=FALSE;
  fileReaderThreadHandle=CreateThread(NULL,0,fileReaderThread,0,0,&dw);

  // Start the playback thread
  playbackThread();

  // Catch exiting file reader thread
  SetEvent(nextInputBufferEvent);
  WaitForSingleObject(fileReaderThreadHandle,INFINITE);

  // Figure out why we exited and what to do next
  if(globalUserCommand=='.') { // Pause
    _getch(); // Wait for resume
    }
  else if(globalUserCommand=='[') { // Back 10
    byteOffset=10*wavFormat.nAvgBytesPerSec;
    if(bytesPlayed>byteOffset) bytesPlayed-=byteOffset;
    else bytesPlayed=0;
    }
  else if(globalUserCommand=='{') { // Back 30
    byteOffset=30*wavFormat.nAvgBytesPerSec;
    if(bytesPlayed>byteOffset) bytesPlayed-=byteOffset;
    else bytesPlayed=0;
    }
  else if(globalUserCommand==']') { // Forward 10
    byteOffset=10*wavFormat.nAvgBytesPerSec;
    if(bytesPlayed+byteOffset<totalDataSize) bytesPlayed+=byteOffset;
    }
  else if(globalUserCommand=='}') { // Forward 30
    byteOffset=30*wavFormat.nAvgBytesPerSec;
    if(bytesPlayed+byteOffset<totalDataSize) bytesPlayed+=byteOffset;
    }

  if(globalUserCommand<=ESCKEY) break; // End of file or ESC exit
   
  // Reset file pointer
  for(i=0;i<numInputFiles;i++) {
    inputFiles[i].dataRead=bytesPlayed;
    SetFilePointer(inputFiles[i].fh,inputFiles[i].dataStart+bytesPlayed,
                   0,FILE_BEGIN);
    }
  }

// Close the audio device
waveOutClose(wavPort);

// Close the open files
abort_close_files_only:
for(i=0;i<numInputFiles;i++) {
  if(inputFiles[i].fh!=INVALID_HANDLE_VALUE) CloseHandle(inputFiles[i].fh);
  }

abort_no_action:
return(errorBailout?1:0);
}



// EOF
