TITLE FixedPointMath_Signed

; FixedPoint Math Interface for Higher Level Languages
; Support for 16- and 32-bit signed FixedPoint Numbers
;
; Requires:
;
; All variables must be declared this way:
;
; Pascal:
; Type Num = (Long, Int);
;      FPNum = Record
;               Case Typ: Num Of
;                Long: (L: Longint);
;                Int : (I: Integer);
;              End;
;
; C/C++:
; typedef union FPNum
;          {
;           L: long;
;           I: int;
;          }


MODEL Small

LOCALS

.386

.DATA

INCLUDE Sinus.Inc

; Gonionometrical constants
; Originally:    = 3.14159265359
;             /2 = 1.57079632679

; 32-bit radian constants

PI           EQU           3243FH      ;3.243FH = 3.14158630371 = 
PI2          EQU           19220H      ;1.9220H = 1.57080078125 = /2

; 16-bit radian constants

IPI          EQU             324H      ;3.24H = 3.140625  = 
IPI2         EQU             192H      ;1.92H = 1.5703125 = /2

; Length of sinus table

MLQrt        DW            00A30H      ;4096b

; Square Root Accuracy

SqrtLimit    EQU           00006H      ;1/10000 = 9.1552734375e-5
ISqrtLimit   EQU               2H      ;1/100   = 7.8125e-3

.CODE

;************************************************************************
;*              32-bit Fixed Point Math - FPXXXX functions              *
;*                  16-bit Integer and Fractional part                  *
;*                           386+ Required                              *
;************************************************************************

; Add two 32-bit numbers A+B
; Result = DX:AX

PUBLIC FPPlus
FPPlus PROC FAR
 ARG B: DWORD, A: DWORD
                    ENTER  0,0
                    MOV    EAX,[A]
                    MOV    EBX,[B]
                    ADD    EAX,EBX
                    MOV    EDX,EAX
                    SHR    EDX,16
                    LEAVE
                    RET    8
ENDP

; Substract two 32-bit numbers A-B
; Result = DX:AX

PUBLIC FPMinus
FPMinus PROC FAR
 ARG B: DWORD, A: DWORD
                    ENTER  0,0
                    MOV    EAX,[A]
                    MOV    EBX,[B]
                    SUB    EAX,EBX
                    MOV    EDX,EAX
                    SHR    EDX,16
                    LEAVE
                    RET    8
ENDP

; Multiply two 32-bit numbers A*B
; Result = DX:AX

PUBLIC FPMul
FPMul PROC FAR
 ARG B: DWORD, A: DWORD
                    ENTER  0,0
                    MOV    EAX,[A]
                    MOV    EBX,[B]
                    IMUL   EBX
                    SHR    EAX,16
                    LEAVE
                    RET    8
ENDP

PUBLIC FPMul64
FPMul64 PROC FAR
 ARG D: DWORD, C: DWORD, B: DWORD, A: DWORD
                    ENTER  0,0
;                    PUSH   BP
;                    MOV    BP,SP
                    MOV    EAX,[A]
                    MOV    EBX,[B]
                    IMUL   EBX
                    LES    DI,C
                    MOV    ES:[DI],EAX
                    LES    DI,D
                    MOV    ES:[DI],EDX
;                    POP    BP
                    LEAVE
                    RET    20
ENDP

; Divide two 32-bit numbers A/B
; Result = DX:AX

PUBLIC FPDiv
FPDiv PROC FAR
 ARG B: DWORD, A: DWORD
                    ENTER  0,0
                    MOV    EBX,[B]
;                    OR     EBX,EBX         ; Null Test
;                    JZ     @FPDZero
                    MOV    EAX,[A]
                    MOV    EDX,EAX
                    SAR    EDX,16
                    SHL    EAX,16
                    IDIV   EBX
                    MOV    EDX,EAX
                    SHR    EDX,16
        @FPDStore:  LEAVE
                    RET    8
;        @FPDZero:   MOV    AX,0FFFFH       ; Infinity X/0
;                    MOV    DX,7FFFH
;                    JMP    @FPDStore
ENDP

PUBLIC FPDivMod
FPDivMod PROC FAR
 ARG M: DWORD, C: DWORD, B: DWORD, A: DWORD
                    PUSH   BP
                    MOV    BP,SP
                    MOV    EAX,[A]
                    MOV    EDX,EAX
                    SHR    EDX,16
                    SHL    EAX,16
                    MOV    EBX,[B]
                    OR     EBX,EBX
                    JZ     @FPDMZero
                    IDIV   EBX
        @FPDMStore: LES    DI,C
                    MOV    ES:[DI],EAX
                    LES    DI,M
                    SHL    EDX,16
                    MOV    ES:[DI],EDX
                    POP    BP
                    LEAVE
                    RET    20
        @FPDMZero:  MOV    EAX,7FFFFFFFH       ; Infinity X/0
                    JMP    @FPDMStore
ENDP

PUBLIC FPDiv64
FPDiv64 PROC FAR
 ARG M: DWORD, C: DWORD, B: DWORD, A: DWORD, D: DWORD
                    PUSH   BP
                    MOV    BP,SP
                    MOV    EAX,[A]
                    MOV    EDX,[D]
                    MOV    EBX,[B]
                    OR     EBX,EBX
                    JZ     @FPD6Zero
                    IDIV   EBX
        @FPD6Store: LES    DI,C
                    MOV    ES:[DI],EAX
                    LES    DI,M
                    MOV    ES:[DI],EDX
                    POP    BP
                    RET    20
        @FPD6Zero:  XOR    EAX,EAX
                    JMP    @FPD6Store
ENDP

; Integer part of 32-bit number A
; Result = DX:AX

PUBLIC FPInt
FPInt PROC FAR
 ARG A: DWORD
                    ENTER  0,0
                    MOV    EAX,[A]
                    MOV    EDX,EAX
                    SHR    EDX,16
                    LEAVE
                    RET    4
ENDP

; Fractional part of 32-bit number A
; Result = DX:AX

PUBLIC FPFrac
FPFrac PROC FAR
 ARG A: DWORD
                    ENTER  0,0
                    MOV    AX,WORD PTR [A]
                    XOR    DX,DX
                    LEAVE
                    RET    4
ENDP

; Integer part of 32-bit number A
; Converts value into a 16-bit number
; Result = AX

PUBLIC FPTrunc
FPTrunc PROC FAR
 ARG A: DWORD
                    ENTER  0,0
                    MOV    EAX,[A]
                    SHR    EAX,16
                    LEAVE
                    RET    4
ENDP

; Opposite value of 32-bit number A
; Computes binary compound of a number A
; Result = DX:AX

PUBLIC FPNeg
FPNeg PROC FAR
 ARG A: DWORD
                    ENTER  0,0
                    MOV    EAX,[A]
                    NEG    EAX
                    MOV    EDX,EAX
                    SHR    EDX,16
                    LEAVE
                    RET    4
ENDP

; Sinus(A), computes Sin(X) of 32-bit number A
; A = angle in radians
; Result = DX:AX

PUBLIC FPSin
FPSin PROC FAR
 ARG A: DWORD
                    ENTER  0,0
                    MOV    EAX,[A]
                    OR     EAX,EAX
                    JNS    @SSign
                    NEG    EAX
                    ADD    EAX,PI
        @SSign:     CMP    EAX,2*PI
                    JNA    @SNAC
                    MOV    ECX,EAX
                    MOV    EBX,2*PI
                    XOR    EDX,EDX
                    DIV    EBX
                    MUL    EBX
                    SUB    ECX,EAX
                    MOV    EAX,ECX
        @SNAC:      MOV    ECX,EAX
                    MOVZX  EBX,[MLQrt]
                    IMUL   EBX
                    SHR    EAX,16
                    MOV    BX,AX
                    CMP    ECX,PI2
                    JNA    @SDirPut
                    SUB    BX,1000H
                    CMP    ECX,PI
                    JB     @SRevert
                    SUB    BX,1001H
                    CMP    ECX,PI+PI2
                    JNA    @SDirPut
                    SUB    BX,1000H
        @SRevert:   MOV    DX,1000H
                    SUB    DX,BX
                    MOV    BX,DX
        @SDirPut:   LEA    SI,SinusTable
                    SHL    BX,1
                    MOV    AX,DS:[SI+BX]
                    CMP    ECX,PI
                    JNA    @SStore
                    NEG    EAX
        @SStore:    MOV    EDX,EAX
                    SHR    EDX,16
                    LEAVE
                    RET    4
ENDP

; Cosinus(A), computes Cos(X) of 32-bit number A
; A = angle in radians
; Result = DX:AX

PUBLIC FPCos
FPCos PROC FAR
 ARG A: DWORD
                    ENTER  0,0
                    MOV    EAX,[A]
                    ADD    EAX,PI2
                    MOV    [A],EAX
                    LEAVE
                    JMP    FPSin
ENDP

PUBLIC FPSqr
FPSqr PROC FAR
ENDP

; Sqrt(A), computes Square Root of 32-bit number A
; Result = DX:AX

PUBLIC FPSqrT
FPSqrT PROC FAR
 ARG A: DWORD
                    ENTER  0,0
		    MOV    EAX,10000H             ; Starting iteration = 1
                    MOV    ECX,[A]                ; ECX=A
        @SqrtIter:  MOV    EBX,EAX                ; EBX=B=Iteration result
                    MOV    EDX,ECX
                    MOV    EAX,ECX
                    SHR    EDX,16
                    SHL    EAX,16                 ; A/B
                    IDIV   EBX
                    ADD    EAX,EBX                ; B=B+A/B
                    SAR    EAX,1                  ; B=1/2*(B+A/B)
                    MOV    EDX,EAX
                    SUB    EDX,EBX
                    JNS    @CmpSqrt               ; Absolute value
                    NEG    EDX
        @CmpSqrt:   CMP    EDX,SqrtLimit          ; X(i+1)-Xi<SqrtLimit?
                    JA     @SqrtIter
                    MOV    EDX,EAX                ; Store result into DX:AX
                    SHR    EDX,16
                    LEAVE
                    RET    4
ENDP

PUBLIC FPExp
FPExp PROC FAR
ENDP

PUBLIC FPLn
FPLn PROC FAR
ENDP

;************************************************************************
;*              32-bit Fixed Point Math - FPEXXXX functions             *
;*                24-bit Integer and 8-bit Fractional part              *
;*                           386+ Required                              *
;************************************************************************

; Add two 32-bit numbers A+B
; Result = AX

PUBLIC FPEPlus
FPEPlus PROC FAR
 ARG B: DWORD, A: DWORD
                    ENTER  0,0
                    MOV    EAX,[A]
                    MOV    EBX,[B]
                    ADD    EAX,EBX
                    MOV    EDX,EAX
                    SHR    EDX,16
                    LEAVE
                    RET    8
ENDP

; Substract two 32-bit numbers A-B
; Result = DX:AX

PUBLIC FPEMinus
FPEMinus PROC FAR
 ARG B: DWORD, A: DWORD
                    ENTER  0,0
                    MOV    EAX,[A]
                    MOV    EBX,[B]
                    SUB    EAX,EBX
                    MOV    EDX,EAX
                    SHR    EDX,16
                    LEAVE
                    RET    8
ENDP

; Multiply two 32-bit numbers A*B
; Result = DX:AX

PUBLIC FPEMul
FPEMul PROC FAR
 ARG B: DWORD, A: DWORD
                    ENTER  0,0
                    MOV    EAX,[A]
                    MOV    EBX,[B]
                    IMUL   EBX
                    SHLD   EDX,EAX,8
                    SHR    EAX,8
                    LEAVE
                    RET    8
ENDP

; Divide two 32-bit numbers A/B
; Result = DX:AX

PUBLIC FPEDiv
FPEDiv PROC FAR
 ARG B: DWORD, A: DWORD
                    ENTER  0,0
                    MOV    EBX,[B]
;                    OR     EBX,EBX         ; Null Test
;                    JZ     @FPEDZero
                    MOV    EAX,[A]
                    MOV    EDX,EAX
                    SAR    EDX,24
                    SHL    EAX,8
                    IDIV   EBX
                    MOV    EDX,EAX
                    SHR    EDX,16
        @FPEDStore: LEAVE
                    RET    8
;        @FPEDZero:  MOV    AX,0FFFFH       ; Infinity X/0
;                    MOV    DX,7FFFH
;                    JMP    @FPEDStore
ENDP

;************************************************************************
;*              16-bit Fixed Point Math, FPIXXXX Functions              *
;*                   8-bit Integer and Fractional part                  *
;*                             186+ Required                            *
;************************************************************************

;NOTE: All arguments are passed as DWORDs
;----------------------------------------

; Add two 16-bit numbers A+B
; Result = AX

PUBLIC FPIPlus
FPIPlus PROC FAR
 ARG B: DWORD, A: DWORD
                    ENTER  0,0
                    MOV    AX,WORD PTR [A]
                    MOV    BX,WORD PTR [B]
                    ADD    AX,BX
                    LEAVE
                    RET    8
ENDP

; Substract two 16-bit numbers A-B
; Result = AX

PUBLIC FPIMinus
FPIMinus PROC FAR
 ARG B: DWORD, A: DWORD
                    ENTER  0,0
                    MOV    AX,WORD PTR [A]
                    MOV    BX,WORD PTR [B]
                    SUB    AX,BX
                    LEAVE
                    RET    8
ENDP

; Multiply two 16-bit numbers A*B
; Result = AX

PUBLIC FPIMul
FPIMul PROC FAR
 ARG B: DWORD, A: DWORD
                    ENTER  0,0
                    MOV    AX,WORD PTR [A]
                    MOV    BX,WORD PTR [B]
                    IMUL   BX
                    SHR    AX,16
                    LEAVE
                    RET    8
ENDP

; Multiply two 16-bit numbers A*B
; Converts value into 32-bit number
; Result = DX:AX

PUBLIC FPIMul32
FPIMul32 PROC FAR
 ARG B: DWORD, A: DWORD
                    ENTER  0,0
                    MOV    AX,WORD PTR [A]
                    MOV    BX,WORD PTR [B]
                    IMUL   BX
                    LEAVE
                    RET    8
ENDP

; Divide two 16-bit numbers A/B
; Result = AX

PUBLIC FPIDiv
FPIDiv PROC FAR
 ARG B: DWORD, A: DWORD
                    ENTER  0,0
                    MOV    BX,WORD PTR [B]       ; Null Test
;                    OR     BX,BX
;                    JZ     @FPDIZero
                    MOV    AX,WORD PTR [A]
                    MOV    DX,AX
                    SAR    DX,8
                    SHL    AX,8
                    IDIV   BX
       @FPDIStore:  LEAVE
                    RET    8
;       @FPDIZero:   MOV    AX,07FFFH       ; Infinity X/0
;                    JMP    @FPDIStore
ENDP

; Divide two 16-bit numbers A/B
; Converts value into 32-bit number
; Result = DX:AX

PUBLIC FPIDiv32
FPIDiv32 PROC FAR
 ARG B: DWORD, A: DWORD
;                    PUSH   BP
;                    MOV    BP,SP
;                    MOV    EAX,[A]
;                    MOV    EBX,[B]
;                    OR     EBX,EBX
;                    JZ     @FPD3Zero
;                    IDIV   EBX
;        @FPD3Store: LES    DI,C
;                    MOV    ES:[DI],EAX
;                    LES    DI,M
;                    MOV    ES:[DI],EDX
;                    POP    BP
;                    RET    20
;        @FPD3Zero:  XOR    EAX,EAX
;                    JMP    @FPD3Store
ENDP

; Integer part of 16-bit number A
; Result = AX

PUBLIC FPIInt
FPIInt PROC FAR
 ARG A: DWORD
                    ENTER  0,0
                    MOV    AX,WORD PTR [A]
                    AND    AX,0FF00H
                    LEAVE
                    RET    4
ENDP

; Fractional part of 16-bit number A
; Result = AX

PUBLIC FPIFrac
FPIFrac PROC FAR
 ARG A: DWORD
                    ENTER  0,0
                    MOV    AX,WORD PTR [A]
                    AND    AX,0FFH
                    LEAVE
                    RET    4
ENDP

; Integer part of 16-bit number A
; Converts result into 8-bit number
; Result = AX

PUBLIC FPITrunc
FPITrunc PROC FAR
 ARG A: DWORD
                    ENTER  0,0
                    MOV    AX,WORD PTR [A]
                    SHR    AX,8
                    LEAVE
                    RET    4
ENDP

; Opposite value of 16-bit number A
; Computes binary compound of a number A
; Result = AX

PUBLIC FPINeg
FPINeg PROC FAR
 ARG A: DWORD
                    ENTER  0,0
                    MOV    AX,WORD PTR [A]
                    NEG    AX
                    LEAVE
                    RET    4
ENDP

; Sinus(A), computes Sin(X) of 16-bit number A
; A = angle in radians
; Result = AX

PUBLIC FPISin
FPISin PROC FAR
 ARG A: DWORD
                    ENTER  0,0
                    MOV    AX,WORD PTR [A]
                    OR     AX,AX
                    JNS    @SISign
                    NEG    AX
                    ADD    AX,IPI
       @SISign:     CMP    AX,2*IPI
                    JNA    @SINAC
                    MOV    CX,AX           ; Normalize angle
                    MOV    BX,2*IPI
                    XOR    DX,DX
                    DIV    BX
                    MUL    BX
                    SUB    CX,AX
                    MOV    AX,CX
       @SINAC:      MOV    CX,AX
                    MOV    BX,[MLQrt]
                    IMUL   BX
                    MOV    AX,DX
                    SHR    AX,8
                    MOV    BX,AX
                    CMP    CX,IPI2
                    JNA    @SIDirPut
                    SUB    BX,1000H
                    CMP    CX,IPI
                    JB     @SIRevert
                    SUB    BX,1001H
                    CMP    CX,IPI+IPI2
                    JNA    @SIDirPut
                    SUB    BX,1000H
       @SIRevert:   MOV    DX,1000H
                    SUB    DX,BX
                    MOV    BX,DX
       @SIDirPut:   LEA    SI,SinusTable
                    SHL    BX,1
                    MOV    AX,DS:[SI+BX]
                    SHR    AX,8
                    CMP    CX,IPI
                    JNA    @SIStore
                    NEG    AX
       @SIStore:    LEAVE
                    RET    4
ENDP

; Cosinus(A), computes Cos(X) of 16-bit number A
; A = angle in radians
; Result = AX

PUBLIC FPICos
FPICos PROC FAR
 ARG A: DWORD
                    ENTER  0,0
                    MOV    AX,WORD PTR [A]
                    ADD    AX,IPI2
                    MOV    WORD PTR [A],AX
                    LEAVE
                    JMP    FPISin
ENDP

PUBLIC FPISqr
FPISqr PROC FAR
ENDP

PUBLIC FPISqrT
FPISqrT PROC FAR
 ARG A: DWORD
                    ENTER  0,0
		    MOV    AX,100H                ; Starting iteration = 1
                    MOV    CX,WORD PTR [A]                ; ECX=A
        @ISqrtIter: MOV    BX,AX                  ; EBX=B=Iteration result
                    MOV    DX,CX
                    MOV    AX,CX
                    SHR    DX,8
                    SHL    AX,8                   ; A/B
                    IDIV   BX
                    ADD    AX,BX                  ; B=B+A/B
                    SAR    AX,1                   ; B=1/2*(B+A/B)
                    MOV    DX,AX
                    SUB    DX,BX
                    JNS    @ICmpSqrt              ; Absolute value
                    NEG    DX
        @ICmpSqrt:  CMP    DX,ISqrtLimit          ; X(i+1)-Xi<SqrtLimit?
                    JA     @ISqrtIter
                    LEAVE
                    RET    4
ENDP

PUBLIC FPIExp
FPIExp PROC FAR
ENDP

PUBLIC FPILn
FPILn PROC FAR
ENDP

END
