{
 MIT - Mouse view.
 Copyright (C) 1994-2009  Henrich Fukna <fuky@azet.sk>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
}
unit MitView;

interface
uses Drivers, Objects, Views;

type
{TMouseView}
  PMouseView = ^TMouseView;              { Mouse View }
  TMouseView = object(TView)
    ButtonCount: Byte;                   { number of buttons }
    PressButtons: Byte;                  { pressed buttons }
    isDragging: Boolean;                 { dragging status }
    constructor Init(Bounds: TRect; aButtonCount: Byte);
                                         { Initialize view }
    procedure Draw; virtual;             { Draw view }
    procedure HandleEvent(var Event: TEvent); virtual;
                                         { Handle event }
    function DataSize: Word; virtual;    { Getting size of view data }
    procedure SetData(var Rec); virtual; { Setting view data }
  end;

implementation

{*
 * Initialize view.
 *
 * @param Bounds view size
 * @param aButtonCount number of buttons
 *}
constructor TMouseView.Init;
begin
  inherited Init(Bounds);                { call parrent }
  DragMode:= dmDragMove or dmLimitAll;
  ButtonCount:= aButtonCount;
  PressButtons:= 0;
  isDragging:= False;
end;

{*
 * Draw view.
 *}
procedure TMouseView.Draw;
var
  I, J: Integer;
  B: TDrawBuffer;                        { draw buffer }
  DragText: String;                      { drag mode text }
  Color: Byte;                           { frame color }
  HalfSize: Integer;                     { view half size }
  ButtonSize: Integer;                   { size of buttons }
  Button1stAddSize: Integer;             { rest size of buttons }
                                         { added for 1st button }
begin
  HalfSize:= Size.X div 2;
  ButtonSize:= (Size.X-ButtonCount-1) div ButtonCount;
  Button1stAddSize:= (Size.X-ButtonCount-1) mod ButtonCount;

  if isDragging then begin               { drag mode }
    Color:= 2;                           { enabled }
    DragText:= 'Drag mode: on';
  end else begin
    Color:= 10;                          { disabled }
    DragText:= 'Drag mode: off';
  end;

  MoveChar(B, #32, 7, Size.X);           { top line }
  MoveChar(B[HalfSize], #186, 9, 1);
  WriteLine(0, 0, Size.X, 1, B);

  MoveChar(B, #205, 9, Size.X);          { frame top line }
  MoveChar(B[HalfSize], #202, 9, 1);
  MoveChar(B[0], #201, Color, 1);
  MoveChar(B[Size.X-1], #187, Color, 1);
  WriteLine(0, 1, Size.X, 1, B);

  for I:= 2 to 5 do begin                { buttons lines }
    MoveChar(B, #32, 7, Size.X);
    MoveChar(B[0], #186, Color, 1);
    MoveChar(B[Size.X-1], #186, Color, 1);
    for J:= 1 to ButtonCount-1 do        { compute buttons separator }
      MoveChar(B[Button1stAddSize + (J * (ButtonSize+1))], #179, 15, 1);

    if (PressButtons and mbLeftButton) <> 0 then
      MoveChar(B[1], #177, 12, Button1stAddSize + ButtonSize);
                                         { selected 1st button }

    if (PressButtons and mbRightButton) <> 0 then
      MoveChar(B[Button1stAddSize + ((ButtonCount-1) * (ButtonSize+1)) + 1], #177, 12, ButtonSize);
                                         { selected last button }

    if (PressButtons and mbCenterButton) <> 0 then
      MoveChar(B[Button1stAddSize + (ButtonSize+1) + 1], #177, 12, ButtonSize);
                                         { selected middle button (2nd for 4+ buttons) }

    if (PressButtons and mb4thButton) <> 0 then
      MoveChar(B[Button1stAddSize + (2 * (ButtonSize+1)) + 1], #177, 12, ButtonSize);
                                         { selected 2nd middle button (for 4+ buttons) }
    WriteLine(0, I, Size.X, 1, B);
  end;

  MoveChar(B, #196, 15, Size.X);         { buttons bottom line }
  MoveChar(B[0], #186, Color, 1);
  MoveChar(B[Size.X-1], #186, Color, 1);
  for J:= 1 to ButtonCount-1 do          { compute buttons separator }
    MoveChar(B[Button1stAddSize + (J * (ButtonSize+1))], #193, 15, 1);
  WriteLine(0, 6, Size.X, 1, B);

  for I:= 7 to Size.Y-4 do begin         { frame lines }
    MoveChar(B, #32, 7, Size.X);
    MoveChar(B[0], #186, Color, 1);
    MoveChar(B[Size.X-1], #186, Color, 1);
    WriteLine(0, I, Size.X, 1, B);
  end;
  MoveChar(B, #32, 7, Size.X);           { drag mode text line }
  MoveChar(B[0], #186, Color, 1);
  MoveChar(B[Size.X-1], #186, Color, 1);
  MoveStr(B[4], DragText, 15);
  WriteLine(0, Size.Y-3, Size.X, 1, B);

  MoveChar(B, #32, 7, Size.X);           { double click text line }
  MoveChar(B[0], #186, Color, 1);
  MoveChar(B[Size.X-1], #186, Color, 1);
  MoveStr(B[3], 'Double-click me!', 15);
  WriteLine(0, Size.Y-2, Size.X, 1, B);

  MoveChar(B, #205, Color, Size.X);      { frame bottom line }
  MoveChar(B[0], #200, Color, 1);
  MoveChar(B[Size.X-1], #188, Color, 1);
  WriteLine(0, Size.Y-1, Size.X, 1, B);
end;

{*
 * Handle events.
 *
 * @param Event application events
 *}
procedure TMouseView.HandleEvent;
var
  R: TRect;
  Min, Max: TPoint;
begin
  inherited HandleEvent(Event);          { call parrent }
  if Event.What = evMouseDown then
    if Event.Double then
      isDragging:= not isDragging        { double-click changes drag mode }
    else
      if isDragging then begin           { when drag mode ebabled }
        PressButtons:= Event.Buttons;    { set pressed buttons }
        DrawView;                        { redraw }
        Owner^.GetExtent(R);             { get owner area }
        R.Grow(-1, -1);                  { decrease 1 }
        SizeLimits(Min, Max);            { self limits }
        DragView(Event, DragMode, R, Min, Max); { drag it on area }
        Event.What:= evMouseMove;        { change to mouse move event }
        Event.Where:= MouseWhere;        { change to new posittion }
        PressButtons:= 0;                { clear pressed buttons }
        DrawView;                        { redraw }
      end;
end;

{*
 * Gets size of view data.
 *
 * @return size of view data
 *}
function TMouseView.DataSize;
begin
  DataSize:= SizeOf(PressButtons);       { compute data size }
end;

{*
 * Sets view data.
 *
 * @param Rec view data record
 *}
procedure TMouseView.SetData;
begin
  PressButtons:= Byte(Rec);              { set view data }
  DrawView;                              { redraw }
end;

{Initialition unit}

end.