{
 MIT - gathering information about mouse.
 Copyright (C) 1994-2009  Henrich Fukna <fuky@azet.sk>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
}
unit MitInfo;

interface
uses Objects;

type

{ Version 6.0+ }
  PSensitivity = ^TSensitivity;
  TSensitivity = record
    HorizontalSpeed: Word;
    VerticalSpeed: Word;
    DoubleThresholdSpeed: Word;
  end;

  PAltIntr = ^TAltIntr;
  TAltIntr = record
    Mask: Word;
    Intr: Pointer;
    Next: PAltIntr;
  end;

  PMouse600 = ^TMouse600;
  TMouse600 = record
    StorageReq: Word;
    PSensitivity: PSensitivity;
    DisplayPageNumber: Word;
    Language: Word;
    PAltIntr: PAltIntr;
  end;

{ Version 6.26+ }
  PVersion = ^TVersion;
  TVersion = record
    Version: Word;
    Interupt: Byte;
    Typ: Byte;
  end;

  PGenInfo = ^TGenInfo;
  TGenInfo = record
    LoadedAs: Byte;
    Newer: Byte;
    CursorType: Byte;
    IntRate: Byte;
    CountDrivers: Byte;
    CursorLockFlag: Word;
    CodeActiveFlag: Word;
    DriverBusyFlag: Word;
  end;

  PVirtCoor = ^TVirtCoor;
  TVirtCoor = record
    Disabled: Word;
    CoorX: Word;
    CoorY: Word;
  end;

  PMouse626 = ^TMouse626;
  TMouse626 = record
    PVersion: PVersion;
    PGenInfo: PGenInfo;
    PVirtCoor: PVirtCoor;
  end;

{ Version 7.0+ }
  PVideoModes = ^TVideoModes;
  TVideoModes = record
    VideoMode: Word;
    Description: PChar;
    Next: PVideoModes;
  end;

  TAccProfileNameField = array[0..15] of Char;
  TAccProfileThresholdField = array[0..31] of Byte;
  TAccProfileFactorField = array[0..31] of Byte;

  TAccProfileName = array[1..4] of TAccProfileNameField;
  TAccProfileThreshold = array[1..4] of TAccProfileThresholdField;
  TAccProfileFactor = array[1..4] of TAccProfileFactorField;

  PAccProfile = ^TAccProfile;
  TAccProfile = record
    Dummy: Word;
    AccProfileLength: array[1..4] of Byte;
    AccProfileThreshold: TAccProfileThreshold;
    AccProfileFactor: TAccProfileFactor;
    AccProfileName: TAccProfileName;
  end;

  PAccProfiles = ^TAccProfiles;
  TAccProfiles = record
    CurrentActive: Word;
    PAccProfile: PAccProfile;
  end;

  PMouse700 = ^TMouse700;
  TMouse700 = record
    PVideoModes: PVideoModes;
    PAccProfiles: PAccProfiles;
  end;

{ Version 7.01+ }
  PMasks = ^TMasks;
  TMasks = record
    ScreenMask: Word;
    CursorMask: Word;
  end;

  PMouse701 = ^TMouse701;
  TMouse701 = record
    PMasks: PMasks;
  end;

{ Version 7.02+ }
  PHotSpot = ^THotSpot;
  THotSpot = record
    HotSpotCol: Word;
    HotSpotRow: Word;
  end;

  PMouse702 = ^TMouse702;
  TMouse702 = record
    PHotSpot: PHotSpot;
  end;

{ Version 7.04+ }
  PBallPoint = ^TBallPoint;
  TBallPoint = record
    Buttons: Word;
    Angle: Word;
    Primary: Byte;
    Secondary: Byte;
  end;

  PMouse704 = ^TMouse704;
  TMouse704 = record
    PBallPoint: PBallPoint;
  end;

{ Version 7.05+ }
  PMinMaxVirtCoor = ^TMinMaxVirtCoor;
  TMinMaxVirtCoor = record
    MinCoorX: Word;
    MinCoorY: Word;
    MaxCoorX: Word;
    MaxCoorY: Word;
  end;

  PActiveAdvFncs = ^TActiveAdvFncs;
  TActiveAdvFncs = record
    SupportedFnc: array[1..16] of Boolean;
  end;

  PAccProfileData = ^TAccProfileData;
  TAccProfileData = record
    Size: Word;
  end;

  PMouse705 = ^TMouse705;
  TMouse705 = record
    PMinMaxVirtCoor: PMinMaxVirtCoor;
    PActiveAdvFncs: PActiveAdvFncs;
    PAccProfileData: PAccProfileData;
  end;

{ Version 8.0+ }
  PMouse800 = ^TMouse800;
  TMouse800 = record
    IniFile: PChar;
  end;

{ Version 8.1+ }
  PAccProfileName = ^TAccProfileName;

  PAccProfileNames = ^TAccProfileNames;
  TAccProfileNames = record
    PAccProfileName: PAccProfileName;
  end;

  PLcdPointer = ^TLcdPointer;
  TLcdPointer = record
    Style: Byte;
    Size: Byte;
    Threshold: Byte;
    ActiveFlag: Byte;
    Delay: Word;
  end;

  PMouse810 = ^TMouse810;
  TMouse810 = record
    PAccProfileNames: PAccProfileNames;
    PLcdPointer: PLcdPointer;
  end;

{ Cute mouse sprecific }
  PCuteMouse = ^TCuteMouse;
  TCuteMouse = record
    WheelSupport: Boolean;
  end;

{ TMouseInfo }
  TMouseInfo = record
    Version: Word;
    Copyright: PChar;
    PMouse600: PMouse600;
    PMouse626: PMouse626;
    PMouse700: PMouse700;
    PMouse701: PMouse701;
    PMouse702: PMouse702;
    PMouse704: PMouse704;
    PMouse705: PMouse705;
    PMouse800: PMouse800;
    PMouse810: PMouse810;
    PCuteMouse: PCuteMouse;
  end;

procedure ReadMotionCounters(var P: TPoint);
procedure ReadMouseInfo(var MouseInfo: TMouseInfo);


implementation
uses Dos;

var
  Mouse600: TMouse600;
    Sensitivity: TSensitivity;
  Mouse626: TMouse626;
    Version: TVersion;
    GenInfo: TGenInfo;
    VirtCoor: TVirtCoor;
  Mouse700: TMouse700;
    VideoModes: TVideoModes;
    AccProfiles: TAccProfiles;
  Mouse701: TMouse701;
    Masks: TMasks;
  Mouse702: TMouse702;
    HotSpot: THotSpot;
  Mouse704: TMouse704;
    BallPoint: TBallPoint;
  Mouse705: TMouse705;
    MinMaxVirtCoor: TMinMaxVirtCoor;
    ActiveAdvFncs: TActiveAdvFncs;
    AccProfileData: TAccProfileData;
  Mouse800: TMouse800;
  Mouse810: TMouse810;
    LcdPointer: TLcdPointer;
    AccProfileNames: TAccProfileNames;
  CuteMouse: TCuteMouse;

const
  AllFncsSupported: TActiveAdvFncs = (
    SupportedFnc: (
      True, True, True, True, True, True, True, True,
      True, True, True, True, True, True, True, True)
  );

{*****************************************************************}

{*
 * Read motion counters (mickey)
 *
 * @param P point structure
 *}
procedure ReadMotionCounters(var P: TPoint);
var
  Reg: Registers;
begin
  Reg.Ax:= $0B;
  Intr($33, Reg);
  P.X:= Reg.Cx;
  P.Y:= Reg.Dx;
end;

{*****************************************************************}

{*
 * Gets mouse driver copyright string pointer.
 *
 * @return pointer to copyright string
 *}
function GetCopyright: PChar;
var
  Reg: Registers;
begin
  Reg.Ax:= $4D;
  Reg.Es:= 0;
  Reg.Di:= 0;
  Intr($33, Reg);
  GetCopyright:= Ptr(Reg.Es, Reg.Di);
end;

{*
 * Gets mouse driver version.
 *
 * @return version
 *}
function GetVersion: Word;
var
  Reg: Registers;
  P: ^Word;
begin
  Reg.Ax:= $6D;
  Reg.Es:= 0;
  Reg.Di:= 0;
  Intr($33, Reg);
  P:= Ptr(Reg.Es, Reg.Di);
  if P <> nil then
    GetVersion:= Swap(P^)
  else
    GetVersion:= 0;
end;

{*
 * Gets driver storage requirements.
 *
 * @return size of buffer needed to store driver state
 *}
function GetStorageReq: Word;
var
  Reg: Registers;
begin
  Reg.Ax:= $15;
  Reg.Bx:= 0;                            { set default value }
  Intr($33, Reg);
  GetStorageReq:= Reg.Bx;
end;

{*
 * Gets user alternative interrupt vector.
 *
 * @return user alternative interrupt vector
 *}
function GetUserAlternateIntr: PAltIntr;
var
  I: Word;
  Reg: Registers;
  Current, Previous, First: PAltIntr;
begin
  Current:= nil;
  Previous:= nil;
  First:= nil;

  for I:= 0 to 255 do begin
    Reg.Cx:= I;
    Reg.Ax:= $19;
    Reg.Bx:= 0;                          { set default values }
    Reg.Dx:= 0;
    Intr($33, Reg);
    if (Reg.Cx <> 0) then begin
      New(Current);
      Current^.Mask:= Reg.Cx;
      Current^.Intr:= Ptr(Reg.Bx, Reg.Dx);
      Current^.Next:= nil;
      if (Previous <> nil) then Previous^.Next:= Current;
      if (First = nil) then First:= Current;
      Previous:= Current;
    end;
  end;

  GetUserAlternateIntr:= First;
end;

{*
 * Gets mouse sensitivity.
 *
 * @return mouse sensitivity
 *}
function GetSensitivity: PSensitivity;
var
  Reg: Registers;
begin
  Reg.Ax:= $1B;
  Reg.Bx:= 0;                            { set default values }
  Reg.Cx:= 0;
  Reg.Dx:= 0;
  Intr($33, Reg);
  with Sensitivity do begin
    HorizontalSpeed:= Reg.Bx;
    VerticalSpeed:= Reg.Cx;
    DoubleThresholdSpeed:= Reg.Dx;
  end;
  GetSensitivity:= @Sensitivity;
end;

{*
 * Gets display page number.
 *
 * @return display page number
 *}
function GetDisplayPageNumber: Word;
var
  Reg: Registers;
begin
  Reg.Ax:= $1E;
  Reg.Bx:= 0;                            { set default value }
  Intr($33, Reg);
  GetDisplayPageNumber:= Reg.Bx;
end;

{*
 * Gets language code for messages.
 *
 * @return language code
 *}
function GetLanguage: Word;
var
  Reg: Registers;
begin
  Reg.Ax:= $23;
  Reg.Bx:= 0;                            { set default value }
  Intr($33, Reg);
  GetLanguage:= Reg.Bx;
end;

{*
 * Gets software version, mouse type and irq.
 *
 * @return software version, mouse type and irq
 *}
function QueryVersion: PVersion;
var
  Reg: Registers;
begin
  Reg.Ax:= $24;
  Reg.Bx:= 0;
  Intr($33, Reg);
  if (Reg.Bx <> 0) then begin
    with Version do begin
      Version:= Reg.Bx;
      Interupt:= Reg.Cl;
      Typ:= Reg.Ch;
    end;
    QueryVersion:= @Version;
  end else begin
    QueryVersion:= nil;
  end;
end;

{*
 * Gets general driver information.
 *
 * @return general driver infoemation
 *}
function GetGeneralInfo: PGenInfo;
var
  Reg:Registers;
begin
  Reg.Ax:= $25;
  Reg.Bx:= 0;
  Reg.Cx:= 0;
  Reg.Dx:= 0;
  Intr($33, Reg);
  with GenInfo do begin
    LoadedAs:= (Reg.Ax and $8000) shr 15;
    Newer:= (Reg.Ax and $4000) shr 14;
    CursorType:= (Reg.Ax and $3000) shr 12;
    IntRate:= (Reg.Ax and $0F00) shr 8;
    if (Newer <> 0) then
      CountDrivers:= (Reg.Ax and $00FF);
    CursorLockFlag:= Reg.Bx;
    CodeActiveFlag:= Reg.Cx;
    DriverBusyFlag:= Reg.Dx;
  end;
  GetGeneralInfo:= @GenInfo;
end;

{*
 * Gets maximum virtual coordinates.
 *
 * @return maximum virtual coordinates
 *}
function GetVirtualCoor: PVirtCoor;
var
  Reg: Registers;
begin
  Reg.Ax:= $26;
  Reg.Bx:= 0;                            { set default value }
  Intr($33, Reg);
  with VirtCoor do begin
    Disabled:= Reg.Bx;
    CoorX:= Reg.Cx;
    CoorY:= Reg.Dx;
  end;
  GetVirtualCoor:= @VirtCoor;
end;

{*
 * Get screen/cursor masks and mickey counts.
 *
 * @return screen/cursor and mickey counts
 *}
function GetMasks: PMasks;
var
  Reg: Registers;
begin
  Reg.Ax:= $27;
  Intr($33, Reg);
  with Masks do begin
    ScreenMask:= Reg.Ax;
    CursorMask:= Reg.Bx;
  end;
  GetMasks:= @Masks;
end;

{*
 * Gets video modes.
 *
 * @return enumerate video modes
 *}
function GetVideoModes: PVideoModes;
var
  Reg: Registers;
  Current, Previous, First: PVideoModes;
begin
  Current:= nil;
  Previous:= nil;
  First:= nil;

  Reg.Cx:= 0;
  repeat
    Reg.Ax:= $29;
    Reg.Dx:= 0;                          { set default values }
    Reg.Ds:= 0;
    Intr($33, Reg);

    if (Reg.Cx <> 0) then begin
      New(Current);
      Current^.VideoMode:= Reg.Cx;
      Current^.Description:= Ptr(Reg.Ds, Reg.Dx);
      Current^.Next:= nil;
      if (Previous <> nil) then Previous^.Next:= Current;
      if (First = nil) then First:= Current;
      Previous:= Current;
    end;
  until Reg.Cx = 0;

  GetVideoModes:= First;
end;

{*
 * Gets cursor hot spot.
 *
 * @return cursor hot spot
 *}
function GetHotSpot: PHotSpot;
var
  Reg: Registers;
begin
  Reg.Ax:= $2A;
  Intr($33, Reg);
  with HotSpot do begin
    HotSpotCol:= Reg.Bx;
    HotSpotRow:= Reg.Cx;
  end;
  GetHotSpot:= @HotSpot;
end;

{*
 * Gets acceleration profiles.
 *
 * @return acceleration profiles
 *}
function GetAccProfiles: PAccProfiles;
var
  Reg: Registers;
begin
  Reg.Ax:= $2C;
  Intr($33, Reg);
  if (Reg.Ax = 0) then begin
    with AccProfiles do begin
      CurrentActive:= Reg.Bx;
      PAccProfile:= Ptr(Reg.Es, Reg.Di);
    end;
    GetAccProfiles:= @AccProfiles;
  end else
    GetAccProfiles:= nil;
end;

{*
 * Gets acceleration profile names.
 *
 * @return acceleration profile names
 *}
function GetAccProfileNames: PAccProfileNames;
var
  Reg: Registers;
begin
  Reg.Ax:= $2E;
  Reg.Bl:= $FF;
  Reg.Es:= 0;
  Reg.Si:= 0;
  Intr($33, Reg);
  if (Reg.Ax = 0) then begin
    with AccProfileNames do begin
      PAccProfileName:= Ptr(Reg.Es, Reg.Si);
    end;
    GetAccProfileNames:= @AccProfileNames;
  end else
    GetAccProfileNames:= nil;
end;

{*
 * Get BallPoint information.
 *
 * @return ballpoint information
 *}
function GetBallPoint: PBallPoint;
var
  Reg: Registers;
begin
  Reg.Ax:= $30;
  Reg.Cx:= 0;
  Intr($33, Reg);
  if (Reg.Ax <> $FFFF) then begin
    with BallPoint do begin
      Buttons:= Reg.Ax;
      Angle:= Reg.Bx;
      Primary:= Reg.Ch;
      Secondary:= Reg.Cl;
    end;
    GetBallPoint:= @BallPoint;
  end else
    GetBallPoint:= nil;
end;

{*
 * Gets current minimum/maximum virtual coordinates.
 *
 * @return current min/max virtual coordinates
 *}
function GetMinMaxVirtCoor: PMinMaxVirtCoor;
var
  Reg: Registers;
begin
  Reg.Ax:= $31;
  Intr($33, Reg);
  with MinMaxVirtCoor do begin
    MinCoorX:= Reg.Ax;
    MinCoorY:= Reg.Bx;
    MaxCoorX:= Reg.Cx;
    MaxCoorY:= Reg.Dx;
  end;
  GetMinMaxVirtCoor:= @MinMaxVirtCoor;
end;

{*
 * Gets active advanced functions.
 *
 * @return active advanced functions
 *}
function GetActiveAdvFncs: PActiveAdvFncs;
var
  Reg: Registers;
begin
  Reg.Ax:= $32;
  Intr($33, Reg);
  if (Reg.Ax = $32) then                 { we don't detect function }
    Reg.Ax:= $FFFF;                      { so we set like older versions }
  with ActiveAdvFncs do begin
    SupportedFnc[1]:= (Reg.Ax and $8000) <> 0; { get general driver infoemarion }
    SupportedFnc[2]:= (Reg.Ax and $4000) <> 0; { get maximum virtual coordinates }
    SupportedFnc[3]:= (Reg.Ax and $2000) <> 0; { get screen/cursor masks and mickey counts }
    SupportedFnc[4]:= (Reg.Ax and $1000) <> 0;
    SupportedFnc[5]:= (Reg.Ax and $0800) <> 0; { enumerate video modes }
    SupportedFnc[6]:= (Reg.Ax and $0400) <> 0; { get cursor hot spot }
    SupportedFnc[7]:= (Reg.Ax and $0200) <> 0;
    SupportedFnc[8]:= (Reg.Ax and $0100) <> 0; { get acceleration profiles }
    SupportedFnc[9]:= (Reg.Ax and $0080) <> 0;
    SupportedFnc[10]:= (Reg.Ax and $0040) <> 0; { get acceleration profile names }
    SupportedFnc[11]:= (Reg.Ax and $0020) <> 0;
    SupportedFnc[12]:= (Reg.Ax and $0010) <> 0; { get ballpoint information }
    SupportedFnc[13]:= (Reg.Ax and $0008) <> 0; { get current min/max virtual coordinates }
    SupportedFnc[14]:= (Reg.Ax and $0004) <> 0; { get active advanced functions }
    SupportedFnc[15]:= (Reg.Ax and $0002) <> 0; { get switch settings and acceleration profile data }
    SupportedFnc[16]:= (Reg.Ax and $0001) <> 0; { get initialization file }
  end;
  GetActiveAdvFncs:= @ActiveAdvFncs;
end;

{*
 * Gets acceleration profile data.
 *
 * @return acceleration profile data
 *}
function GetAccProfileData: PAccProfileData;
var
  Reg: Registers;
begin
  Reg.Ax:= $33;
  Reg.Cx:= 0;
  Intr($33, Reg);
  if (Reg.Ax = 0) then begin
    with AccProfileData do begin
      Size:= Reg.Cx;
    end;
    GetAccProfileData:= @AccProfileData;
  end else
    GetAccProfileData:= nil;
end;

{*
 * Gets initialization file.
 *
 * @return initialization file
 *}
function GetIniFile: PChar;
var
  Reg: Registers;
begin
  Reg.Ax:= $34;
  Reg.Dx:= 0;                            { set default values }
  Reg.Es:= 0;
  Intr($33, Reg);
  GetIniFile:= Ptr(Reg.Es, Reg.Dx);
end;

{*
 * Gets LCD screen large pointer support.
 *
 * @return LCD screen large pointer support
 *}
function GetLcdPointer: PLcdPointer;
var
  Reg: Registers;
begin
  Reg.Ax:= $35;
  Reg.Bx:= $FFFF;
  Intr($33, Reg);

  if (Reg.Ax = 0) then begin
    with LcdPointer do begin
      Style:= Reg.Bh;
      Size:= Reg.Bl;
      Threshold:= Reg.Ch;
      ActiveFlag:= Reg.Cl;
      Delay:= Reg.Dx;
    end;
    GetLcdPointer:= @LcdPointer;
  end else
    GetLcdPointer:= nil;
end;

{*
 * Gets mouse wheel support.
 * (Only cutemouse driver specific.)
 *
 * @return mouse wheel support
 *}
function GetMouseWheelSupport: PCuteMouse;
var
  Reg: Registers;
begin
  Reg.Ax:= $11;
  Intr($33, Reg);

  if (Reg.Ax = $574D) then begin
    with CuteMouse do begin
      WheelSupport:= (Reg.Cx and $0001) <> 0;
    end;
    GetMouseWheelSupport:= @CuteMouse;
  end else
    GetMouseWheelSupport:= nil;
end;

{*****************************************************************}

{*
 * Read mouse/mouse driver informations.
 *
 * @param MouseInfo mouse structure
 *}
procedure ReadMouseInfo(var MouseInfo: TMouseInfo);
var
  P: Word;
begin
  MouseInfo.Copyright:= GetCopyright;
  MouseInfo.Version:= GetVersion;

  { When we don't detected version, we try the another function. }
  if (MouseInfo.Version = 0) or (MouseInfo.Version >= $626) then begin
    { Version 6.26+ }
    Mouse626.PVersion:= QueryVersion;
    { When we detected version, we set into Version field. }
    if (Mouse626.PVersion <> nil) then
      MouseInfo.Version:= Mouse626.PVersion^.Version;
  end;

  { Initialize structures }
  MouseInfo.PMouse600:= nil;
  MouseInfo.PMouse626:= nil;
  MouseInfo.PMouse700:= nil;
  MouseInfo.PMouse701:= nil;
  MouseInfo.PMouse702:= nil;
  MouseInfo.PMouse704:= nil;
  MouseInfo.PMouse705:= nil;
  MouseInfo.PMouse800:= nil;
  MouseInfo.PMouse810:= nil;
  MouseInfo.PCuteMouse:= nil;

  { Version 6.0+ }
  if (MouseInfo.Version >= $600) then begin
    Mouse600.StorageReq:= GetStorageReq;
    Mouse600.PSensitivity:= GetSensitivity;
    Mouse600.DisplayPageNumber:= GetDisplayPageNumber;
    Mouse600.Language:= GetLanguage;
    Mouse600.PAltIntr:= GetUserAlternateIntr;
    MouseInfo.PMouse600:= @Mouse600;
  end;

  { When version is 7.05+ functions are supported by the mask. }
  { For older versions is supposed to functions are aviable. }
  if (MouseInfo.Version >= $705) then
    { Version 7.05+ }
    Mouse705.PActiveAdvFncs:= GetActiveAdvFncs
  else
    Mouse705.PActiveAdvFncs:= @AllFncsSupported;

  { Version 6.26+ }
  if (MouseInfo.Version >= $626) then begin
    if (Mouse705.PActiveAdvFncs^.SupportedFnc[1]) then
      Mouse626.PGenInfo:= GetGeneralInfo;
    if (Mouse705.PActiveAdvFncs^.SupportedFnc[2]) then
      Mouse626.PVirtCoor:= GetVirtualCoor;
    MouseInfo.PMouse626:= @Mouse626;
  end;

  { Version 7.0+ }
  if (MouseInfo.Version >= $700) then begin
    if (Mouse705.PActiveAdvFncs^.SupportedFnc[5]) then
      Mouse700.PVideoModes:= GetVideoModes;
    if (Mouse705.PActiveAdvFncs^.SupportedFnc[8]) then
      Mouse700.PAccProfiles:= GetAccProfiles;
    MouseInfo.PMouse700:= @Mouse700;
  end;

  { Version 7.01+ }
  if (MouseInfo.Version >= $701) then begin
    if (Mouse705.PActiveAdvFncs^.SupportedFnc[3]) then
      Mouse701.PMasks:= GetMasks;
    MouseInfo.PMouse701:= @Mouse701;
  end;

  { Version 7.02+ }
  if (MouseInfo.Version >= $702) then begin
    if (Mouse705.PActiveAdvFncs^.SupportedFnc[6]) then
      Mouse702.PHotSpot:= GetHotSpot;
    MouseInfo.PMouse702:= @Mouse702;
  end;

  { Version 7.04+ }
  if (MouseInfo.Version >= $704) then begin
    if (Mouse705.PActiveAdvFncs^.SupportedFnc[12]) then
      Mouse704.PBallPoint:= GetBallPoint;
    MouseInfo.PMouse704:= @Mouse704;
  end;

  { Version 7.05+ }
  if (MouseInfo.Version >= $705) then begin
    if (Mouse705.PActiveAdvFncs^.SupportedFnc[13]) then
      Mouse705.PMinMaxVirtCoor:= GetMinMaxVirtCoor;
    if (Mouse705.PActiveAdvFncs^.SupportedFnc[15]) then
      Mouse705.PAccProfileData:= GetAccProfileData;
    MouseInfo.PMouse705:= @Mouse705;
  end;

  { Version 8.0+ }
  if (MouseInfo.Version >= $800) then begin
    if (Mouse705.PActiveAdvFncs^.SupportedFnc[16]) then
      Mouse800.IniFile:= GetIniFile;
    MouseInfo.PMouse800:= @Mouse800;
  end;

  { Version 8.1+ }
  if (MouseInfo.Version >= $810) then begin
    if (Mouse705.PActiveAdvFncs^.SupportedFnc[10]) then
      Mouse810.PAccProfileNames:= GetAccProfileNames;
    Mouse810.PLcdPointer:= GetLcdPointer;
    MouseInfo.PMouse810:= @Mouse810;
  end;

  { CuteMouse specific }
  MouseInfo.PCuteMouse:= GetMouseWheelSupport;
end;

{Initialition unit}

end.
