{
 MIT - Mouse Information and Test for DOS.
 Copyright (C) 1994-2009  Henrich Fukna <fuky@azet.sk>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
}
program MIT;
uses App, Dialogs, Drivers, Menus, Objects, Strings, Views,
     Lib,
     MitText, MitInfo,
     MitShare, MitView, MitWin1, MitWin2, MitWin3, MitWin4, MitWin5;

const
  Copyright: String = 'Mouse Information and Test';
                                         { identity string }
  Version = '2.2.0';                     { version string constant }
  ExeName = 'MIT';                       { project name }

  MaxWindows = 5;                        { number of infos window }

type
  TLPoint = record                       { Point structure }
    X, Y: LongInt;                       { X, Y coordinate }
  end;

  TData = record                         { Application data }
    MouseCoor: TLPoint;                  { mouse coordinates }
    MouseMotion: TLPoint;                { mouse motions }
    MousePres: Byte;                     { mouse button status }
    MouseWheel: LongInt;                 { mouse wheel status }
  end;

  TInfoWindows = array[0..MaxWindows-1] of PDialog;
                                         { array of infos window type }

{TMitApp}
  TMitApp = object(TApplication)         { Application }
    Info: TMouseInfo;                    { mouse informations }
    CurrWin: Word;                       { current infos window index }
    InfoWin: TInfoWindows;               { array of infos window }
    SuppFncsWin: PSuppFncsWindow;        { supported functions window }
    AltIntrWin: PAltIntrWindow;          { alternate interrupt vector window }
    VideoModesWin: PVideoModesWindow;    { video modes }
    Data: TData;
    constructor Init;                    { Initialize application }
    function GetPalette: PPalette; virtual;
                                         { Sets application palette }
    procedure HandleEvent(var Event: TEvent); virtual;
                                         { Handle events }
    procedure InitMenuBar; virtual;      { Initialize menu }
    procedure InitStatusLine; virtual;   { Initialize status }
    procedure SetMotionDistance;         { Sets mouse motion distance }
  end;

{********** Application ******************************************}

{*
 * Initialize applicaion.
 *}
constructor TMitApp.Init;
var
  S, T: String;
  Rect: TRect;
begin
  inherited Init;                        { call parrent }

  if not(MouseEvents) then begin
    GetExtent(Rect);                     { mouse driver not pressent }
    Rect.A.X:= Rect.B.X - 48;            { show warning message }
    Rect.B.X:= Rect.B.X - 1;
    Rect.A.Y:= Rect.A.Y + 2;
    Rect.B.Y:= Rect.A.Y + 3;
    Insert(New(PColoredStaticText, Init(Rect,
      ^M + ^C + 'Mouse driver not installed !' + ^M, $0C)));
  end else begin
    ReadMouseInfo(Info);                 { read mouse driver informations }

    GetExtent(Rect);
    Rect.A.Y:= Rect.B.Y - 3;
    Rect.B.Y:= Rect.A.Y + 1;             { initialize and show coodinates text }
    S:= 'Current mouse coordinates ( X, Y ) = ( %2d, %2d )';
    Insert(New(PColoredParamText, Init(Rect, ^C + S, $0F, 2)));
    Rect.A.Y:= Rect.B.Y;
    Rect.B.Y:= Rect.A.Y + 1;             { initialize and show distance text }
    S:= '                Motion distance ( X, Y ) = ( %5s, %5s )';
    Insert(New(PColoredParamText, Init(Rect, ^C + S, $0F, 2)));

    Rect.Assign(4, 4, 25, 18);           { initialize and show mouse-view }
    Insert(New(PMouseView, Init(Rect, ButtonCount)));

    if (Info.PCuteMouse <> nil) and
        Info.PCutemouse^.WheelSupport
    then begin
      GetExtent(Rect);
      Rect.A.Y:= Rect.B.Y - 4;
      Rect.B.Y:= Rect.A.Y + 1;           { initialize and show wheel text }
      S:= '             Wheel change = %2d';
      Insert(New(PColoredParamText, Init(Rect, ^C + S, $0F, 1)));
    end;

    GetExtent(Rect);                     { initialize all infos windows }
    Rect.A.X:= Rect.B.X - 48;
    Rect.B.X:= Rect.B.X - 1;
    Rect.A.Y:= Rect.A.Y + 2;
    Rect.B.Y:= Rect.A.Y + 17;
    InfoWin[0]:= New(PInfoWindow1, Init(Rect, Info));
    InfoWin[1]:= New(PInfoWindow2, Init(Rect, Info));
    InfoWin[2]:= New(PInfoWindow3, Init(Rect, Info));
    InfoWin[3]:= New(PInfoWindow4, Init(Rect, Info));
    InfoWin[4]:= New(PInfoWindow5, Init(Rect, Info));
    CurrWin:= 0;                         { show first infos window }
    InsertBefore(InfoWin[CurrWin], Desktop);

    GetExtent(Rect);
    Rect.A.X:= Rect.B.X - 48;
    Rect.B.X:= Rect.B.X - 1;
    Rect.A.Y:= Rect.A.Y + 2;
    Rect.B.Y:= Rect.A.Y + 19;
    SuppFncsWin:= New(PSuppFncsWindow, Init(Rect, Info));
    AltIntrWin:= New(PAltIntrWindow, Init(Rect, Info));
    VideoModesWin:= New(PVideoModesWindow, Init(Rect, Info));

    with Data do begin
      MouseCoor.X:= MouseWhere.X;        { horizontal coordinate }
      MouseCoor.Y:= MouseWhere.Y;        { vertical coordinate }
      SetMotionDistance;                 { set distance texts }
      SetData(Data);                     { set application data }
    end;
  end;
end;

{*
 * Sets application palette.
 *
 * @return palette
 *}
function TMitApp.GetPalette: PPalette;
const
  CNewColor: Char = #$07;                { desktop color }
  P: array[apColor..apMonochrome] of TPalette = (
    CAppColor, CAppBlackWhite, CAppMonochrome);
begin
  Move(CNewColor, P[apColor, 1], SizeOf(CNewColor));
                                         { change desktop color for color modes }
  GetPalette:= @P[AppPalette];
end;

{*
 * Handle events.
 *
 * @param Event application events
 *}
procedure TMitApp.HandleEvent(var Event: TEvent);

  {*
   * Mouse move action.
   *
   * @param P point
   *}
  procedure MouseMove(P: TPoint);
  begin
    Data.MouseWheel:= 0;                 { reset wheel status }
    Data.MouseCoor.X:= P.X;              { horizontal coordinate }
    Data.MouseCoor.Y:= P.Y;              { vertical coordinate }
    SetMotionDistance;                   { set distance texts }
    SetData(Data);                       { set application data }
  end;

  {*
   * Mouse press / release button action.
   *
   * @param Buttons mouse buttons status
   *}
  procedure MousePress(Buttons: Byte);
  begin
    Data.MouseWheel:= 0;                 { reset wheel status }
    Data.MousePres:= Buttons;            { buttons status }
    SetMotionDistance;                   { set distance texts }
    SetData(Data);                       { set application data }
  end;

  {*
   * Mouse wheel action.
   *
   * @param Wheel wheel counter status
   *}
  procedure MouseWheel(Wheel: Integer);
  begin
    Data.MouseWheel:= Wheel;             { wheel status }
    SetMotionDistance;                   { set distance texts }
    SetData(Data);                       { set application data }
  end;

var
  I: Word;
begin
  inherited HandleEvent(Event);          { call parrent }
  case Event.What of
    evKeyDown:
      EndModal(cmQuit);                  { end application }

    evMouseMove:
      MouseMove(Event.Where);            { activate mouse move action }

    evMouseUp,
    evMouseDown:
      MousePress(Event.Buttons);         { activate mouse press/release action }

    evMouseWheel:
      MouseWheel(Event.Wheel);           { activate mouse wheel acion }

    evCommand:
      case Event.Command of
        cmNextWin:
          begin
            I:= (CurrWin+1) mod MaxWindows;
                                         { compute next window (cyclic) }
            InsertBefore(InfoWin[I], DeskTop);
                                         { show next window over current window }
            Delete(InfoWin[CurrWin]);    { remove current window }
            CurrWin:= I;                 { set new current index }
          end;
        cmPrevWin:
          begin
            if CurrWin > 0 then
              I:= (CurrWin-1) mod MaxWindows
            else
              I:= Maxwindows-1;
                                         { compute previous window (cyclic) }
            InsertBefore(InfoWin[I], DeskTop);
                                         { show next window over current window }
            Delete(InfoWin[CurrWin]);    { remove current window }
            CurrWin:= I;                 { set new current index }
          end;
        cmSuppFncs:
          begin
            ExecView(SuppFncsWin);
          end;
        cmAltIntr:
          begin
            ExecView(AltIntrWin);
          end;
        cmVideoModes:
          begin
            ExecView(VideoModesWin);
          end;
      else
        Exit;                            { otherwise exit handler }
      end;
    else
      Exit;                              { otherwise exit handler }
  end;
  ClearEvent(Event);                     { event handled - clear it }
end;

{*
 * Initialize menu.
 *}
procedure TMitApp.InitMenuBar;
var
  Rect: TRect;
begin
  GetExtent(Rect);                       { get desktop coordinates }
  Rect.B.Y:= Rect.A.Y + 1;               { insert text (status not used) }
  Insert(New(PColoredStaticText, Init(Rect,
    ^C + 'Mouse Info & Test 2.2.0 by Henrich Fukna, 24. November 2009', $0B)));
end;

{*
 * Initialize status.
 *}
procedure TMitApp.InitStatusLine;
var
  Rect: TRect;
begin
  GetExtent(Rect);                       { get desktop coordinates }
  Rect.A.Y:= Rect.B.Y - 1;               { insert text (status not used) }
  Insert(New(PColoredStaticText, Init(Rect,
    ^C + 'Press any key to exit...', $8E)));
end;

{*
 * Sets mouse motion distance.
 *}
procedure TMitApp.SetMotionDistance;
const
  S: array[0..4] of string[5] = ('left', 'right', 'up', 'down', 'stall');
                                         { texts for motion track }
var
  P: TPoint;                             { distance coodinates }
begin
  ReadMotionCounters(P);                 { read distance }

                                         { horizontal distance text }
  if P.X < 0 then Data.MouseMotion.X:= LongInt(@S[0])
  else if P.X > 0 then Data.MouseMotion.X:= LongInt(@S[1])
  else Data.MouseMotion.X:= LongInt(@S[4]);

                                         { vertical distance text }
  if P.Y < 0 then Data.MouseMotion.Y:= LongInt(@S[2])
  else if P.Y > 0 then Data.MouseMotion.Y:= LongInt(@S[3])
  else Data.MouseMotion.Y:= LongInt(@S[4]);
end;

{********** main module ******************************************}

{*
 * Prints the syntax.
 *}
procedure Usage;
begin
  Writeln('Usage: ', ExeName, ' [-h?L]');
  Writeln;
  Writeln('Options:');
  Writeln('  h, ?     display this help screen');
  Writeln('  L        display software licence');
  Halt(1);
end;

var
  I: Byte;
  Args: String;
  MitApp: TMitApp;

begin
  Write(Copyright, ' ', Version, ', ');
  Writeln('Copyright (C) 1994-2009 Henrich Fukna');
  LicenceLine(Copyright, ExeName);
  Writeln;

  I:= 1;
  while I <= ParamCount do begin
    Args:= ParamStr(I);
    if (Args[1] = '-') then begin
      Delete(Args, 1, 1);
      while Length(Args) > 0 do begin
        case Args[1] of
          '?',
          'h': Usage;
          'L':
            begin
              Licence;
              Halt(1);
            end;
          else begin
            OptArgErrMsg(ExeName, False);
            Halt(2);
          end;
        end;
        Delete(Args, 1, 1);
      end;
    end else begin
      OptArgErrMsg(ExeName, True);
      Halt(2);
    end;
    Inc(I);
  end;

  MitApp.Init;                           { initialize application }
  MitApp.Run;                            { run application }
  MitApp.Done;                           { destroy application }
end.