{
 Library - The library functions.
 Copyright (C) 2002-2003  Henrich Fukna <fuky@azet.sk>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
}
unit Lib;

interface

{ ******** PRINTS ******************* }

{*
 * Prints the licence.
 *}
procedure Licence;

{*
 * Prints the licence line notify.
 *
 * @param Name                  project name
 * @param ExeName               main module name
 *}
procedure LicenceLine(Name, ExeName: String);

{*
 * Prints option or argument error message.
 *
 * @param ExeName               main module name
 * @param OptArg                option/argument flag
 *}
procedure OptArgErrMsg(ExeName: String; OptArg: Boolean);

{ ******** COMPARES ***************** }

type
  PString = ^String;

{*
 * Compares two pascal style strings.
 *
 * @param Str1                  pointer to first string
 * @param Str2                  pointer to second string
 * @return true if strings matches, otherwise false
 *}
function StrComp(Str1, Str2: PString): Boolean;

{*
 * Compares filename with wildcard mask.
 *
 * @param Mask                  wildcard mask
 * @param Name                  filename
 * @return true if matches, otherwise false
 *}
function CmpName(Mask, Name: String): Boolean;

{ ******** CONVERSIONS ************** }

{*
 * Clean string to printable chars.
 *
 * @param Str                   string to be cleaned
 * @return cleaned string
 *}
function StrCleanUp(Str: String): String;

{*
 * Upper-cases string.
 *
 * @param Str                   input string
 * @return upper-case string
 *}
function StrUpr(Str: String): String;

{*
 * Converts hexadecimal to integer.
 *
 * @param S                     hexadecimal string
 * @return long-integer, or -1 if error
 *}
function Hex2Int(S: String): LongInt;

{*
 * Converts integer to hexadecimal.
 *
 * @param V                     integer to convert
 * @return hexadecimal string, or empty string if error occured
 *}
function Int2Hex(V: LongInt): String;

{ ******** PARSE PARAMETERS ********* }

{*
 * Get the parameters count.
 *
 * @param ArgS                  argument string
 * @param Separ                 separator char
 * @return number of parameters
 *}
function GetParseCount(ArgS: String; Separ: Char): Integer;

{*
 * Get the parameter.
 *
 * @param Index                 index of parameter
 * @param Args                  argument string
 * @param Separ                 separator char
 * @return parameter as pascal style string
 *}
function GetParseString(Index: Integer; ArgS: String; Separ: Char): String;

{ ******** INIFILE HANDLE *********** }

{*
 * Get integer value from a INI file.
 *
 * @param ApplicationName       section name in file
 * @param KeyName               property name in section
 * @param Default               default value
 * @param FileName              INI file name
 * @return integer value from INI or default value
 *}
function GetPrivateProfileInt(ApplicationName, KeyName: String;
  Default: Integer; FileName: String): Integer;

{*
 * Get string value from a INI file.
 *
 * @param ApplicationName       section name in file
 * @param KeyName               property name in section
 * @param Default               default value
 * @param ReturnedString        returned string value from INI or default
 * @param Size                  maximal size for the returned string
 * @param FileName              INI file name
 * @return size of returned string
 *}
function GetPrivateProfileString(ApplicationName, KeyName,
  Default: String; var ReturnedString: String;
  Size: Byte; FileName: String): Integer;

{*
 * Write the string value to a INI file.
 *
 * @param ApplicationName       section name in file
 * @param KeyName               property name in section
 * @param KeyStr                value
 * @param FileName              INI file name
 * @return write status flag
 *}
function WritePrivateProfileString(ApplicationName, KeyName,
  KeyStr: String; FileName: String): Boolean;

implementation

{ ******** PRINTS ******************* }

{*
 * Prints the licence.
 *}
procedure Licence;
begin
  Writeln('This program is free software; you can redistribute it and/or modify');
  Writeln('it under the terms of the GNU General Public License as published by');
  Writeln('the Free Software Foundation; either version 2 of the License, or');
  Writeln('(at your option) any later version.');
  Writeln;
  Writeln('This program is distributed in the hope that it will be useful,');
  Writeln('but WITHOUT ANY WARRANTY; without even the implied warranty of');
  Writeln('MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the');
  Writeln('GNU General Public License for more details.');
  Writeln;
  Writeln('You should have received a copy of the GNU General Public License');
  Writeln('along with this program; if not, write to the Free Software');
  Writeln('Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA');
end;

{*
 * Prints the licence line notify.
 *
 * @param Name                  project name
 * @param ExeName               main module name
 *}
procedure LicenceLine(Name, ExeName: String);
begin
  Writeln(Name, ' comes with ABSOLUTELY NO WARRANTY;');
  Writeln('for details type `', ExeName, ' -L''.');
end;

{*
 * Prints option or argument error message.
 *
 * @param ExeName               main module name
 * @param OptArg                option/argument flag
 *}
procedure OptArgErrMsg(ExeName: String; OptArg: Boolean);
begin
  Write('Invalid ');
  if OptArg then Write('option') else Write('argument');
  Writeln('.');
  Writeln('Type: `', ExeName, ' -h'' for help.');
end;

{ ******** COMPARES ***************** }

{*
 * Compares two pascal style strings.
 *
 * @param str1                  pointer to first string
 * @param str2                  pointer to second string
 * @return true if strings matches, otherwise false
 *}
function StrComp(Str1, Str2: PString): Boolean;
var
  Cnt: Byte;
  Result: Boolean;
begin
  Cnt:= Length(Str1^);
  Result:= (Cnt = Length(Str2^));
  while Result and (Cnt > 0) do begin
    Result:= (Str1^[Cnt] = Str2^[Cnt]);
    Dec(Cnt);
  end;
  StrComp:= Result;
end;

{*
 * Compares filename with wildcard mask.
 *
 * @param Mask                  wildcard mask
 * @param Name                  filename
 * @return true if matches, otherwise false
 *}
function CmpName;
var
  I, J: Integer;
begin
  I:= 1;
  J:= 1;
  while True do begin
    if Mask[I] = '*' then begin
      while (Mask[I] <> '.') and (I <= Length(Mask)) do Inc(I);
      while (Name[J] <> '.') and (J <= Length(Name)) do Inc(J);
    end;
    if I > Length(Mask) then begin
      CmpName:= J > Length(Name);
      Exit;
    end;
    if (J > Length(Name)) and (Mask[I] = '.') then begin
      Inc(I);
      Continue;
    end;
    if (UpCase(Mask[I]) = UpCase(Name[J])) or (Mask[I] = '?') and
       (J <= Length(Name)+1) then begin
         Inc(I);
         Inc(J);
    end else begin
      CmpName:= False;
      Exit;
    end;
  end;
end;
{ ******** CONVERSIONS ************** }

{*
 * Clean string to printable chars.
 *
 * @param Str                   string to be cleaned
 * @return cleaned string
 *}
function StrCleanUp(Str: String): String;
var
  I: Integer;
begin
  I:= 1;
  while (I <= Length(Str)) and (Str[I] in [#32..#127]) do
    Inc(I);

  Delete(Str, I, Length(Str)-I+1);
  StrCleanUp:= Str;
end;

{*
 * Upper-cases string.
 *
 * @param Str                   input string
 * @return upper-case string
 *}
function StrUpr(Str: String): String;
var
  I: Integer;
begin
  for I:= 1 to Length(Str) do
    Str[I]:= UpCase(Str[I]);

  StrUpr:= Str;
end;

{*
 * Converts hexadecimal to integer.
 *
 * @param S                     hexadecimal string
 * @return long-integer, or -1 if error
 *}
function Hex2Int(S: String): LongInt;
var
  I, Result: LongInt;
begin
  Result:= 0;
  for I:= 1 to Length(S) do begin
    case S[I] of
      '0'..'9': Result:= Result*16 + (Ord(S[I])-Ord('0'));
      'a'..'f': Result:= Result*16 + (Ord(S[I])-Ord('a')+10);
      'A'..'F': Result:= Result*16 + (Ord(S[I])-Ord('A')+10);
      else begin
        Hex2Int:= -1;
        Exit;
      end;
    end;
  end;
  Hex2Int:= Result;
end;

{*
 * Converts integer to hexadecimal.
 *
 * @param V                     integer to convert
 * @return hexadecimal string, or empty string if error occured
 *}
function Int2Hex(V: LongInt): String;
var
  M: LongInt;
  Result: String;
begin
  Result:= '';
  repeat
    M:= V mod 16;
    V:= V div 16;
    if M < 10 then
      Result:= Concat(Chr(Ord('0')+M), Result)
    else
      Result:= Concat(Chr(Ord('A')+M-10), Result);
  until V = 0;
  Int2Hex:= Result;
end;

{ ******** PARSE PARAMETERS ********* }

{*
 * Get the parameters count.
 *
 * @param ArgS                  argument string
 * @param Separ                 separator char
 * @return number of parameters
 *}
function GetParseCount(ArgS: String; Separ: Char): Integer;
var
  I, L, Result: Integer;
begin
  Result:= 0;
  I:= 1;
  L:= Length(ArgS);
  while I <= Length(ArgS) do begin
    while I <= Length(ArgS) do begin
      if ArgS[I] <> Separ then Break;
      Inc(I);
    end;
    if I <= Length(ArgS) then begin
      Inc(Result);
      while I <= Length(ArgS) do begin
        if ArgS[I] = Separ then Break;
        Inc(I);
      end;
    end;
  end;
  GetParseCount:= Result;
end;

{*
 * Get the parameter.
 *
 * @param Index                 index of parameter
 * @param Args                  argument string
 * @param Separ                 separator char
 * @return parameter as pascal style string
 *}
function GetParseString(Index: Integer; ArgS: String; Separ: Char): String;
var
  I, J, K: Integer;
  Result: String;
begin
  Result:= '';
  I:= GetParseCount(ArgS, Separ);
  if (Index > -1) and (Index < I) then begin
    I:= -1;
    J:= 1;
    K:= J;
    while (J <= Length(ArgS)) and (I < Index) do begin
      while J <= Length(ArgS) do begin
        if ArgS[J] <> Separ then Break;
        Inc(J);
      end;
      if J <= Length(ArgS) then begin
        K:= J;
        Inc(I);
        while J <= Length(ArgS) do begin
          if ArgS[J] = Separ then Break;
          Inc(J);
        end;
      end;
    end;
    Result:= Copy(ArgS, K, J-K);
  end;
  GetParseString:= Result;
end;

{ ******** INIFILE HANDLE *********** }

type
  BufferType = array[0..65527] of Char;

{*
 * Reads string from the buffer.
 *
 * @param B                     buffer
 * @param ptrB                  current pointer in buffer
 * @param sizeB                 size of buffer
 * @return string from a buffer or (empty)
 *}
function ReadStr(var B: BufferType; var ptrB, sizeB: Word): String;
var
  Ch, Ch1: Char;
  Result: String;
begin
  Result:= '';
  Ch1:= #0;
  while ptrB < sizeB do begin
    Ch:= B[ptrB];
    Inc(ptrB);
    Result:= Concat(Result, Ch);
    if (Ch = #10) and (Ch1 = #13) then Break;
    Ch1:= Ch;
  end;
  ReadStr:= Result;
end;

{*
 * Writes string to the buffer.
 *
 * @param S                     string to write
 * @param B                     buffer
 * @param ptrB                  current pointer in buffer
 * @param sizeB                 size of buffer
 *}
procedure WriteStr(S: String; var B: BufferType; var ptrB, sizeB: Word);
var
  I: Integer;
begin
  I:= 1;
  while (ptrB < sizeB) and (I <= Length(S)) do begin
    B[ptrB]:= S[I];
    Inc(ptrB);
    Inc(I);
  end;
end;

{*
 * Truncates string for whitespace(s).
 *
 * @param S                     string to trim
 * @return trimed string
 *}
function TrimStr(S: String): String;
var
  Result: String;
begin
  Result:= S;
  while (Length(Result) > 0) and (Result[1] in [#9, #10, #13, #32]) do
    Delete(Result, 1, 1);
  while (Length(Result) > 0) and (Result[Length(Result)] in [#9, #10, #13, #32]) do
    Delete(Result, Length(Result), 1);
  TrimStr:= Result;
end;

{*
 * Removes comments from string.
 *
 * @param S                     string
 * @return string witout comments
 *}
function RemoveComments(S: String): String;
var
  I, J: Integer;
  Result: String;
begin
  Result:= S;
  I:= Pos(#13#10, Result);
  if I = 0 then I:= Length(Result);

  J:= Pos('#', Result);
  if J <> 0 then Delete(Result, J, I-J);

  J:= Pos(';', Result);
  if J <> 0 then Delete(Result, J, I-J);
  RemoveComments:= Result;
end;

{*
 * Get integer value from a INI file.
 *
 * @param ApplicationName       section name in file
 * @param KeyName               property name in section
 * @param Default               default value
 * @param FileName              INI file name
 * @return integer value from INI or default value
 *}
function GetPrivateProfileInt;
var
  S: String;
  Code, V, Result: Integer;
begin
  Result:= Default;
  Code:= GetPrivateProfileString(ApplicationName, KeyName, '',
    S, 255, FileName);
  if Code <> 0 then begin
    Val(S, V, Code);
    if Code = 0 then Result:= V;
  end;
  GetPrivateProfileInt:= Result;
end;

{*
 * Get string value from a INI file.
 *
 * @param ApplicationName       section name in file
 * @param KeyName               property name in section
 * @param Default               default value
 * @param ReturnedString        returned string value from INI or default
 * @param Size                  maximal size for the returned string
 * @param FileName              INI file name
 * @return size of returned string
 *}
function GetPrivateProfileString;
var
  F: File;
  P: Byte;
  S, T: String;
  Buffer: ^BufferType;
  BufferPtr, BufferSize: Word;
begin
  ReturnedString:= Default;
  GetPrivateProfileString:= Length(Default);

  { Open the INI file }
  Assign(F, FileName);
  P:= FileMode;
  FileMode:= 0;
  {$I-}
  Reset(F, 1);
  {$I+}
  FileMode:= P;
  if IOResult = 0 then begin
    { Alocate buffer for the file }
    BufferPtr:= 0;
    BufferSize:= FileSize(F);
    if BufferSize > SizeOf(BufferType) then BufferSize:= SizeOf(BufferType);
    GetMem(Buffer, BufferSize);

    { Load INI file into the buffer }
    {$I-}
    BlockRead(F, Buffer^, BufferSize);
    {$I+}
    if IOResult = 0 then begin
      { Find section name (ApplicationName) }
      T:= ConCat('[', ApplicationName, ']');
      repeat
        S:= ReadStr(Buffer^, BufferPtr, BufferSize);
        if Length(S) = 0 then Break;
        S:= RemoveComments(S);
        S:= TrimStr(S);
      until S = T;

      if S = T then begin
        { Find property name (KeyName) }
        repeat
          S:= ReadStr(Buffer^, BufferPtr, BufferSize);
          if Length(S) = 0 then Break;
          S:= RemoveComments(S);
          S:= TrimStr(S);
          T:= GetParseString(1, S, '=');
          S:= GetParseString(0, S, '=');
        until (Copy(S, 1, 1) = '[') or (S = KeyName);

        if S = KeyName then begin
          { Truncate to maximal size of value }
          if Length(T) > Size then T[0]:= Chr(Size);
          { return the value }
          ReturnedString:= T;
          GetPrivateProfileString:= Length(T);
        end;
      end;
    end;
    FreeMem(Buffer, BufferSize);
    Close(F);
  end;
end;

{*
 * Write the string value to a INI file.
 *
 * @param ApplicationName       section name in file
 * @param KeyName               property name in section
 * @param KeyStr                value
 * @param FileName              INI file name
 * @return write status flag
 *}
function WritePrivateProfileString;
var
  F: File;
  P: Byte;
  S, T: String;
  WritePos: Word;
  CrApFlag: Boolean;

  Buffer: ^BufferType;
  BufferPtr, BufferSize: Word;
  TmpBuffer: ^BufferType;
  TmpBufferPtr, TmpBufferSize: Word;
begin
  WritePrivateProfileString:= False;

  { Open the INI file }
  Assign(F, FileName);
  {$I-}
  Reset(F, 1);
  {$I+}
  if IOResult <> 0 then begin
    {$I-}
    Rewrite(F, 1);
    {$I+}
    if IOResult <> 0 then Exit;

    { If INI file does not exists - create it }
    CrApFlag:= True;
    BufferSize:= 0;

  end else begin
    { If INI file exists }
    CrApFlag:= False;
    BufferSize:= FileSize(F);
  end;

  { Alocate buffer for the file }
  BufferPtr:= 0;
  if BufferSize > SizeOf(BufferType) then BufferSize:= SizeOf(BufferType);
  GetMem(Buffer, BufferSize);

  if not(CrApFlag) then begin
    { Load INI file into the buffer }
    {$I-}
    BlockRead(F, Buffer^, BufferSize);
    {$I+}
    if IOResult <> 0 then begin
      FreeMem(Buffer, BufferSize);
      Close(F);
      Exit;
    end;
  end;

  { Find section name (ApplicationName) }
  T:= ConCat('[', ApplicationName, ']');
  repeat
    S:= ReadStr(Buffer^, BufferPtr, BufferSize);
    if Length(S) = 0 then Break;
    S:= RemoveComments(S);
    S:= TrimStr(S);
  until S = T;

  WritePos:= BufferPtr;

  if S = T then
    { Find property name (KeyName) }
    repeat
      WritePos:= BufferPtr;

      S:= ReadStr(Buffer^, BufferPtr, BufferSize);
      if Length(S) = 0 then Break;
      S:= RemoveComments(S);
      S:= TrimStr(S);
      T:= GetParseString(1, S, '=');
      S:= GetParseString(0, S, '=');
    until (Copy(S, 1, 1) = '[') or (S = KeyName)
  else
    CrApFlag:= True;

  { Alocate temporary buffer }
  TmpBuffer:= Pointer(Buffer);
  TmpBufferPtr:= BufferPtr;
  TmpBufferSize:= BufferSize;
  Inc(BufferSize, Length(ApplicationName)+Length(KeyName)+Length(KeyStr)+7);
  if BufferSize > SizeOf(BufferType) then BufferSize:= SizeOf(BufferType);
  GetMem(Buffer, BufferSize);

  { Copy all the Data to write position }
  Move(TmpBuffer^, Buffer^, WritePos);
  BufferPtr:= WritePos;

  { Write section name (ApplicationName) }
  if CrApFlag then begin
    WriteStr('[', Buffer^, BufferPtr, BufferSize);
    WriteStr(ApplicationName, Buffer^, BufferPtr, BufferSize);
    WriteStr(']', Buffer^, BufferPtr, BufferSize);
    WriteStr(#13#10, Buffer^, BufferPtr, BufferSize);
  end;

  { Write property name (KeyName) }
  WriteStr(KeyName, Buffer^, BufferPtr, BufferSize);
  WriteStr('=', Buffer^, BufferPtr, BufferSize);
  WriteStr(KeyStr, Buffer^, BufferPtr, BufferSize);
  WriteStr(#13#10, Buffer^, BufferPtr, BufferSize);

  { Copy all the Data after write position }
  Move(TmpBuffer^[TmpBufferPtr], Buffer^[BufferPtr], TmpBufferSize-TmpBufferPtr);
  Inc(BufferPtr, TmpBufferSize-TmpBufferPtr);

  { Destroy temporary buffer }
  FreeMem(TmpBuffer, TmpBufferSize);

  { Save buffer to the INI file }
  {$I-}
  Seek(F, 0);
  BlockWrite(F, Buffer^, BufferPtr);
  if TmpBufferSize > BufferPtr then Truncate(F);
  {$I+}
  if IOResult = 0 then WritePrivateProfileString:= True;

  Close(F);
  FreeMem(Buffer, BufferSize);
end;

{Initialition unit}

end.