unit lm78ctrl;
{$DEFINE DLPortIO}

(*******************************************************)
(*   National Semiconductor LM78 and Winbond W83781D   *)
(*                     Control Unit                    *)
(*           NEW Support for Winbond W83782D           *)
(*         Copyright (c) 2000 Rainbow Software         *)
(*******************************************************)
// Documents used:
//  - National Semiconductor LM78 Microprocessor System
//     Hardware Monitor documentation (February 2000)
//  - Winbond W83781D H/W Monitoring IC Data Sheet v0.63
interface
 type TValueRAM=packed record
   // Measured values
   VCoreA:Byte;
   VCoreB:Byte;
   V33:Byte;
   V5:Byte;
   V12:Byte;
   _V5:Byte;
   _V12:Byte;
   Temp:Byte;
   Fan1:Byte;
   Fan2:Byte;
   Fan3:Byte;
   // Limits
   VCoreA_High : Byte;
   VCoreA_Low  : Byte;
   VCoreB_High : Byte;
   VCoreB_Low  : Byte;
   V33_High : Byte;
   V33_Low  : Byte;
   V5_High : Byte;
   V5_Low  : Byte;
   V12_High : Byte;
   V12_Low  : Byte;
   _V5_High: Byte;
   _V5_Low : Byte;
   _V12_High: Byte;
   _V12_Low : Byte;
   OverTemp : Byte;   // Over Temperature Limit (High)
   TempHyst : Byte;   // Temperature Hysteresis Limit (Low)
   Fan1_Low : Byte;   // Fan1 Count Limit
   Fan2_Low : Byte;   // Fan2 Count Limit
   Fan3_Low : Byte;   // Fan3 Count Limit
   Reserved1: Byte;
   Reserved2: Byte;
 end;
 type TLimits=packed record
   VCoreA_High : Byte;
   VCoreA_Low  : Byte;
   VCoreB_High : Byte;
   VCoreB_Low  : Byte;
   V33_High : Byte;
   V33_Low  : Byte;
   V5_High : Byte;
   V5_Low  : Byte;
   V12_High : Byte;
   V12_Low  : Byte;
   _V5_High: Byte;
   _V5_Low : Byte;
   _V12_High: Byte;
   _V12_Low : Byte;
   OverTemp : Byte;   // Over Temperature Limit (High)
   TempHyst : Byte;   // Temperature Hysteresis Limit (Low)
   Fan1_Low : Byte;   // Fan1 Count Limit
   Fan2_Low : Byte;   // Fan2 Count Limit
   Fan3_Low : Byte;   // Fan3 Count Limit
 end;
 type TBeepControl=packed record
   Enabled:Boolean;
   Fan1:Boolean;
   Fan2:Boolean;
   Fan3:Boolean;
   Temp1:Boolean;
   Temp23:Boolean;
   VCoreA:Boolean;
   VCoreB:Boolean;
   V33:Boolean;
   V5:Boolean;
   V5N:Boolean;
   V12:Boolean;
   V12N:Boolean;
   CaseOpen:Boolean;
 end;
 type TRealTimeStatus=packed record
   VCoreA:Boolean;
   VCoreB:Boolean; // = VINR0
   V5SB:Boolean;
   VBat:Boolean;
   V33:Boolean;
   V5:Boolean;
   V12:Boolean;
   _V5:Boolean;
   _V12:Boolean;
   Temp1:Boolean;
   Temp2:Boolean;
   Temp3:Boolean;
   Fan1:Boolean;
   Fan2:Boolean;
   Fan3:Boolean;
   CaseOpen:Boolean;
 end;
 type TValueRAM2=packed record
   V5SB:Byte;
   VBat:Byte;
   Reserved1:Byte;
   Reserved2:Byte;
   V5SB_High:Byte;
   V5SB_Low:Byte;
   VBat_High:Byte;
   VBat_Low:Byte;
 end;
 type TLimits2=packed record
   V5SB_High : Byte;
   V5SB_Low  : Byte;
   VBat_High : Byte;
   VBat_Low  : Byte;
 end;
 type TLM78=class
 protected
   FAddress  : Word;   // Base address of LM78 (at the ISA bus)
   FChipType : String;
   FValueRAM : TValueRAM;
   FFan1Div  : Byte;
   FFan2Div  : Byte;
   FFan3Div  : Byte;
   Winbond   : Boolean;
   W83782D   : Boolean;
   FValueRAM2: TValueRAM2;
   // Basic register I/O
   procedure WaitReady;
   function  ReadReg(Reg:Byte):Byte;
   procedure WriteReg(Reg,Value:Byte);
   // Auto-incrementation register I/O
   procedure ReadRegs(StartReg:Byte;var Buffer:Array of Byte;Count:Byte);
   procedure WriteRegs(StartReg:Byte;Buffer:Array of Byte;Count:Byte);
   // LM78 revision detection
   function  GetChipType:String;
   // Winbond-specific basic I/O
   procedure SelectBank(Bank:Byte);
   procedure SelectHiByte(What:Boolean);
   // Interupt status registers
   function  Int1Read:Byte;
   procedure Int1Write(Value:Byte);
   function  Int2Read:Byte;
   procedure Int2Write(Value:Byte);
   // SMI Mask Registers
   function  SMI1Read:Byte;
   procedure SMI1Write(Value:Byte);
   function  SMI2Read:Byte;
   procedure SMI2Write(Value:Byte);
   // NMI Mask Registers
   function  NMI1Read:Byte;
   procedure NMI1Write(Value:Byte);
   function  NMI2Read:Byte;
   procedure NMI2Write(Value:Byte);
   // VID (Voltage ID)
   function  VIDRead:Byte;
   // Divisors for Fan 1 and Fan 2
   function  F1DivRead:Byte;
   procedure F1DivWrite(Value:Byte);
   function  F2DivRead:Byte;
   procedure F2DivWrite(Value:Byte);
   function  F3DivRead:Byte; // Always 2 on LM78
   procedure F3DivWrite(Value:Byte);
   // Serial Bus Address register
   function  SerialBusRead:Byte;
   procedure SerialBusWrite(Value:Byte);
   // Winbond-specific
   function  Temp2EnableRead:Boolean;
   procedure Temp2EnableWrite(Value:Boolean);
   function  Temp2AdrRead:Byte;
   procedure Temp2AdrWrite(Value:Byte);
   function  Temp3EnableRead:Boolean;
   procedure Temp3EnableWrite(Value:Boolean);
   function  Temp3AdrRead:Byte;
   procedure Temp3AdrWrite(Value:Byte);
   function  ADCClockRead:Byte;
   procedure ADCClockWrite(Value:Byte);
   function  ClockInputRead:Byte;
   procedure ClockInputWrite(Value:Byte);
   function  OVT2EnabledRead:Boolean;
   procedure OVT2EnabledWrite(Value:Boolean);
   function  OVT3EnabledRead:Boolean;
   procedure OVT3EnabledWrite(Value:Boolean);
   function  OVTPolarityRead:Boolean;
   procedure OVTPolarityWrite(Value:Boolean);
   function  IRQEdgeRead:Boolean;
   procedure IRQEdgeWrite(Value:Boolean);
   function  PowerOnMonRead:Boolean;
   procedure PowerOnMonWrite(Value:Boolean);
   function  GPOSelectRead:Boolean;
   procedure GPOSelectWrite(Value:Boolean);
   function  Fan1PowerRead:Boolean;
   procedure Fan1PowerWrite(Value:Boolean);
   function  Fan2PowerRead:Boolean;
   procedure Fan2PowerWrite(Value:Boolean);
   function  Fan3PowerRead:Boolean;
   procedure Fan3PowerWrite(Value:Boolean);
   function  BeepControlRead:TBeepControl;
   procedure BeepControlWrite(Values:TBeepControl);
   function  Temp2Read:SmallInt;
   function  Temp2CfgRead:Byte;
   procedure Temp2CfgWrite(Value:Byte);
   function  Temp2HystRead:Word;
   procedure Temp2HystWrite(Value:Word);
   function  Temp2OVTRead:Word;
   procedure Temp2OVTWrite(Value:Word);
   function  Temp3Read:SmallInt;
   function  Temp3CfgRead:Byte;
   procedure Temp3CfgWrite(Value:Byte);
   function  Temp3HystRead:Word;
   procedure Temp3HystWrite(Value:Word);
   function  Temp3OVTRead:Word;
   procedure Temp3OVTWrite(Value:Word);
   function  Temp1DiodeRead:Boolean;
   procedure Temp1DiodeWrite(Value:Boolean);
   function  Temp2DiodeRead:Boolean;
   procedure Temp2DiodeWrite(Value:Boolean);
   function  Temp3DiodeRead:Boolean;
   procedure Temp3DiodeWrite(Value:Boolean);
   // Winbond W83782D ONLY
   function  PWMOut1Read:Byte;
   procedure PWMOut1Write(Value:Byte);
   function  PWMOut2Read:Byte;
   procedure PWMOut2Write(Value:Byte);
   function  PWMOut3Read:Byte;
   procedure PWMOut3Write(Value:Byte);
   function  PWMOut4Read:Byte;
   procedure PWMOut4Write(Value:Byte);
   function  PWMOut1ClockRead:Byte;
   procedure PWMOut1ClockWrite(Value:Byte);
   function  PWMOut2ClockRead:Byte;
   procedure PWMOut2ClockWrite(Value:Byte);
   function  PWMOut3ClockRead:Byte;
   procedure PWMOut3ClockWrite(Value:Byte);
   function  PWMOut4ClockRead:Byte;
   procedure PWMOut4ClockWrite(Value:Byte);
   function  Int3Read:Byte;
   procedure Int3Write(Value:Byte);
   function  SMI3Read:Byte;
   procedure SMI3Write(Value:Byte);
   function  BeepControl3Read:Byte;
   procedure BeepControl3Write(Value:Byte);
   function  RealTimeStatusRead:TRealTimeStatus;
 public
   Error:Byte;
   // Properties
   property Address:Word read FAddress;
   property ChipType:String read FChipType;
   property Winbond_:Boolean read Winbond;
   property W83782D_:Boolean read W83782D;
   property IntStatus1:Byte read Int1Read write Int1Write;
   property IntStatus2:Byte read Int2Read write Int2Write;
   property SMIMask1:Byte read SMI1Read write SMI1Write;
   property SMIMask2:Byte read SMI2Read write SMI2Write;
   property NMIMask1:Byte read NMI1Read write NMI1Write;
   property NMIMask2:Byte read NMI2Read write NMI2Write;
   property VID:Byte read VIDRead; // Read-only (we can't set CPU voltage!)
   property F1Div:Byte read F1DivRead write F1DivWrite;
   property F2Div:Byte read F2DivRead write F2DivWrite;
   property F3Div:Byte read F3DivRead write F3DivWrite;
   property SerialBusAdr:Byte read SerialBusRead write SerialBusWrite;
   property ValueRAM:TValueRAM read FValueRAM;
   // Winbond-specific
   property Temp2Enabled:Boolean read Temp2EnableRead write Temp2EnableWrite;
   property Temp2SerAdr:Byte read Temp2AdrRead write Temp2AdrWrite;
   property Temp3Enabled:Boolean read Temp3EnableRead write Temp3EnableWrite;
   property Temp3SerAdr:Byte read Temp3AdrRead write Temp3AdrWrite;
   property ADCClockDiv:Byte read ADCClockRead write ADCClockWrite;
   property ClockInput:Byte read ClockInputRead write ClockInputWrite;
   property OVT2Enabled:Boolean read OVT2EnabledRead write OVT2EnabledWrite;
   property OVT3Enabled:Boolean read OVT3EnabledRead write OVT3EnabledWrite;
   property OVTPolarityActiveHigh:Boolean read OVTPolarityRead write OVTPolarityWrite;
   property IRQEdgeActive:Boolean read IRQEdgeRead write IRQEdgeWrite;
   property PowerOnMonitorEnabled:Boolean read PowerOnMonRead write PowerOnMOnWrite;
   property GPOSelect:Boolean read GPOSelectRead write GPOSelectWrite;
   property Fan1Power:Boolean read Fan1PowerRead write Fan1PowerWrite;
   property Fan2Power:Boolean read Fan2PowerRead write Fan2PowerWrite;
   property Fan3Power:Boolean read Fan3PowerRead write Fan3PowerWrite;
   property BeepControl:TBeepControl read BeepControlRead write BeepControlWrite;
   property Temp2:SmallInt read Temp2Read;
   property Temp2Cfg:Byte read Temp2CfgRead write Temp2CfgWrite;
   property Temp2Hyst:Word read Temp2HystRead write Temp2HystWrite;
   property Temp2OVT:Word read Temp2OVTRead write Temp2OVTWrite;
   property Temp3:SmallInt read Temp3Read;
   property Temp3Cfg:Byte read Temp3CfgRead write Temp3CfgWrite;
   property Temp3Hyst:Word read Temp3HystRead write Temp3HystWrite;
   property Temp3OVT:Word read Temp3OVTRead write Temp3OVTWrite;
   // Winbond W83782D ONLY
   property Temp1DiodeP2:Boolean read Temp1DiodeRead write Temp1DiodeWrite;
   property Temp2DiodeP2:Boolean read Temp2DiodeRead write Temp2DiodeWrite;
   property Temp3DiodeP2:Boolean read Temp3DiodeRead write Temp3DiodeWrite;
   property PWMOut1:Byte read PWMOut1Read write PWMOut1Write;
   property PWMOut2:Byte read PWMOut2Read write PWMOut2Write;
   property PWMOut3:Byte read PWMOut3Read write PWMOut3Write;
   property PWMOut4:Byte read PWMOut4Read write PWMOut4Write;
   property PWMOut1Clock:Byte read PWMOut1ClockRead write PWMOut1ClockWrite;
   property PWMOut2Clock:Byte read PWMOut2ClockRead write PWMOut2ClockWrite;
   property PWMOut3Clock:Byte read PWMOut3ClockRead write PWMOut3ClockWrite;
   property PWMOut4Clock:Byte read PWMOut4ClockRead write PWMOut4ClockWrite;
   property IntStatus3:Byte read Int3Read write Int3Write;
   property SMIMask3:Byte read SMI3Read write SMI3Write;
   property BeepControl3:Byte read BeepControl3Read write BeepControl3Write;
   property RealTimeStatus:TRealTimeStatus read RealTimeStatusRead;
   property ValueRAM2:TValueRAM2 read FValueRAM2;
   // Procedures
   constructor Create(Adr:Word);
   destructor Done;
   procedure Power(What:Boolean);
   procedure SMIEnable(What:Boolean);
   procedure NMIIRQEnable(What:Boolean);
   procedure INTClear(What:Boolean);
   procedure Reset;
   procedure NMISelect(What:Boolean);
   procedure PowerBypass(What:Boolean);
   procedure Initialize;
   procedure ChipReset;
   procedure RefreshValues;
   procedure SetLimits(Limits:TLimits);
   procedure SetLimits2(Limits2:TLimits2);
 end;

const LM78_Config_Reg=$40;      // Configuration register
      LM78_Int1_Reg=$41;        // Interrupt status reg 1
      LM78_Int2_Reg=$42;        // Interrupt status reg 2
      LM78_SMI1_Reg=$43;        // SMI Mask Register 1
      LM78_SMI2_Reg=$44;        // SMI Mask Register 2
      LM78_NMI1_Reg=$45;        // NMI Mask Register 1
      LM78_NMI2_Reg=$46;        // NMI Mask Register 2
      LM78_VID_FanDiv_Reg=$47;  // VID/Fan Divisor reg
      LM78_SerialBus_Reg=$48;   // Serial Bus Address reg
      LM78_Reset_ID_Reg=$49;    // Chipset Reset/ID reg
      LM78_POST_RAM=$0;         // POST RAM starting reg
      LM78_Value_RAM=$20;       // Value RAM starting reg
      LM78_Value_RAM_AI=$60;    // Value RAM starting reg
                                //  (with auto-incrementation)
      W_Temp23_SerialAdr_Reg=$4A;
      W_PinControl_Reg=$4B;
      W_RQOVT_Property_Reg=$4C;
      W_FAN_BEEP_Control_Reg=$4D;
      W_Bank_Select=$4E;
      W_VendorID_Reg=$4F;
      W_ResistorTable1_Reg=$50;
      W_ResistorTable2_Reg=$51;
      W_Fan_PreDiv2_Reg=$55;
      W_BeepControl1_Reg=$56;
      W_BeepControl2_Reg=$57;
      W_ChipID_Reg=$58;
      // Bank l - temp2, bank 2 - temp3
      W_Temp_Sensor_HI_Reg=$50;
      W_Temp_Sensor_LO_Reg=$51;
      W_Temp_Config_Reg=$52;
      W_Temp_Hyst_HI_Reg=$53;
      W_Temp_Hyst_LO_Reg=$54;
      W_Temp_Over_HI_Reg=$55;
      W_Temp_Over_LO_Reg=$56;

      // Winbond W83782D specific
      W_DiodeSelect_Reg=$59;
      W_PWMOUT2_Reg=$5A;
      W_PWMOUT1_Reg=$5B;
      W_PMW12Clock_Reg=$5C;
      W_VBAT_Reg=$5D;
      W_PWMOUT3_Reg=$5E;
      W_PWMOUT4_Reg=$5F;
      // Bank 4
      W_Int3_Reg=$50;
      W_SMI3_Reg=$51;
      W_BeepControl3_Reg=$53;
      W_RealTime1_Reg=$59;
      W_RealTime2_Reg=$5A;
      W_RealTime3_Reg=$5B;
      W_PMW34Clock_Reg=$5C;
      // Bank 5
      W_ValueRAM2_AI=$50;
      // Bank 6 - Winbond test register $50
      // Errors
      ER_NoError=0;
      ER_TimeOut=1;
      // Chip Time out count
      TimeOut=65535;

implementation
uses SysUtils, HwMonU;

constructor TLM78.Create(Adr:Word);
begin
  FAddress:=Adr;
  Error:=ER_NoError;
  ChipReset;
  if Error<>ER_NoError then Exit;
  FFan1Div:=0;
  FFan2Div:=0;
  FFan3Div:=0;
  Initialize;
  FChipType:=GetChipType;
  Power(True);
  RefreshValues;
end;

destructor TLM78.Done;
begin
  Power(False);
end;

procedure TLM78.WaitReady;
var Adr,Counter:Word;
    A:Byte;
begin
  Adr:=Address+5;
  Counter:=0;
  repeat // Read address register until bit 7 is not set
{$IFDEF DlPortIO}
    A:=HardwareMon.DLPortIO.Port[Adr];
{$ELSE}
    asm
      mov dx,Adr
      in  al,dx
      mov A,al
    end;
{$ENDIF}
    Inc(Counter);
  until (A and 128=0) or (Counter=TimeOut);
  if Counter=TimeOut then Error:=ER_TimeOut else Error:=ER_NoError;
end;

function TLM78.ReadReg(Reg:Byte):Byte;
var Adr:Word;
begin
  WaitReady;
  Adr:=Address+5;
{$IFDEF DlPortIO}
  HardwareMon.DLPortIO.Port[Adr]:=Reg;
  Result:=HardwareMon.DLPortIO.Port[Adr+1];
{$ELSE}
  asm
    mov dx,Adr
    mov al,Reg
    out dx,al
    inc dx
    in  al,dx
    mov Result,al
  end;
{$ENDIF}
end;

procedure TLM78.WriteReg(Reg,Value:Byte);
var Adr:Word;
begin
  WaitReady;
  Adr:=Address+5;
{$IFDEF DlPortIO}
  HardwareMon.DLPortIO.Port[Adr]:=Reg;
  HardwareMon.DLPortIO.Port[Adr+1]:=Value;
{$ELSE}
  asm
    mov dx,Adr
    mov al,Reg
    out dx,al
    inc dx
    mov al,Value
    out dx,al
  end;
{$ENDIF}
end;

procedure TLM78.ReadRegs(StartReg:Byte;var Buffer:Array of Byte;Count:Byte);
var Adr:Word;
    A,B:Byte;
begin
  WaitReady;
  Adr:=Address+5;
  // Set starting register
{$IFDEF DlPortIO}
  HardwareMon.DLPortIO.Port[Adr]:=StartReg;
{$ELSE}
  asm
    mov dx,Adr
    mov al,StartReg
    out dx,al
  end;
{$ENDIF}
  Inc(Adr);
  // Read Count bytes from registers to buffer
  for A:=0 to Count-1 do
   begin
     {$IFDEF DlPortIO}
     B:=HardwareMon.DLPortIO.Port[Adr];
     {$ELSE}
     asm
       mov dx,Adr
       in  al,dx
       mov B,al
     end;
     {$ENDIF}
     Buffer[A]:=B;
   end;
end;

procedure TLM78.WriteRegs(StartReg:Byte;Buffer:Array of Byte;Count:Byte);
var Adr:Word;
    A,B:Byte;
begin
  WaitReady;
  Adr:=Address+5;
  // Set starting register
{$IFDEF DlPortIO}
  HardwareMon.DLPortIO.Port[Adr]:=StartReg;
{$ELSE}
  asm
    mov dx,Adr
    mov al,StartReg
    out dx,al
  end;
{$ENDIF}
  Inc(Adr);
  // Write Count bytes from buffer to registers
  for A:=0 to Count-1 do
   begin
     B:=Buffer[A];
     {$IFDEF DlPortIO}
     HardwareMon.DLPortIO.Port[Adr]:=B;
     {$ELSE}
     asm
       mov dx,Adr
       mov al,B
       out dx,al
     end;
     {$ENDIF}
   end;
end;

procedure TLM78.Power(What:Boolean);
// On:  Enables startup of monitoring operations
// Off: Puts LM78 in standby mode
begin
  if What then WriteReg(LM78_Config_Reg,ReadReg(LM78_Config_Reg) or 1)
   else WriteReg(LM78_Config_Reg,ReadReg(LM78_Config_Reg) and 254);
end;

procedure TLM78.SMIEnable(What:Boolean);
// On:  Enables SMI interrupt output
// Off: Disables SMI interrupt output
begin
  if What then WriteReg(LM78_Config_Reg,ReadReg(LM78_Config_Reg) or 2)
   else WriteReg(LM78_Config_Reg,ReadReg(LM78_Config_Reg) and 253);
end;

procedure TLM78.NMIIRQEnable(What:Boolean);
// On:  Enables NMI/IRQ interrupt output
// Off: Disables NMI/IRQ interrupt output
begin
  if What then WriteReg(LM78_Config_Reg,ReadReg(LM78_Config_Reg) or 4)
   else WriteReg(LM78_Config_Reg,ReadReg(LM78_Config_Reg) and 251);
end;

procedure TLM78.INTClear(What:Boolean);
// On:  Disables SMI and NMI/IRQ outputs without affecting
//      the contents of Interrupt Status Registers
//      The device will stop monitoring
// Off: Resume monitoring
begin
  if What then WriteReg(LM78_Config_Reg,ReadReg(LM78_Config_Reg) or 8)
   else WriteReg(LM78_Config_Reg,ReadReg(LM78_Config_Reg) and 247);
end;

procedure TLM78.Reset;
// Outputs at least a 20ms active low reset signal at RESET
// it <7>=1 in SMI Mask Register 2
begin
  WriteReg(LM78_Config_Reg,ReadReg(LM78_Config_Reg) or 16);
end;

procedure TLM78.NMISelect(What:Boolean);
// LM78:    On:  Selects NMI
//          Off: Selects IRQ
// Winbond: On:  IRQ active low          IRQ Polarity select
//          Off: IRQ active high
begin
  if What then WriteReg(LM78_Config_Reg,ReadReg(LM78_Config_Reg) or 32)
   else WriteReg(LM78_Config_Reg,ReadReg(LM78_Config_Reg) and 223);
end;

procedure TLM78.PowerBypass(What:Boolean);
// On:  Drives a zero on Power Switch Bypass pin
// Off: Returns Power Swich Bypass pin back
begin
  if What then WriteReg(LM78_Config_Reg,ReadReg(LM78_Config_Reg) or 64)
   else WriteReg(LM78_Config_Reg,ReadReg(LM78_Config_Reg) and 191);
end;

procedure TLM78.Initialize;
// Restores power on default value to all registers except
// Serial Bus Address register
begin
  WriteReg(LM78_Config_Reg,ReadReg(LM78_Config_Reg) or 128);
end;

function TLM78.GetChipType:String;
{ Two LM78 revisions are known - original LM78 and new LM78-J }
var A:Word;
begin
  Winbond:=False;
  W83782D:=False;
  Result:='LM78';
  if ReadReg(LM78_Reset_ID_Reg) and 64=64 then Result:=Result+'-J';
  { If bit 6 of Chip Reset/ID Register is 1, it's LM78-J }
  SelectHiByte(True);
  A:=256*ReadReg(W_VendorID_Reg);
  SelectHiByte(False);
  A:=A+ReadReg(W_VendorID_Reg);
  if A=$5CA3 then  // $5CA3 - Winbond vendor ID
   begin
     Result:='Winbond ';
     Winbond:=True;
   end
  else Exit;
  A:=ReadReg(W_ChipID_Reg);
  case A and 240 of
    16:Result:=Result+'W83781D';
    48:begin Result:=Result+'W83782D';W83782D:=True;end;
    else Result:=Result+'type '+IntToStr(A);
  end;
  Result:=Result+' rev. '+IntToStr(A and 15);
end;

procedure TLM78.SelectBank(Bank:Byte);
// Selects bank for registers $50-$5F (Winbond only)
begin
  if not Winbond then Exit;
  WriteReg(W_Bank_Select,ReadReg(W_Bank_Select) and 248+Bank);
end;

procedure TLM78.SelectHiByte(What:Boolean);
// Selects high/low-byte access for reg $4F (Winbond only)
begin
//   This is correct according to documentation, but it
//   does not work correctly (sometimes work, sometimes not)
//  if What then WriteReg(W_Bank_Select,ReadReg(W_Bank_Select) or 128)
//   else WriteReg(W_Bank_Select,ReadReg(W_Bank_Select) and 127);
  if What then WriteReg(W_Bank_Select,128) else WriteReg(W_Bank_Select,0);
end;

function Div2Num(A:Byte):Byte;
var B:Integer;
begin
  Result:=1;
  for B:=1 to A do Result:=Result*2; // Easy to count
                                     // (see Num2Div...)
(*  Result:=1;
  case A of
    1: Result:=2;
    2: Result:=4;
    3: Result:=8;
  end;*)
end;

function Num2Div(A:Byte):Byte;
begin
  Result:=0;
  case A of // Much simpler than counting this (not-so-easy)
    2: Result:=1;
    4: Result:=2;
    8: Result:=3;
   16: Result:=4;
   32: Result:=5;
   64: Result:=6;
  128: Result:=7;
  end;
end;

// Interupt status registers
function TLM78.Int1Read:Byte;
begin
  Result:=ReadReg(LM78_Int1_Reg);
end;

procedure TLM78.Int1Write(Value:Byte);
begin
  WriteReg(LM78_Int1_Reg,Value);
end;

function TLM78.Int2Read:Byte;
begin
  Result:=ReadReg(LM78_Int2_Reg);
end;

procedure TLM78.Int2Write(Value:Byte);
begin
  WriteReg(LM78_Int2_Reg,Value);
end;

// SMI Mask Registers
function TLM78.SMI1Read:Byte;
begin
  Result:=ReadReg(LM78_SMI1_Reg);
end;

procedure TLM78.SMI1Write(Value:Byte);
begin
  WriteReg(LM78_SMI1_Reg,Value);
end;

function TLM78.SMI2Read:Byte;
begin
  Result:=ReadReg(LM78_SMI2_Reg);
end;

procedure TLM78.SMI2Write(Value:Byte);
begin
  WriteReg(LM78_SMI2_Reg,Value);
end;

// NMI Mask Registers
function TLM78.NMI1Read:Byte;
begin
  Result:=ReadReg(LM78_NMI1_Reg);
end;

procedure TLM78.NMI1Write(Value:Byte);
begin
  WriteReg(LM78_NMI1_Reg,Value);
end;

function TLM78.NMI2Read:Byte;
begin
  Result:=ReadReg(LM78_NMI2_Reg);
end;

procedure TLM78.NMI2Write(Value:Byte);
begin
  WriteReg(LM78_NMI2_Reg,Value);
end;

function TLM78.VIDRead:Byte;
begin
  Result:=ReadReg(LM78_VID_FanDiv_Reg) and 15;
  if Winbond then Result:=Result+ReadReg(LM78_Reset_ID_Reg) and 1 shl 4;
end;

// Divisors for Fan 1 and Fan 2
function TLM78.F1DivRead:Byte;
begin
  if FFan1Div=0 then
   begin
     Result:=ReadReg(LM78_VID_FanDiv_Reg) shr 4 and 3;
     if W83782D then Result:=Result+ReadReg(W_VBAT_Reg) shr 3 and 4;
     Result:=Div2Num(Result);
     FFan1Div:=Result;
   end
  else Result:=FFan1Div;
end;

procedure TLM78.F1DivWrite(Value:Byte);
begin
  WriteReg(LM78_VID_FanDiv_Reg,ReadReg(LM78_VID_FanDiv_Reg) and 207+Num2Div(Value) and 3 shl 4);
  if W83782D then WriteReg(W_VBAT_Reg,ReadReg(W_VBAT_Reg) and 223+Num2Div(Value) and 4 shl 3);
  FFan1Div:=Value;
end;

function TLM78.F2DivRead:Byte;
begin
  if FFan2Div=0 then
   begin
     Result:=ReadReg(LM78_VID_FanDiv_Reg) shr 6;
     if W83782D then Result:=Result+ReadReg(W_VBAT_Reg) shr 4 and 4;
     Result:=Div2Num(Result);
     FFan2Div:=Result;
   end
  else Result:=FFan2Div;
end;

procedure TLM78.F2DivWrite(Value:Byte);
begin
  WriteReg(LM78_VID_FanDiv_Reg,ReadReg(LM78_VID_FanDiv_Reg) and 63+Num2Div(Value) and 3 shl 6);
  if W83782D then WriteReg(W_VBAT_Reg,ReadReg(W_VBAT_Reg) and 191+Num2Div(Value) and 4 shl 4);
  FFan2Div:=Value;
end;

function TLM78.F3DivRead:Byte;
begin
  Result:=2;
  if not Winbond then Exit;
  if FFan3Div=0 then
   begin
     Result:=ReadReg(W_PinControl_Reg) shr 6;
     if W83782D then Result:=Result+ReadReg(W_VBAT_Reg) shr 5 and 4;
     Result:=Div2Num(Result);
     FFan3Div:=Result;
   end
  else Result:=FFan3Div;
end;

procedure TLM78.F3DivWrite(Value:Byte);
begin
  if not Winbond then Exit;
  WriteReg(W_PinControl_Reg,ReadReg(W_PinControl_Reg) and 63+Num2Div(Value) and 3 shl 6);
  if W83782D then WriteReg(W_VBAT_Reg,ReadReg(W_VBAT_Reg) and 127+Num2Div(Value) and 4 shl 5);
  FFan3Div:=Value;
end;

// Serial Bus Address register
function TLM78.SerialBusRead:Byte;
begin
  Result:=ReadReg(LM78_SerialBus_Reg);
end;

procedure TLM78.SerialBusWrite(Value:Byte);
begin
  WriteReg(LM78_SerialBus_Reg,Value);
end;

procedure TLM78.ChipReset;
begin
  WriteReg(LM78_Reset_ID_Reg,ReadReg(LM78_Reset_ID_Reg) or 32);
end;

procedure TLM78.RefreshValues;
var A:array[1..32] of Byte;
begin
  ReadRegs(LM78_Value_RAM_AI,A,SizeOf(A));
  Move(A,FValueRAM,SizeOf(FValueRAM));
  if W83782D then  // Refresh 2nd Value RAM for W83782D
   begin
     ReadRegs(W_ValueRAM2_AI,A,10);
     Move(A,FValueRAM2,SizeOf(FValueRAM2));   
   end;
end;

procedure TLM78.SetLimits(Limits:TLimits);
var A:Array[1..19] of Byte;
begin
  Move(Limits,A,SizeOf(A));
  WriteRegs(LM78_Value_RAM_AI+11,A,SizeOf(A));
end;

// Winbond-specific
function TLM78.Temp2EnableRead:Boolean; // Temp 2
begin
  Result:=False;
  if not Winbond then Exit;
  Result:=ReadReg(W_Temp23_SerialAdr_Reg) and 8=0;
end;

procedure TLM78.Temp2EnableWrite(Value:Boolean);
var A:Byte;
begin
  if not Winbond then Exit;
  if Value then A:=8 else A:=0;
  WriteReg(W_Temp23_SerialAdr_Reg,ReadReg(W_Temp23_SerialAdr_Reg) and 247+A);
end;

function TLM78.Temp2AdrRead:Byte;
begin
  Result:=72+ReadReg(W_Temp23_SerialAdr_Reg) and 7;
end;

procedure TLM78.Temp2AdrWrite(Value:Byte);
begin
  WriteReg(W_Temp23_SerialAdr_Reg,ReadReg(W_Temp23_SerialAdr_Reg) and 248+Value and 7);
end;

function TLM78.Temp3EnableRead:Boolean; // Temp 3
begin
  Result:=False;
  if not Winbond then Exit;
  Result:=ReadReg(W_Temp23_SerialAdr_Reg) and 128=0;
end;

procedure TLM78.Temp3EnableWrite(Value:Boolean);
var A:Byte;
begin
  if not Winbond then Exit;
  if Value then A:=128 else A:=0;
  WriteReg(W_Temp23_SerialAdr_Reg,ReadReg(W_Temp23_SerialAdr_Reg) and 127+A);
end;

function TLM78.Temp3AdrRead:Byte;
begin
  Result:=72+ReadReg(W_Temp23_SerialAdr_Reg) shr 4 and 7;
end;

procedure TLM78.Temp3AdrWrite(Value:Byte);
begin
  WriteReg(W_Temp23_SerialAdr_Reg,ReadReg(W_Temp23_SerialAdr_Reg) and 143+Value and 7 shl 4);
end;

function TLM78.ADCClockRead:Byte;
var A:Byte;
begin // Returns divider: ADC clock=22.5kHz/divider
  A:=ReadReg(W_PinControl_Reg) shr 4 and 3;
  Result:=1;
  case A of
    1: Result:=4;
    2: Result:=16;
    3: Result:=64;
  end;
end;

procedure TLM78.ADCClockWrite(Value:Byte);
var A:Byte;
begin
  A:=0;
  case Value of
    4: A:=1;
   16: A:=2;
   64: A:=3;
  end;
  WriteReg(W_PinControl_Reg,ReadReg(W_PinControl_Reg) and 207+A shl 4);
end;

function TLM78.ClockInputRead:Byte;
// Clock input select:
// 0 - Pin 3 (CLKIN) select 14.318 MHz clock
// 1 - Pin 3 (CLKIN) select 24 MHz clock (default)
// 2 - Pin 3 (CLKIN) select 48 MHz clock
// 3 - Pin 3 no clock input (reserved)
begin
  Result:=1;
  if not Winbond then Exit;
  Result:=ReadReg(W_PinControl_Reg) shr 2 and 3;
end;

procedure TLM78.ClockInputWrite(Value:Byte);
begin
  if not Winbond then Exit;
  WriteReg(W_PinControl_Reg,ReadReg(W_PinControl_Reg) and 243+Value and 3 shl 2);
end;

function TLM78.OVT2EnabledRead:Boolean;
begin
  Result:=ReadReg(W_RQOVT_Property_Reg) shr 3 and 1=0;
end;

procedure TLM78.OVT2EnabledWrite(Value:Boolean);
var A:Byte;
begin
  if not Value then A:=8 else A:=0;
  WriteReg(W_RQOVT_Property_Reg,ReadReg(W_RQOVT_Property_Reg) and 247+A);
end;

function TLM78.OVT3EnabledRead:Boolean;
begin
  Result:=ReadReg(W_RQOVT_Property_Reg) shr 4 and 1=0;
end;

procedure TLM78.OVT3EnabledWrite(Value:Boolean);
var A:Byte;
begin
  if not Value then A:=16 else A:=0;
  WriteReg(W_RQOVT_Property_Reg,ReadReg(W_RQOVT_Property_Reg) and 239+A);
end;

function TLM78.OVTPolarityRead:Boolean;
begin
  Result:=ReadReg(W_RQOVT_Property_Reg) shr 2 and 1=1;
end;

procedure TLM78.OVTPolarityWrite(Value:Boolean);
var A:Byte;
begin
  if not Value then A:=4 else A:=0;
  WriteReg(W_RQOVT_Property_Reg,ReadReg(W_RQOVT_Property_Reg) and 251+A);
end;

function TLM78.IRQEdgeRead:Boolean;
begin
  Result:=ReadReg(W_RQOVT_Property_Reg) shr 1 and 1=1;
end;

procedure TLM78.IRQEdgeWrite(Value:Boolean);
var A:Byte;
begin
  if not Value then A:=2 else A:=0;
  WriteReg(W_RQOVT_Property_Reg,ReadReg(W_RQOVT_Property_Reg) and 253+A);
end;

function TLM78.PowerOnMonRead:Boolean;
begin
  Result:=ReadReg(W_FAN_BEEP_Control_Reg) shr 7=0;
end;

procedure TLM78.PowerOnMonWrite(Value:Boolean);
var A:Byte;
begin
  if not Value then A:=128 else A:=0;
  WriteReg(W_FAN_BEEP_Control_Reg,ReadReg(W_FAN_BEEP_Control_Reg) and 127+A);
end;

function TLM78.GPOSelectRead:Boolean;
begin
  Result:=ReadReg(W_FAN_BEEP_Control_Reg) shr 6 and 1=1;
end;

procedure TLM78.GPOSelectWrite(Value:Boolean);
var A:Byte;
begin
  if Value then A:=64 else A:=0;
  WriteReg(W_FAN_BEEP_Control_Reg,ReadReg(W_FAN_BEEP_Control_Reg) and 191+A);
end;

function TLM78.Fan1PowerRead:Boolean;
var A:Byte;
begin
  A:=ReadReg(W_FAN_BEEP_Control_Reg) and 3;
  Result:=True;
  if A=2 then Result:=False;
end;

procedure TLM78.Fan1PowerWrite(Value:Boolean);
var A:Byte;
begin
  if Value then A:=1 else A:=2;
  WriteReg(W_FAN_BEEP_Control_Reg,ReadReg(W_FAN_BEEP_Control_Reg) and 252+A);
end;

function TLM78.Fan2PowerRead:Boolean;
var A:Byte;
begin
  A:=ReadReg(W_FAN_BEEP_Control_Reg) shr 2 and 3;
  Result:=True;
  if A=2 then Result:=False;
end;

procedure TLM78.Fan2PowerWrite(Value:Boolean);
var A:Byte;
begin
  if Value then A:=4 else A:=8;
  WriteReg(W_FAN_BEEP_Control_Reg,ReadReg(W_FAN_BEEP_Control_Reg) and 243+A);
end;

function TLM78.Fan3PowerRead:Boolean;
var A:Byte;
begin
  A:=ReadReg(W_FAN_BEEP_Control_Reg) shr 4 and 3;
  Result:=True;
  if A=2 then Result:=False;
end;

procedure TLM78.Fan3PowerWrite(Value:Boolean);
var A:Byte;
begin
  if Value then A:=16 else A:=32;
  WriteReg(W_FAN_BEEP_Control_Reg,ReadReg(W_FAN_BEEP_Control_Reg) and 207+A);
end;

function TLM78.BeepControlRead:TBeepControl;
var A:Byte;
begin
  A:=ReadReg(W_BeepControl1_Reg);
  with Result do
   begin
     Fan2:=A and 128=128;
     Fan1:=A and 64=64;
     Temp23:=A and 32=32;
     Temp1:=A and 16=16;
     V5:=A and 8=8;
     V33:=A and 4=4;
     VCoreB:=A and 2=2;
     VCoreA:=A and 1=1;
   end;
  A:=ReadReg(W_BeepControl2_Reg);
  with Result do
   begin
     Enabled:=A and 128=128;
     CaseOpen:=A and 16=16;
     Fan3:=A and 8=8;
     V5N:=A and 4=4;
     V12N:=A and 2=2;
     V12:=A and 1=1;
   end;
end;

procedure TLM78.BeepControlWrite(Values:TBeepControl);
var A:Byte;
begin
  A:=0;
  with Values do
   begin
     if Fan2 then A:=A or 128;
     if Fan1 then A:=A or 64;
     if Temp23 then A:=A or 32;
     if Temp1 then A:=A or 16;
     if V5 then A:=A or 8;
     if V33 then A:=A or 4;
     if VCoreB then A:=A or 2;
     if VCoreA then A:=A or 1;
   end;
  WriteReg(W_BeepControl1_Reg,A);
  A:=0;
  with Values do
   begin
     if Enabled then A:=A or 128;
     if CaseOpen then A:=A or 16;
     if Fan3 then A:=A or 8;
     if V5N then A:=A or 4;
     if V12N then A:=A or 2;
     if V12 then A:=A or 1;
   end;
  WriteReg(W_BeepControl2_Reg,A);
end;

function Temp9Bit(Temp:Word):SmallInt;
begin // Temperature=Temp9bit/10
  if Temp and 256=256 then // Negative temperature
   begin
     Result:=-5*(512-Temp)
   end
  else // Positive temperature
   begin
     Result:=(Temp*5)
   end;
end;

function TLM78.Temp2Read:SmallInt;
begin
  Result:=0;
  if not Winbond then Exit;
  SelectBank(1);
  Result:=Temp9Bit(ReadReg(W_Temp_Sensor_HI_Reg) shl 1+ReadReg(W_Temp_Sensor_LO_Reg) shr 7);
  SelectBank(0);
end;

function TLM78.Temp2CfgRead:Byte;
begin
  Result:=0;
  if not Winbond then Exit;
  SelectBank(1);
  Result:=ReadReg(W_Temp_Config_Reg);
  SelectBank(0);
end;

procedure TLM78.Temp2CfgWrite(Value:Byte);
begin
  if not Winbond then Exit;
  SelectBank(1);
  WriteReg(W_Temp_Config_Reg,Value);
  SelectBank(0);
end;

function TLM78.Temp2HystRead:Word;
begin
  Result:=0;
  if not Winbond then Exit;
  SelectBank(1);
  Result:=ReadReg(W_Temp_Hyst_HI_Reg) shl 1+ReadReg(W_Temp_Hyst_LO_Reg) shr 7;
  SelectBank(0);
end;

procedure TLM78.Temp2HystWrite(Value:Word);
begin
  if not Winbond then Exit;
  SelectBank(1);
  WriteReg(W_Temp_Hyst_HI_Reg,Value shr 1);
  WriteReg(W_Temp_Hyst_LO_Reg,Value shl 7 and 128);
  SelectBank(0);
end;

function TLM78.Temp2OVTRead:Word;
begin
  Result:=0;
  if not Winbond then Exit;
  SelectBank(1);
  Result:=ReadReg(W_Temp_Over_HI_Reg) shl 1+ReadReg(W_Temp_Over_LO_Reg) shr 7;
  SelectBank(0);
end;

procedure TLM78.Temp2OVTWrite(Value:Word);
begin
  if not Winbond then Exit;
  SelectBank(1);
  WriteReg(W_Temp_Over_HI_Reg,Value shr 1);
  WriteReg(W_Temp_Over_LO_Reg,Value shl 7 and 128);
  SelectBank(0);
end;

function TLM78.Temp3Read:SmallInt;
begin
  Result:=0;
  if not Winbond then Exit;
  SelectBank(2);
  Result:=Temp9Bit(ReadReg(W_Temp_Sensor_HI_Reg) shl 1+ReadReg(W_Temp_Sensor_LO_Reg) shr 7);
  SelectBank(0);
end;

function TLM78.Temp3CfgRead:Byte;
begin
  Result:=0;
  if not Winbond then Exit;
  SelectBank(2);
  Result:=ReadReg(W_Temp_Config_Reg);
  SelectBank(0);
end;

procedure TLM78.Temp3CfgWrite(Value:Byte);
begin
  if not Winbond then Exit;
  SelectBank(2);
  WriteReg(W_Temp_Config_Reg,Value);
  SelectBank(0);
end;

function TLM78.Temp3HystRead:Word;
begin
  Result:=0;
  if not Winbond then Exit;
  SelectBank(2);
  Result:=ReadReg(W_Temp_Hyst_HI_Reg) shl 1+ReadReg(W_Temp_Hyst_LO_Reg) shr 7;
  SelectBank(0);
end;

procedure TLM78.Temp3HystWrite(Value:Word);
begin
  if not Winbond then Exit;
  SelectBank(2);
  WriteReg(W_Temp_Hyst_HI_Reg,Value shr 1);
  WriteReg(W_Temp_Hyst_LO_Reg,Value shl 7 and 128);
  SelectBank(0);
end;

function TLM78.Temp3OVTRead:Word;
begin
  Result:=0;
  if not Winbond then Exit;
  SelectBank(2);
  Result:=ReadReg(W_Temp_Over_HI_Reg) shl 1+ReadReg(W_Temp_Over_LO_Reg) shr 7;
  SelectBank(0);
end;

procedure TLM78.Temp3OVTWrite(Value:Word);
begin
  if not Winbond then Exit;
  SelectBank(2);
  WriteReg(W_Temp_Over_HI_Reg,Value shr 1);
  WriteReg(W_Temp_Over_LO_Reg,Value shl 7 and 128);
  SelectBank(0);
end;

// Winbond W83782D ONLY
function TLM78.Temp1DiodeRead:Boolean;
begin
  Result:=False;
  if not W83782D then Exit;
  Result:=ReadReg(W_DiodeSelect_Reg) and 16=16;
end;

procedure TLM78.Temp1DiodeWrite(Value:Boolean);
var A:Byte;
begin
  if not W83782D then Exit;
  if Value then A:=16 else A:=0;
  WriteReg(W_DiodeSelect_Reg,ReadReg(W_DiodeSelect_Reg) and 239+A);
end;

function TLM78.Temp2DiodeRead:Boolean;
begin
  Result:=False;
  if not W83782D then Exit;
  Result:=ReadReg(W_DiodeSelect_Reg) and 32=32;
end;

procedure TLM78.Temp2DiodeWrite(Value:Boolean);
var A:Byte;
begin
  if not W83782D then Exit;
  if Value then A:=32 else A:=0;
  WriteReg(W_DiodeSelect_Reg,ReadReg(W_DiodeSelect_Reg) and 223+A);
end;

function TLM78.Temp3DiodeRead:Boolean;
begin
  Result:=False;
  if not W83782D then Exit;
  Result:=ReadReg(W_DiodeSelect_Reg) and 64=64;
end;

procedure TLM78.Temp3DiodeWrite(Value:Boolean);
var A:Byte;
begin
  if not W83782D then Exit;
  if Value then A:=64 else A:=0;
  WriteReg(W_DiodeSelect_Reg,ReadReg(W_DiodeSelect_Reg) and 191+A);
end;

function TLM78.PWMOut1Read:Byte;
begin
  Result:=0;
  if not W83782D then Exit;
  Result:=ReadReg(W_PWMOUT1_Reg);
end;

procedure TLM78.PWMOut1Write(Value:Byte);
begin
  if not W83782D then Exit;
  WriteReg(W_PWMOUT1_Reg,Value);
end;

function TLM78.PWMOut2Read:Byte;
begin
  Result:=0;
  if not W83782D then Exit;
  Result:=ReadReg(W_PWMOUT2_Reg);
end;

procedure TLM78.PWMOut2Write(Value:Byte);
begin
  if not W83782D then Exit;
  WriteReg(W_PWMOUT2_Reg,Value);
end;

function TLM78.PWMOut3Read:Byte;
begin
  Result:=0;
  if not W83782D then Exit;
  Result:=ReadReg(W_PWMOUT3_Reg);
end;

procedure TLM78.PWMOut3Write(Value:Byte);
begin
  if not W83782D then Exit;
  WriteReg(W_PWMOUT3_Reg,Value);
end;

function TLM78.PWMOut4Read:Byte;
begin
  Result:=0;
  if not W83782D then Exit;
  Result:=ReadReg(W_PWMOUT4_Reg);
end;

procedure TLM78.PWMOut4Write(Value:Byte);
begin
  if not W83782D then Exit;
  WriteReg(W_PWMOUT4_Reg,Value);
end;

function TLM78.PWMOut1ClockRead:Byte;
// Table from docs:
// 0 .. 46.87kHz   (div=1)
// 1 .. 23.43kHz   (div=2)
// 2 .. 11.72kHz   (div=4)
// 3 ..  5.85kHz   (div=8)
// 4 ..  2.93kHz   (div=16)
// Looks like divisor=2^number, so Clock=46.87/2^PWMOutClock
begin
  Result:=0;
  if not W83782D then Exit;
  Result:=ReadReg(W_PMW12Clock_Reg) shr 4 and 7;
end;

procedure TLM78.PWMOut1ClockWrite(Value:Byte);
begin
  if not W83782D then Exit;
  WriteReg(W_PMW12Clock_Reg,ReadReg(W_PMW12Clock_Reg) and 143+Value and 7 shl 4);
end;

function TLM78.PWMOut2ClockRead:Byte;
begin
  Result:=0;
  if not W83782D then Exit;
  Result:=ReadReg(W_PMW12Clock_Reg) and 7;
end;

procedure TLM78.PWMOut2ClockWrite(Value:Byte);
begin
  if not W83782D then Exit;
  WriteReg(W_PMW12Clock_Reg,ReadReg(W_PMW12Clock_Reg) and 258+Value and 7);
end;

function TLM78.PWMOut3ClockRead:Byte;
begin
  Result:=0;
  if not W83782D then Exit;
  SelectBank(4);
  Result:=ReadReg(W_PMW34Clock_Reg) and 7;
  SelectBank(0);
end;

procedure TLM78.PWMOut3ClockWrite(Value:Byte);
begin
  if not W83782D then Exit;
  SelectBank(4);
  WriteReg(W_PMW34Clock_Reg,ReadReg(W_PMW34Clock_Reg) and 258+Value and 7);
  SelectBank(0);
end;

function TLM78.PWMOut4ClockRead:Byte;
begin
  Result:=0;
  if not W83782D then Exit;
  SelectBank(4);
  Result:=ReadReg(W_PMW34Clock_Reg) shr 4 and 7;
  SelectBank(0);
end;

procedure TLM78.PWMOut4ClockWrite(Value:Byte);
begin
  if not W83782D then Exit;
  SelectBank(4);
  WriteReg(W_PMW34Clock_Reg,ReadReg(W_PMW34Clock_Reg) and 143+Value and 7 shl 4);
  SelectBank(0);
end;

function TLM78.Int3Read:Byte;
begin
  Result:=0;
  if not W83782D then Exit;
  SelectBank(4);
  Result:=ReadReg(W_Int3_Reg);
  SelectBank(0);
end;

procedure TLM78.Int3Write(Value:Byte);
begin
  if not W83782D then Exit;
  SelectBank(4);
  WriteReg(W_Int3_Reg,Value);
  SelectBank(0);
end;

function TLM78.SMI3Read:Byte;
begin
  Result:=0;
  if not W83782D then Exit;
  SelectBank(4);
  Result:=ReadReg(W_SMI3_Reg);
  SelectBank(0);
end;

procedure TLM78.SMI3Write(Value:Byte);
begin
  if not W83782D then Exit;
  SelectBank(4);
  WriteReg(W_SMI3_Reg,Value);
  SelectBank(0);
end;

function TLM78.BeepControl3Read:Byte;
begin
  Result:=0;
  if not W83782D then Exit;
  SelectBank(4);
  Result:=ReadReg(W_BeepControl3_Reg);
  SelectBank(0);
end;

procedure TLM78.BeepControl3Write(Value:Byte);
begin
  if not W83782D then Exit;
  SelectBank(4);
  WriteReg(W_BeepControl3_Reg,Value);
  SelectBank(0);
end;

function TLM78.RealTimeStatusRead:TRealTimeStatus;
var A:Byte;
begin
  FillChar(Result,SizeOf(Result),0);
  if not W83782D then Exit;
  SelectBank(4);
  A:=ReadReg(W_RealTime1_Reg);
  SelectBank(0);
  with Result do
   begin
     Fan2:=A and 128=128;
     Fan1:=A and 64=64;
     Temp2:=A and 32=32;
     Temp1:=A and 16=16;
     V5:=A and 8=8;
     V33:=A and 4=4;
     VCoreB:=A and 2=2;
     VCoreA:=A and 1=1;
   end;
  SelectBank(4);
  A:=ReadReg(W_RealTime2_Reg);
  SelectBank(0);
  with Result do
   begin
     Temp3:=A and 32=32;
     CaseOpen:=A and 16=16;
     Fan3:=A and 8=8;
     _V5:=A and 4=4;
     _V12:=A and 2=2;
     V12:=A and 1=1;
   end;
  SelectBank(4);
  A:=ReadReg(W_RealTime3_Reg);
  SelectBank(0);
  with Result do
   begin
     VBat:=A and 2=2;
     V5SB:=A and 1=1;
   end;
end;

procedure TLM78.SetLimits2(Limits2:TLimits2);
var A:Array[1..4] of Byte;
begin
  Move(Limits2,A,SizeOf(A));
  WriteRegs(W_ValueRAM2_AI+11,A,SizeOf(A));
end;

end.
