{
 Com to Keyboard - The Com to Keyboard utility for DOS.
 Copyright (C) 1999-2002  Henrich Fukna <fuky@azet.sk>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
}
program ComKbd;

{$M 3072, 0, 0}

uses Dos, Lib;

const
  Copyright: String = 'COM to KEYBOARD'; { identity string }
  Version = '1.2';                       { version string constant }
  ExeName = 'COMKBD';                    { project name }

  EOI = $20;                             { "end-of-interrupt" constant }

  ComPort: Word = $3F8;                  { serial base port }
  ComIntr: Byte = $0C;                   { serial hardware/PIC interrupt }
  IRQctrl: Byte = $20;                   { interrupt PIC OCW2 port }

var
  MySS, MySP, OldSS, OldSP: Word;        { stacks pointers }
  OldCom, OldMultiplex: Pointer;         { old interrupt handlers }
  Reg: Registers;                        { cpu registers structure }

  MyTask: Byte;                          { resident task number }
  Active: Boolean;                       { activity flag }
  Switch: Integer;                       { stack switched flag }

  Data: Byte;                            { serial data }

{$F+}

{********** common interrupt functions ***************************}

{*
 * Call the old interrupt handler.
 *
 * @param Address               pointer to old interrupt handler
 * @param Reg                   cpu registers structure
 * @return changed Reg cpu registers structure
 *}
procedure CallOldIntr(Address: Pointer; var Reg: Registers);
var
  _Flags, _AX, _BX, _CX, _DX, _SI, _DI, _DS, _ES, _BP: Word;
begin
  with Reg do begin                      { set registers }
    _Flags:= Flags;
    _AX:= Ax; _BX:= Bx; _CX:= Cx; _DX:= Dx;
    _SI:= Si; _DI:= Di;
    _DS:= Ds; _ES:= Es;
    _BP:= Bp;
  end;
  asm
    PUSH  DS
    PUSH  BP

    PUSH  _Flags
    PUSH  CS
{$IFOPT G+}
    PUSH  OFFSET @@1
{$ELSE}
    MOV   AX, OFFSET @@1
    PUSH  AX
{$ENDIF}
    PUSH  WORD (Address+2)
    PUSH  WORD (Address)

    MOV   AX, _AX
    MOV   BX, _BX
    MOV   CX, _CX
    MOV   DX, _DX
    MOV   SI, _SI
    MOV   DI, _DI
    MOV   DS, _DS
    MOV   ES, _ES
    MOV   BP, _BP

    RETF                                 { call :-) }
@@1:
    PUSH  BP

    MOV   BP, SP
    MOV   BP, SS:[BP + 2]

    MOV   _AX, AX
    MOV   _BX, BX
    MOV   _CX, CX
    MOV   _DX, DX
    MOV   _SI, SI
    MOV   _DI, DI
    MOV   _DS, DS
    MOV   _ES, ES
    POP   AX
    MOV   _BP, AX
    PUSHF
    POP   AX
    MOV   _FLAGS, AX

    POP   BP
    POP   DS
  end;
  with Reg do begin                      { set registers }
    Flags:= _Flags;
    Ax:= _AX; Bx:= _BX; Cx:= _CX; Dx:= _DX;
    Si:= _SI; Di:= _DI;
    Ds:= _DS; Es:= _ES;
    Bp:= _BP;
  end;
end;

{********** interrupt handlers ***********************************}

{*
 * PIC interrupt handler.
 *}
procedure NewCom; interrupt;
begin
  if Active then begin                   { if active }
    Inc(Switch);                         { switch the stack context }
    if Switch = 0 then begin
      OldSS:= SSeg;
      OldSP:= SPtr;
      asm
        MOV  SS,MySS
        MOV  SP,MySP
      end;
    end;

    Data:= Port[ComPort];                { get data from serial port }
    Port[IRQctrl]:= EOI;                 { send PIC "end-of-interrupt" }

    Reg.Ah:= 5;                          { put data to the keyboard buffer }
    Reg.Ch:= 0;                          { don't care kbd "scan code" }
    Reg.Cl:= Data;
    Intr($16, Reg);

    Dec(Switch);                         { switch the stack context }
    if Switch < 0 then begin
      asm
        MOV  SS,OldSS
        MOV  SP,OldSP
      end;
    end;
  end else begin                         { call previous routine }
    asm
      PUSHF
      CALL  OldCom
    end;
  end
end;

function Terminate: Word; forward;

{*
 * DOS Multiplex handler.
 *
 * @param _Flags                cpu flags register
 * @param _CS                   cpu code segment register
 * @param _IP                   cpu instruction pointer register
 * @param _AX                   cpu accumulator register
 * @param _BX                   cpu base register
 * @param _CX                   cpu counter register
 * @param _DX                   cpu data register
 * @param _SI                   cpu source index register
 * @param _DI                   cpu destination index register
 * @param _DS                   cpu data segment register
 * @param _ES                   cpu extra segment register
 * @param _BP                   cpu base pointer register
 *}
procedure NewMultiplex(_Flags, _CS, _IP, _AX, _BX, _CX, _DX,
  _SI, _DI, _DS, _ES, _BP: Word); interrupt;
begin
  asm CLI end;
  with Reg do begin                      { get registers }
    Flags:= _Flags;
    Ax:= _AX; Bx:= _BX; Cx:= _CX; Dx:= _DX;
    Si:= _SI; Di:= _DI;
    Ds:= _DS; Es:= _ES;
    Bp:= _BP;
  end;

  Inc(Switch);                           { switch the stack context }
  if Switch = 0 then begin
    OldSS:= SSeg;
    OldSP:= SPtr;
    asm
      MOV  SS,MySS
      MOV  SP,MySP
    end;
  end;
  asm STI end;

  if Reg.Ah = MyTask then begin
    case Reg.Al of
      $0:                                { get your presence and }
        begin                            { identity string }
          Reg.Al:= $FF;
          Reg.Es:= Seg(Copyright);
          Reg.Bx:= Ofs(Copyright);
        end;
      $C0:                               { unload itself }
        Reg.Ax:= Terminate;
    end;
  end else
    CallOldIntr(OldMultiplex, Reg);

  asm CLI end;
  Dec(Switch);                           { switch the stack context }
  if Switch < 0 then begin
    asm
      MOV  SS,OldSS
      MOV  SP,OldSP
    end;
  end;
  with Reg do begin                      { set registers }
    _Flags:= Flags;
    _AX:= Ax; _BX:= Bx; _CX:= Cx; _DX:= Dx;
    _SI:= Si; _DI:= Di;
    _DS:= Ds; _ES:= Es;
    _BP:= Bp;
  end;
  asm STI end;
end;

{*
 * Terminate the resident part
 *
 * @return terminate status; non zero if terminate, otherwise zero
 *}
function Terminate: Word;
var
  CanDo: Boolean;
  Com, Mlp: Pointer;
begin
                                         { get current interrupt handler }
  GetIntVec(ComIntr, Com);               { addresses }
  GetIntVec($2F, Mlp);
                                         { check match for current handlers }
  CanDo:= (Com = Addr(NewCom)) and
          (Mlp = Addr(NewMultiplex));

  if CanDo then begin
    Active:= False;                      { remove activity }
    asm CLI end;
    SetIntVec(ComIntr, OldCom);          { set old interrupt handlers }
    SetIntVec($2F, OldMultiplex);
    asm STI end;

    Reg.Ah:= $49;                        { remove itself from memory }
    Reg.Es:= PrefixSeg;
    MsDos(Reg);
  end;
  Terminate:= Word(CanDo);
end;

{$F-}

{********** non resident part ************************************}

{*
 * Prints the syntax.
 *}
procedure Usage;
begin
  Writeln('Usage: ', ExeName, ' [-uqh?L iPORT,IRQ,BAUD,DATA,PARITY,STOP]');
  Writeln;
  Writeln('Options:');
  Writeln('  u        remove program from memory');
  Writeln('  q        be quiet');
  Writeln('  h, ?     display this help screen');
  Writeln('  L        display software licence');
  Writeln;
  Writeln('  iPORT,IRQ,BAUD,DATA,PARITY,STOP');
  Writeln('   PORT    port address (hexadecimal)');
  Writeln('   IRQ     interrupt request: 3, 4, 5, 9, 10, 11, 12 or 15');
  Writeln('   BAUD    baud rate (decimal)');
  Writeln('   DATA    data bits: 5, 6, 7 or 8');
  Writeln('   PARITY  parity bit: (n)one, (e)ven or (o)dd, (m)ark, (s)pace');
  Writeln('   STOP    stop bits: 1 or 2');
  Writeln;
  Writeln('Default: -i3F8,4,9600,8,N,1');
  Halt(1);
end;

const
 BaudRate: Word = $0C;                  { Baud Rate: 9600 }
 DataBits: Byte = 3;                    { Data: 8 bits }
 StopBits: Byte = 0;                    { Stop: 1 bit }
 Parity: Byte = 0;                      { Parity: none }

 ComMask: Byte = $EF;                   { interrupt PIC mask - IRQ 4 }
 IRQmask: Byte = $21;                   { interrupt PIC OCW1 port - mask }

{*
 * Set parametes for serial line.
 *
 * @param Args                  arguments
 *}
procedure Params(Args: String);
const
 SEP = ',';                             { arguments separator }
 SLIPadd = $80;                         { slave interrupt PIC port adder }
var
  I, Code: Integer;
  V: LongInt;
  S: String;
begin
  for I:= 1 to Length(Args) do Args[I]:= UpCase(Args[I]);
  for I:= 0 to GetParseCount(Args, SEP)-1 do begin
    if I > 6 then Break;                 { only first 6 arguments }
    S:= GetParseString(I, Args, SEP);
    case I of
      0:                                 { serial port }
        begin
          V:= Hex2Int(S);
          if V = -1 then begin
            OptArgErrMsg(ExeName, True);
            Halt(2);
          end;
          ComPort:= V;
        end;
      1:                                 { interrupt request }
        begin
          Val(S, V, Code);
          if Code <> 0 then begin
            OptArgErrMsg(ExeName, True);
            Halt(2);
          end;
          if not(V in [3, 4, 5, 9, 10, 11, 12, 15]) then begin
            OptArgErrMsg(ExeName, True);
            Halt(2);
          end;
          if V > 8 then begin
            Inc(IRQctrl, SLIPadd);
            Inc(IRQmask, SLIPadd);
            ComIntr:= V + $68;
            ComMask:= not(1 shl (V-8));
          end else begin
            ComIntr:= V + 8;
            ComMask:= not(1 shl V);
          end;
        end;
      2:                                 { baud rate }
        begin
          Val(S, V, Code);
          if Code <> 0 then begin
            OptArgErrMsg(ExeName, True);
            Halt(2);
          end;
          if (V < 1) or (V > 115200) then begin
            OptArgErrMsg(ExeName, True);
            Halt(2);
          end;
          BaudRate:= 115200 div V;
        end;
      3:                                 { data bits }
        begin
          Val(S, V, Code);
          if Code <> 0 then begin
            OptArgErrMsg(ExeName, True);
            Halt(2);
          end;
          if (V < 5) or (V > 8) then begin
            OptArgErrMsg(ExeName, True);
            Halt(2);
          end;
          DataBits:= V-5;
        end;
      4:                                 { parity }
        begin
          if S = 'N' then Parity:= 0
          else if S = 'O' then Parity:= 1
          else if S = 'E' then Parity:= 3
          else if S = 'M' then Parity:= 5
          else if S = 'S' then Parity:= 7
          else begin
            OptArgErrMsg(ExeName, True);
            Halt(2);
          end;
        end;
      5:                                 { stop bits }
        begin
          Val(S, V, Code);
          if Code <> 0 then begin
            OptArgErrMsg(ExeName, True);
            Halt(2);
          end;
          if (V < 1) or (V > 2) then begin
            OptArgErrMsg(ExeName, True);
            Halt(2);
          end;
          StopBits:= V-1;
        end;
    end;
  end;
end;

{*
 * Initialize the resident part.
 *}
procedure Install;
begin
  Switch:= -1;                           { set default values }
  Active:= False;

  asm CLI end;
  MySS:= SSeg;                           { get stack pointers for handlers }
  MySP:= SPtr;
  GetIntVec(ComIntr, OldCom);            { get current interrupt handler }
  GetIntVec($2F, OldMultiplex);          { addresses }
  asm STI end;

                                         { initialize the serial PIC }
  Port[ComPort+3]:= $80;                 { set DLAB }
  Port[ComPort]:= Lo(BaudRate);          { set communication baud rate }
  Port[ComPort+1]:= Hi(BaudRate);
                                         { set parity, stop, data bits }
  Port[ComPort+3]:= (DataBits or (StopBits shl 2) or (Parity shl 3));
  Port[ComPort+1]:= 1;                   { enable interrupt PIC when data aviable }
  Port[ComPort+4]:= $0B;                 { set OUT2, RTS, DTR }

  while (Port[ComPort+5] and 1) <> 0 do  { cleanup receive serial buffer }
    Data:= Port[ComPort];                { ! active wait -> never ending loop }

  asm CLI end;
  SetIntVec(ComIntr, Addr(NewCom));      { set the new interrupt handlers }
  SetIntVec($2F, Addr(NewMultiplex));
  asm STI end;
                                         { initialize the interrupt PIC }
  Port[IRQmask]:= Port[IRQmask] and ComMask;

  Reg.Ah:= $49;                          { free enviroments }
  Reg.Es:= MemW[PrefixSeg : $2C];
  MsDos(Reg);

  Active:= True;                         { set activity }
end;

var
  I: Byte;
  UnInstall, Quiet: Boolean;
  Args: String;

begin
  Write(Copyright, ' ', Version, ', ');
  Writeln('Copyright (C) 1999-2002 Henrich Fukna');
  LicenceLine(Copyright, ExeName);
  Writeln;

  UnInstall:= False;
  Quiet:= False;

  I:= 1;
  while I <= ParamCount do begin
    Args:= ParamStr(I);
    if (Args[1] = '-') then begin
      Delete(Args, 1, 1);
      while Length(Args) > 0 do begin
        case Args[1] of
          '?',
          'h': Usage;
          'L':
            begin
              Licence;
              Halt(1);
            end;
          'u': UnInstall:= True;
          'q': Quiet:= True;
          'i':
            begin
              Params(Copy(Args, 2, Length(Args)-1));
              Args:= '';
            end;
          else begin
            OptArgErrMsg(ExeName, False);
            Halt(2);
          end;
        end;
        Delete(Args, 1, 1);
      end;
    end else begin
      OptArgErrMsg(ExeName, True);
      Halt(2);
    end;
    Inc(I);
  end;

  I:= $FF;                               { find first free multiplex service }
  MyTask:= 0;
  while (I > $7F) do begin
    Reg.Ah:= I;
    Reg.Al:= 0;
    Intr($2F, Reg);
    if Reg.Al = $FF then begin           { check presence }
      if StrComp(Addr(Copyright), Ptr(Reg.Es, Reg.Bx)) then
        if UnInstall then begin          { uninstall request }
          Reg.Ah:= I;
          Reg.Al:= $C0;
          Intr($2F, Reg);
          if (Reg.Ax) = 0 then begin
            if not(Quiet) then
              Writeln('Resident can not be unloaded.');
            Halt(3);
          end else begin
            if not(Quiet) then
              Writeln('Resident was unloaded.');
            Halt(0);
          end;
        end else begin
          if not(Quiet) then
            Writeln('Resident is already loaded.');
          Halt(4);
        end;
    end else
      if (MyTask = 0) and (Reg.Al = 0) then MyTask:= I;
    Dec(I);
  end;
  if UnInstall then begin                { resident part not present }
    if not(Quiet) then
      Writeln('Resident is not in memory.');
    Halt(4);
  end;
  if MyTask = 0 then begin               { no free multiplex service }
    if not(Quiet) then
      Writeln('Resident can not be loaded.');
    Halt(3);
  end;

  if not(Quiet) then begin               { print the values }
    Writeln('Resident was loaded.');

    Writeln('Port adress: ', Int2Hex(ComPort));
    Write('Interrupt request: ');
    if ComIntr > $70 then
      Writeln(ComIntr - $68)
    else
      Writeln(ComIntr - 8);
    Writeln('Baud rate: ', 115200 div BaudRate);
    Writeln('Data bits: ', DataBits + 5);
    Writeln('Stop bits: ', StopBits + 1);
    Write('Parity: ');
    case Parity of
      0: Writeln('None');
      1: Writeln('Odd');
      3: Writeln('Even');
      5: Writeln('Mark');
      7: Writeln('Space');
    end;
  end;

  Install;                               { initialize TSR }
  Keep(0);                               { keep resident }
end.