/* spc.c - Secure Partition Configurator
 * Author: Emil LAURENTIU
 * Date: 25 March 1998
 * Last modified: 29 August 1998
 *
 * Not a very portable source :) a mixture of C and asm
 * very compiler specific (which happens to be Watcom C)
 * This software is postcard-ware (or freeware if you prefer).
 * (eventually you can write me an email to show your gratitude :)
 * You can use it in any way you want as long as you
 * give me credit and do not remove my copyright notice
 * Please note that idea, des, blowfish, arcfour and tss
 * are copyrighted by their authors (those are not free)
 */

#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include <malloc.h>
#include <string.h>
#include <dos.h>
#include <ctype.h>
#include "machine.h"
#include "getopt.h"

#define	 MAX_ENTRIES	  8
#define	 OPER		  0x77
#define	 SIGNATURE	  '1'
#define	 ENC_TYPE	  '2'
#define	 SET_KEY	  '3'
#define	 DESTROY_CONTEXT  '4'
#define	 GET_DRIVE	  '5'
#define	 SET_DRIVE	  '6'
#define	 SET_LOW_LIMIT	  '7'
#define	 SET_HIGH_LIMIT	  '8'
#define	 REM_DRIVE	  '9'

#define printf if(!sw.quiet) printf

typedef struct
{
  word32	sect;
  word16	nsecs;
  char far     *buffer;
}	      abs_disk_t;

struct switch_t
{
  unsigned int	pass:1;
  unsigned int	forget:1;
  unsigned int	disk:1;
  unsigned int	safe_mode:1;
  unsigned int	remove:1;
  unsigned int	encrypt:1;
  unsigned int	decrypt:1;
  unsigned int	verbose:1;
  unsigned int	list:1;
  unsigned int	quiet:1;
};

byte far     *drive_table;
int	      s_cyl, s_head, s_sect, e_cyl, e_head, e_sect;
int	      sec_track, heads, tot_cyl;

#pragma aux asm_mk_fp = value [bx es];
extern char far *asm_mk_fp( void );
extern byte far *asm_int2f( void );
extern int    asm_read_absolute( void );
extern void   asm_flush_disks( void );
extern int    asm_int13( int oper );
extern void   asm_disk_req_return( void );
extern int
diskreq( byte oper, byte suboper, byte drive, byte head,
	 byte track, byte sector, word16 off, word16 seg );

#pragma aux disk_req parm[] value[ax] modify[ax bx cx dx es];
int
disk_req( byte oper, byte suboper, byte drive, byte head,
	  byte track, byte sector, word16 off, word16 seg )
{
#pragma aux diskreq = \
  "les  bx, dword ptr [off]" \
  "int  13h" \
  parm [ah] [al] [dl] [dh] [ch] [cl] caller value[ax] modify[bx es];

  return diskreq( oper, suboper, drive, head, track, sector, off, seg );
}

static void
set_passwd(  )
{
  int		i;
  char		c;
  char		line[80];

  fprintf( stderr, "Password: " );
  i = 0;
  while ( ( c = getch(	) ) != '\r' )
  {
    if ( c != '\b' )
    {
      line[i++] = c;
      fputc( '*', stderr );
    }
    else
    {
      if ( i > 0 )
      {
	i--;
	fputc( c, stderr );
	fputc( ' ', stderr );
	fputc( c, stderr );
      }
    }
  }
  line[i] = 0;
  disk_req( OPER, DESTROY_CONTEXT, 0, 0, 0, 0, 0, 0 );
  if ( i != 0 )
  {
    disk_req( OPER, SET_KEY, 0, 0, 0, 0, FP_OFF( line ), FP_SEG( line ) );
  }
  fputc( '\n', stderr );
}

static void
usage( char *progname )
{
  fprintf( stderr, "\nUsage: %s [-srledpfvq] drive\n", progname );
  fprintf( stderr, "\t-s put drive in safe mode\n" );
  fprintf( stderr, "\t-r remove a drive from list\n" );
  fprintf( stderr, "\t-l list all drives\n" );
  fprintf( stderr, "\t-e encrypt drive/partition\n" );
  fprintf( stderr, "\t-d decrypt drive/partition\n" );
  fprintf( stderr, "\t-p set passphrase\n" );
  fprintf( stderr, "\t-f forget disk passphrase\n" );
  fprintf( stderr, "\t-v verbose (print encrypted disk table)\n" );
  fprintf( stderr, "\t-q quiet (no output)\n" );
  exit( 0 );
}

static void
drive_data_table( int drive, int *p_log_drive, int *p_ph_drive, int display )
{
  div_t		mega_size;
  word32	first_sector, end_sector, total_sector;

  /* get drive data table */
#pragma aux asm_int2f = \
  "push ds" \
  "mov ax, 803h" \
  "int 2fh" \
  "mov dx, ds" \
  "mov ax, di" \
  "pop ds" \
  value [ax dx] modify [dx di];
  drive_table = asm_int2f(  );
  if ( display )
    fprintf( stderr, "%2s%s%10s%s%13s%s%14s%s%7s%s\n", "", "Drive", "",
		     "Total", "", "Start", "", "End", "", "Capacity");
  do
  {
    *p_ph_drive = *( drive_table + 4 );
    *p_log_drive = *( drive_table + 5 );
    if ( !( *( int far * ) ( drive_table + 0x1f ) & 0x40 ) )
    {
      /* removable media */
      disk_req( 8, 0, *p_ph_drive, 0, 0, 0, 0, 0 );
#pragma aux asm_disk_req_return = \
      "mov al, dh" \
      "cbw" \
      "mov word ptr [heads], ax" \
      "mov ax, cx" \
      "and ax, 3fh" \
      "mov word ptr [sec_track], ax" \
      "xchg cl, ch" \
      "shr ch, 6" \
      "mov word ptr [tot_cyl], cx"
      asm_disk_req_return(  );
      heads++;
      tot_cyl++;
      first_sector = 0L;
      total_sector = ( word32 ) sec_track *heads * tot_cyl;
      *( ( int far * ) ( drive_table + 0x13 ) ) = sec_track;
      *( ( int far * ) ( drive_table + 0x15 ) ) = heads;
      *( ( int far * ) ( drive_table + 0x25 ) ) = tot_cyl;
      *( ( word32 far * ) ( drive_table + 0x17 ) ) = first_sector;
      *( ( word16 far * ) ( drive_table + 0x0e ) ) = ( word16 ) total_sector;
    }
    else
    {
      sec_track = *( ( int far * ) ( drive_table + 0x13 ) );
      heads = *( ( int far * ) ( drive_table + 0x15 ) );
      tot_cyl = *( ( int far * ) ( drive_table + 0x25 ) );
      first_sector = *( ( word32 far * ) ( drive_table + 0x17 ) );
      total_sector = ( word32 ) * ( ( word16 far * ) ( drive_table + 0x0e ) );
      if ( !total_sector )
	total_sector = *( ( word32 far * ) ( drive_table + 0x1b ) );
    }
    end_sector = first_sector + total_sector - 1;
    s_cyl = first_sector / sec_track / heads;
    first_sector -= ( word32 ) s_cyl *sec_track * heads;
    s_head = first_sector / sec_track;
    s_sect = first_sector % sec_track + 1;
    e_cyl = end_sector / sec_track / heads;
    end_sector -= ( word32 ) e_cyl *sec_track * heads;
    e_head = end_sector / sec_track;
    e_sect = end_sector % sec_track + 1;
    mega_size = div( ( total_sector + 100 ) / 200, 10 );
    if ( display )
      fprintf( stderr, "%c: (0x%02X)  %4d / %3d / %2d   "
	       "%4d / %3d / %2d   %4d / %3d / %2d   %4d.%dM\n",
	       'A' + *p_log_drive, *p_ph_drive, tot_cyl, heads, sec_track,
	       s_cyl, s_head, s_sect, e_cyl, e_head, e_sect,
	       mega_size.quot, mega_size.rem );
    else if ( *p_log_drive == drive )
      break;
  } while ( FP_OFF( drive_table = *( ( byte far * far * ) drive_table ) )
	    != 0xffff );
}

static void
walk_partition( int ph_drive )
{
  int		c_cyl, c_head, c_sect, n_sect;
  int		tot_cyls;
  char far     *b64k;
  word16	segment;
  char far     *buffer;

  /* allocate 64K of buffer - to read a maximum of 63 sectors */
  b64k = _fmalloc( ( 2 * 63 + 1 ) * 0x200 );
  if ( b64k == NULL )
  {
    fprintf( stderr, "Sorry cannot allocate a 64k buffer\n" );
    exit( -1 );
  }
  /*
   * a hack to get a non boundary cross 0x7e00 buffer DMA cannot work with
   * addresses across segments
   */
  segment = FP_SEG( b64k ) + ( FP_OFF( b64k ) >> 4 );
  if ( ( segment & 0xfff ) < 0x7f0 )
    buffer = MK_FP( segment & 0xf000, 0x8000 );
  else
    buffer = MK_FP( ( segment & 0xf000 ) + 0x1000, 0 );

  tot_cyls = ( e_cyl - s_cyl ) * heads + e_head - s_head;
  c_cyl = s_cyl;
  c_head = s_head;
  c_sect = s_sect;
  n_sect = sec_track - s_sect + 1;

#pragma aux asm_int13 = \
  "mov  ah, al" \
  "mov  al, byte ptr [n_sect]" \
  "mov  dl, byte ptr [ph_drive]" \
  "mov  dh, byte ptr [c_head]" \
  "mov  cx, word ptr [c_cyl]" \
  "xchg ch, cl" \
  "shl  cl, 6" \
  "or   cl, byte ptr [c_sect]" \
  "les  bx, dword ptr [buffer]" \
  "int  13h" \
  "mov  ax, 0" \
  "jnc  after" \
  "mov  ax, -1" \
"after:" \
  parm[ax] value[ax] modify[ax bx cx dx es];

  while ( 1 )
  {
    fprintf( stderr, "\rCyl: %4d Head: %3d %3d%%", c_cyl, c_head,
	100L * ( ( c_cyl - s_cyl ) * heads + c_head - s_head ) / tot_cyls );

    if ( asm_int13( READ ) == -1 )
    {
      fprintf( stderr, "Error while reading drive 0x%02X cylinder %d head "
	       "%d\n", ph_drive, c_cyl, c_head );
      exit( -1 );
    }
    if ( asm_int13( WRITE ) == -1 )
    {
      fprintf( stderr, "Error while writing drive 0x%02X cylinder %d head "
	       "%d\n", ph_drive, c_cyl, c_head );
      exit( -1 );
    }
    c_head++;
    c_sect = 1;
    n_sect = sec_track;
    if ( c_head >= heads )
    {
      c_head = 0;
      c_cyl++;
    }
    if ( c_cyl > e_cyl )
      break;
    else if ( c_cyl == e_cyl )
    {
      if ( c_head == e_head )
	n_sect = e_sect;
      else if ( c_head > e_head )
	break;
    }
  }
  _ffree( b64k );
}

static int
get_drive( int drive, int *p_log_drive, int *p_ph_drive )
{
  char far     *buffer;

  drive_data_table( drive, p_log_drive, p_ph_drive, 0 );
  if ( *p_log_drive != drive )
  {
    fprintf( stderr, "Invalid requested drive %c:\n", 'A' + drive );
    exit( -1 );
  }
  buffer = _fmalloc( 0x200 );
  if ( buffer == NULL )
  {
    fprintf( stderr, "Alloc error\n" );
    exit( -1 );
  }
  /* read boot sector */
#pragma aux asm_read_absolute = \
  "mov  ax, 201h" \
  "mov  bx, word ptr [p_ph_drive]" \
  "mov  dl, [bx]" \
  "mov  dh, byte ptr [s_head]" \
  "mov  cx, word ptr [s_cyl]" \
  "xchg ch, cl" \
  "shl  cl, 6" \
  "or   cl, byte ptr [s_sect]" \
  "les  bx, dword ptr [buffer]" \
  "int  13h" \
  "mov  ax, 0" \
  "jnc  after" \
  "mov  ax, -1" \
"after:" \
  parm[ax] value[ax] modify[ax bx cx dx es];
  if ( asm_read_absolute(  ) != 0 )
  {
    fprintf( stderr, "Cannot access boot sector in drive %c:\n",
	     'A' + drive );
    exit( -1 );
  }
  _ffree( buffer );
  /* flush disks */
#pragma aux asm_flush_disks = \
  "mov ah, 0dh" \
  "int 21h";
  asm_flush_disks(  );
  return drive;
}

static int
insert_drive( int log_drive, int ph_drive, int flag )
{
  int		entry;

  for ( entry = 0; entry < MAX_ENTRIES; entry++ )
  {
    if ( ( disk_req( OPER, GET_DRIVE, 0, 0, 0, 0, entry, 0 ) & 0x1fff ) ==
	 ( log_drive << 8 | ph_drive ) )
      break;
  }
  if ( entry == MAX_ENTRIES )	/* find a free entry */
    for ( entry = 0; entry < MAX_ENTRIES; entry++ )
    {
      if ( disk_req( OPER, GET_DRIVE, 0, 0, 0, 0, entry, 0 ) == -1 )
	break;
    }
  if ( entry == MAX_ENTRIES )	/* table full */
  {
    fprintf( stderr, "\nTable full\n" );
    exit( -1 );
  }
  disk_req( OPER, SET_LOW_LIMIT, ( byte ) s_cyl, s_cyl >> 8,
	    s_head, s_sect, entry, 0 );
  disk_req( OPER, SET_HIGH_LIMIT, ( byte ) e_cyl, e_cyl >> 8,
	    e_head, e_sect, entry, 0 );
  disk_req( OPER, SET_DRIVE, ph_drive, log_drive | flag,
	    heads, sec_track, entry, 0 );
  return entry;
}

static int
remove_drive( int log_drive )
{
  int		entry;

  for ( entry = 0; entry < MAX_ENTRIES; entry++ )
  {
    if ( ( disk_req( OPER, GET_DRIVE, 0, 0, 0, 0, entry, 0 ) & 0x1f00 ) ==
	 ( log_drive << 8 ) )
      break;
  }
  if ( entry == MAX_ENTRIES )
    return -1;			/* not found */
  disk_req( OPER, REM_DRIVE, 0, log_drive, 0, 0, entry, 0 );
  return entry;
}

int
main( int argc, char *argv[] )
{
  int		i;
  char		progname[9];
  int		log_drive, ph_drive;
  int		drive;
  int		is_pass;
  struct switch_t sw;

  setbuf( stdout, NULL );
  setbuf( stderr, NULL );
  _splitpath( argv[0], NULL, NULL, progname, NULL );

  *( unsigned int * ) ( &sw ) = 0;
  while ( ( i = getopt( argc, argv, "edlpfsrvqh?" ) ) != EOF )
  {
    switch ( i )
    {
      case 'e':
	sw.encrypt = 1;
	break;
      case 'd':
	sw.decrypt = 1;
	break;
      case 'l':
	sw.list = 1;
	break;
      case 'p':
	sw.pass = 1;
	break;
      case 'f':
	sw.forget = 1;
	break;
      case 's':
	sw.safe_mode = 1;
	break;
      case 'r':
	sw.remove = 1;
	break;
      case 'v':
	sw.verbose = 1;
	break;
      case 'q':
	sw.quiet = 1;
	break;
      case 'h':
      case '?':
	usage( progname );
	break;
      default:
	fprintf( stderr, "Invalid option %c", i );
	return -1;
    }
  }
  printf( "\n%s - Secure Partition Configurator - "
	  "EmilSoft(C)1998\n\n", progname );
  if ( _osmajor < 4 )		/* get dos version */
  {
    fprintf( stderr, "DOS version 4 or higher required\n" );
    exit( -1 );
  }
  i = disk_req( OPER, SIGNATURE, 0, 0, 0, 0, 0, 0 );
  if ( ( i | 1 ) != 0x7531 )
  {
    printf( "Encryption daemon not loaded\n" );
    return -1;
  }
  is_pass = i & 1;
  disk_req( OPER, ENC_TYPE, 0, 0, 0, 0, 0, 0 );
  printf( "Daemon encryption type: %Fs\n", asm_mk_fp(  ) );
  if ( sw.list )
  {
    drive_data_table( drive, &log_drive, &ph_drive, 1 );
    return 0;
  }
  if ( sw.pass )
  {
    set_passwd(	 );
    is_pass = 1;
  }
  if ( sw.forget )
  {
    printf( "Forgeting passphrase\n" );
    disk_req( OPER, DESTROY_CONTEXT, 0, 0, 0, 0, 0, 0 );
    is_pass = 0;
  }
  if ( optind == argc )
  {
    if ( !sw.verbose )
    {
      if ( sw.pass || sw.forget )
	return 0;
      else
	usage( progname );
    }
  }
  else
  {
    if ( !sw.safe_mode && !sw.remove && !sw.encrypt && !sw.decrypt )
      sw.disk = 1;
    drive = toupper( *argv[optind] ) - 'A';
    get_drive( drive, &log_drive, &ph_drive );
    while ( 1 )
    {
      if ( sw.safe_mode )
      {
	printf( "\nPuting drive %c: in safe mode\n", 'A' + drive );
	insert_drive( log_drive, ph_drive, 0x80 );
	break;
      }
      if ( sw.remove )
      {
	printf( "\nDrive %c: ", 'A' + drive );
	if ( remove_drive( log_drive ) != -1 )
	  printf( "removed\n" );
	else
	  printf( "not found in the encrypted disk table\n" );
	break;
      }
      if ( !is_pass )
      {
	set_passwd(  );
	is_pass = 1;
      }
      if ( sw.disk )
      {
	insert_drive( log_drive, ph_drive, 0 );
	printf( "\nDisk %c: activated\n", 'A' + log_drive );
	break;
      }
      if ( sw.encrypt )
      {
	fprintf( stderr, "\nConfirm drive %c: encryption [N/y] ",
		 'A' + drive );
	if ( tolower( getch(  ) ) != 'y' )
	  break;
	fprintf( stderr, "\nEncrypting drive %c: ...\n", 'A' + drive );
	insert_drive( log_drive, ph_drive, 0x40 );
	walk_partition( ph_drive );
	insert_drive( log_drive, ph_drive, 0 );
	break;
      }
      if ( sw.decrypt )
      {
	fprintf( stderr, "\nConfirm drive %c: decryption [N/y] ",
		 'A' + drive );
	if ( tolower( getch(  ) ) != 'y' )
	  break;
	fprintf( stderr, "\nDecrypting drive %c: ...\n", 'A' + drive );
	insert_drive( log_drive, ph_drive, 0x20 );
	walk_partition( ph_drive );
	remove_drive( log_drive );
	break;
      }
    }
    /* strip "unformatted" marker for a drive */
    *( ( word16 far * ) ( drive_table + 0x23 ) ) &= ~0x200;
  }
  if ( sw.verbose )
  {
    printf( "\nEncrypted disk table:\n" );
    for ( i = 0; i < MAX_ENTRIES; i++ )
    {
      drive = disk_req( OPER, GET_DRIVE, 0, 0, 0, 0, i, 0 );
      if ( drive != -1 )
	printf( "Drive %c: (0x%02X) %s\n", 'A' + ( ( drive >> 8 ) & 0x1f ),
		( byte ) drive, ( drive & 0x8000 ) ? "inactive" : "active" );
    }
  }
  return 0;
}
