/* SEAL is designed by IBM, and is very fast. A patent is pending.
 * Anyone who wants to use the SEAL algorithm should check with IBM
 * (Director of Licences, IBM Corporation,
 * 500 Columbus Ave., Thurnwood, NY, 10594).
 *
 * Adapted for DOS s-e-p secure encrypted partitions (Linux compatible)
 * by Emil Laurentiu <emil@techteam.static.golden.net>
 * Date: Wednesday, 25 March 1998
 */

#include <string.h>
#include "machine.h"
#include "seal.h"

SealContext   seal_ctx;
word32        h[5];

#pragma aux encrypt_type value [es bx];
char far     *
encrypt_type( void )
{
  return "SEAL";
}

#pragma aux transfer parm [] caller modify [ax bx cx dx es];
int
transfer( int cmd, unsigned char far * buffer,
	  int size, unsigned long block )
{
  cmd = 0;
  seal_crypt( block, buffer, size );
  return 0;
}

#define ROT2(x) (((x) >> 2) | ((x) << 30))
#define ROT9(x) (((x) >> 9) | ((x) << 23))
#define ROT8(x) (((x) >> 8) | ((x) << 24))
#define ROT16(x) (((x) >> 16) | ((x) << 16))
#define ROT24(x) (((x) >> 24) | ((x) << 8))
#define ROT27(x) (((x) >> 27) | ((x) << 5))

#define F1(x,y,z) (((x) & (y)) | ((~(x)) & (z)))
#define F2(x,y,z) ((x)^(y)^(z))
#define F3(x,y,z) (((x) & (y)) | ((x) & (z)) | ((y) & (z)))
#define F4(x,y,z) ((x)^(y)^(z))

void
g( byte far * in, int i, word32 * h )
{
  word32        h0, h1, h2, h3, h4, a, b, c, d, e, w[80], temp;

  h0 = htonl( ( ( word32 * ) in )[0] );
  h1 = htonl( ( ( word32 * ) in )[1] );
  h2 = htonl( ( ( word32 * ) in )[2] );
  h3 = htonl( ( ( word32 * ) in )[3] );
  h4 = htonl( ( ( word32 * ) in )[4] );

  w[0] = i;
  for ( i = 1; i < 16; i++ )
    w[i] = 0;
  for ( i = 16; i < 80; i++ )
    w[i] = w[i - 3] ^ w[i - 8] ^ w[i - 14] ^ w[i - 16];

  a = h0;
  b = h1;
  c = h2;
  d = h3;
  e = h4;

  for ( i = 0; i < 20; i++ )
  {
    temp = ROT27( a ) + F1( b, c, d ) + e + w[i] + 0x5a827999;
    e = d;
    d = c;
    c = ROT2( b );
    b = a;
    a = temp;
  }
  for ( i = 20; i < 40; i++ )
  {
    temp = ROT27( a ) + F2( b, c, d ) + e + w[i] + 0x6ed9eba1;
    e = d;
    d = c;
    c = ROT2( b );
    b = a;
    a = temp;
  }
  for ( i = 40; i < 60; i++ )
  {
    temp = ROT27( a ) + F3( b, c, d ) + e + w[i] + 0x8f1bbcdc;
    e = d;
    d = c;
    c = ROT2( b );
    b = a;
    a = temp;
  }
  for ( i = 60; i < 80; i++ )
  {
    temp = ROT27( a ) + F4( b, c, d ) + e + w[i] + 0xca62c1d6;
    e = d;
    d = c;
    c = ROT2( b );
    b = a;
    a = temp;
  }
  h[0] = h0 + a;
  h[1] = h1 + b;
  h[2] = h2 + c;
  h[3] = h3 + d;
  h[4] = h4 + e;
}

void
set_key( unsigned char far * key )
{
  int           i;

  for ( i = 0; i < 510; i += 5 )
    g( key, i / 5, seal_ctx.t + i );
  g( key, 510 / 5, h );
  for ( i = 510; i < 512; i++ )
    seal_ctx.t[i] = h[i - 510];

  g( key, ( -1 + 0x1000 ) / 5, h );
  for ( i = 0; i < 4; i++ )
    seal_ctx.s[i] = h[i + 1];
  for ( i = 4; i < 254; i += 5 )
    g( key, ( i + 0x1000 ) / 5, seal_ctx.s + i );
  g( key, ( 254 + 0x1000 ) / 5, h );
  seal_ctx.s[254] = h[0];
  seal_ctx.s[255] = h[1];

  g( key, ( -2 + 0x2000 ) / 5, h );
  for ( i = 0; i < 3; i++ )
    seal_ctx.r[i] = h[i + 2];
  g( key, ( 3 + 0x2000 ) / 5, seal_ctx.r + 3 );
  g( key, ( 8 + 0x2000 ) / 5, seal_ctx.r + 8 );
  g( key, ( 13 + 0x2000 ) / 5, h );
  for ( i = 13; i < 16; i++ )
    seal_ctx.r[i] = h[i - 13];
}

void
destroy_context( void )
{
  memset( &seal_ctx, 0, sizeof( seal_ctx ) );
}

void
seal_crypt( word32 block, unsigned char far * buffer, unsigned int len )
{
  int           i, j, l, k;
  word32        seed, a, b, c, d, p, q, n1, n2, n3, n4;

  for ( l = 0; l < len / 0x100; l++ )
  {
    if ( ( l & 1 ) == 0 )
    {
      seed = ( block + l / 2 ) / 2;
      k = ( seed & 3 ) << 2;
      a = seed ^ seal_ctx.r[k];
      b = ROT8( seed ) ^ seal_ctx.r[k + 1];
      c = ROT16( seed ) ^ seal_ctx.r[k + 2];
      d = ROT24( seed ) ^ seal_ctx.r[k + 3];
    }
    for ( j = 0; j < 2; j++ )
    {
      p = a & 0x7fc;
      b += seal_ctx.t[p >> 2];
      a = ROT9( a );

      p = b & 0x7fc;
      c += seal_ctx.t[p >> 2];
      b = ROT9( b );

      p = c & 0x7fc;
      d += seal_ctx.t[p >> 2];
      c = ROT9( c );

      p = d & 0x7fc;
      a += seal_ctx.t[p >> 2];
      d = ROT9( d );
    }

    n1 = d;
    n2 = b;
    n3 = a;
    n4 = c;

    p = a & 0x7fc;
    b += seal_ctx.t[p >> 2];
    a = ROT9( a );

    p = b & 0x7fc;
    c += seal_ctx.t[p >> 2];
    b = ROT9( b );

    p = c & 0x7fc;
    d += seal_ctx.t[p >> 2];
    c = ROT9( c );

    p = d & 0x7fc;
    a += seal_ctx.t[p >> 2];
    d = ROT9( d );

    for ( i = 0; i < 256; i += 4 )
    {
      p = a & 0x7fc;
      b += seal_ctx.t[p >> 2];
      a = ROT9( a );
      b ^= a;

      q = b & 0x7fc;
      c ^= seal_ctx.t[q >> 2];
      b = ROT9( b );
      c += b;

      p = ( p + c ) & 0x7fc;
      d += seal_ctx.t[p >> 2];
      c = ROT9( c );
      d ^= c;

      q = ( q + d ) & 0x7fc;
      a += seal_ctx.t[q >> 2];
      d = ROT9( d );
      a += d;

      p = ( p + a ) & 0x7fc;
      b ^= seal_ctx.t[p >> 2];
      a = ROT9( a );

      q = ( q + b ) & 0x7fc;
      c += seal_ctx.t[q >> 2];
      b = ROT9( b );

      p = ( p + c ) & 0x7fc;
      d ^= seal_ctx.t[p >> 2];
      c = ROT9( c );

      q = ( q + d ) & 0x7fc;
      a += seal_ctx.t[q >> 2];
      d = ROT9( d );

      *buffer++ ^= b + seal_ctx.s[i];
      *buffer++ ^= c + seal_ctx.s[i + 1];
      *buffer++ ^= d + seal_ctx.s[i + 2];
      *buffer++ ^= a + seal_ctx.s[i + 3];

      if ( i & 4 )
      {
	a += n3;
	c += n4;
      }
      else
      {
	a += n1;
	c += n2;
      }
    }
  }
}
