/* idea.c
 * Author: Tatu Ylonen <ylo@cs.hut.fi>
 * Copyright (c) 1995 Tatu Ylonen <ylo@cs.hut.fi>, Espoo, Finland
 *		      All rights reserved
 * Created: Sun Jun 25 02:59:39 1995 ylo
 *
 * This code is based on Xuejia Lai: On the Design and Security of Block
 * Ciphers, ETH Series in Information Processing, vol. 1, Hartung-Gorre
 * Verlag, Konstanz, Switzerland, 1992.	 Another source was Bruce
 * Schneier: Applied Cryptography, John Wiley & Sons, 1994.
 *
 * The IDEA mathematical formula may be covered by one or more of the
 * following patents: PCT/CH91/00117, EP 0 482 154 B1, US Pat. 5,214,703.
 *
 * Adapted for DOS s-e-p secure encrypted partitions (Linux compatible)
 * by Emil Laurentiu <emil@techteam.static.golden.net>
 * Date: Wednesday, 25 March 1998
 */

#include <string.h>
#include "machine.h"
#include "idea.h"

IDEAContext   idea_ctx;
word32	      out[2];

#pragma aux mulop parm [CX BX] caller modify [ax bx cx dx];
word32	      mulop( word16, word16 );

#pragma aux encrypt_type value [es bx];
char far     *
encrypt_type( void )
{
  return "IDEA";
}

#pragma aux transfer parm [] caller modify [ax bx cx dx es];
int
transfer( int cmd, unsigned char far * buffer,
	  unsigned int size, word32 block )
{
  if ( cmd == READ )
  {
    idea_cfb_decrypt( block, buffer, size );
  }
  else
  {
    idea_cfb_encrypt( block, buffer, size );
  }
  return 0;
}

/* Sets idea key for encryption. */
#pragma aux set_key parm [BX ES] caller modify [ax bx cx dx es];
void
set_key( unsigned char far * key )
{
  int		i;
  word16       *keys;
  unsigned char *d;

  /* Keys for the first round are directly taken from the user-supplied key. */
  d = ( unsigned char * ) idea_ctx.key_schedule;
  for ( i = 0; i < 16; i++ )
  {
    if ( *key == 0 )
      break;
    *d++ = *key++;
  }
  /* Get pointer to the keys. */
  keys = idea_ctx.key_schedule;

  /*
   * Each round uses the key of the previous key, rotated to the left by 25
   * bits.  The last four keys (output transform) are the first four keys
   * from what would be the ninth round.
   */
  for ( i = 8; i < 52; i++ )
  {
    if ( ( i & 7 ) == 0 )
      keys += 8;
    keys[i & 7] = ( ( keys[( ( i + 1 ) & 7 ) - 8] << 9 ) |
		    ( keys[( ( i + 2 ) & 7 ) - 8] >> 7 ) ) & 0xffff;
  }
}

/* Destroys any sensitive data in the context. */

void
destroy_context( void )
{
  memset( &idea_ctx, 0, sizeof( idea_ctx ) );
}

/* Performs the IDEA cipher transform on a block of data. */
static void
idea_transform( word32 l, word32 r, word32 * output )
{
  unsigned int	round;
  word16       *keys;
  word32	t1, t2, x1, x2, x3, x4;

  keys = idea_ctx.key_schedule;
  x1 = l >> 16;
  x2 = l;
  x3 = r >> 16;
  x4 = r;
  for ( round = 0; round < 8; round++ )
  {
    x1 = mulop( x1 & 0xffff, keys[0] );
    x3 = x3 + keys[2];
    x4 = mulop( x4 & 0xffff, keys[3] );
    x2 = x2 + keys[1];
    t1 = x1 ^ x3;
    t2 = x2 ^ x4;
    t1 = mulop( t1 & 0xffff, keys[4] );
    t2 = t1 + t2;
    t2 = mulop( t2 & 0xffff, keys[5] );
    t1 = t1 + t2;
    x1 = x1 ^ t2;
    x4 = x4 ^ t1;
    t1 = t1 ^ x2;
    x2 = t2 ^ x3;
    x3 = t1;
    keys += 6;
  }

  x1 = mulop( x1 & 0xffff, keys[0] );
  x3 = ( x2 + keys[2] ) & 0xffff;
  x2 = t1 + keys[1];		/* t1 == old x3 */
  x4 = mulop( x4 & 0xffff, keys[3] );
  output[0] = ( x1 << 16 ) | ( x2 & 0xffff );
  output[1] = ( x3 << 16 ) | ( x4 & 0xffff );
}

/* Encrypts len bytes from src to dest in CFB mode.
   iv will be modified at end to a value suitable for
   continuing encryption. */
void
idea_cfb_encrypt( word32 block, unsigned char far * data, unsigned int len )
{
  word32	iv0, iv1;
  unsigned int	i;

  for ( i = 0; i < len; i += 8 )
  {
    if ( i % 0x200 == 0 )
    {
      iv0 = ( block + i / 0x200 ) / 2;
      iv1 = 0L;
    }
    idea_transform( iv0, iv1, out );
    iv0 = out[0] ^ GET_32BIT_LSB_FIRST( data + i );
    iv1 = out[1] ^ GET_32BIT_LSB_FIRST( data + i + 4 );
    PUT_32BIT_LSB_FIRST( data + i, iv0 );
    PUT_32BIT_LSB_FIRST( data + i + 4, iv1 );
  }
}

/* Decrypts len bytes from src to dest in CFB mode.
   iv will be modified at end to a value suitable
   for continuing decryption. */
void
idea_cfb_decrypt( word32 block, unsigned char far * data, unsigned int len )
{
  word32	iv0, iv1, plain0, plain1;
  unsigned int	i;

  for ( i = 0; i < len; i += 8 )
  {
    if ( i % 0x200 == 0 )
    {
      iv0 = ( block + i / 0x200 ) / 2;
      iv1 = 0L;
    }
    idea_transform( iv0, iv1, out );
    iv0 = GET_32BIT_LSB_FIRST( data + i );
    iv1 = GET_32BIT_LSB_FIRST( data + i + 4 );
    plain0 = out[0] ^ iv0;
    plain1 = out[1] ^ iv1;
    PUT_32BIT_LSB_FIRST( data + i, plain0 );
    PUT_32BIT_LSB_FIRST( data + i + 4, plain1 );
  }
}
