
#include <string.h>
#include "pgp.h"
#include "block.h"
#include "randpool.h"

/*************************************************************************/

void blockCfbReinit(struct BlockCfbContext *context, byte const *iv)
{
    if (iv)
	memcpy(context->iv, iv, 8);
    else
	fill0(context->iv, 8);
    context->bufleft = 0;
}

void blockCfbInit(struct BlockCfbContext *context, byte const key[16], int Cipher)
{
    context->Cipher = Cipher;
    if (context->Cipher == IDEA_ALGORITHM_BYTE)
        ideaExpandKey(key, context->idea_key);
    else
    if (context->Cipher == CAST5_ALGORITHM_BYTE)
        cast_setkey(&context->C5ctx, key, 16);
    blockCfbReinit(context, 0);
}

void blockCfbDestroy(struct BlockCfbContext *context)
{
    burn(*context);
}

/*
 * Okay, explanation time:
 * Phil invented a unique way of doing CFB that's sensitive to semantic
 * boundaries within the data being encrypted.  One way to phrase
 * CFB en/decryption is to say that you XOR the current 8 bytes with
 * IDEA/CAST5(previous 8 bytes of ciphertext).  Normally, you repeat this
 * at 8-byte intervals, but Phil decided to resync things on the
 * boundaries between elements in the stream being encrypted.
 *
 * That is, the last 4 bytes of a 12-byte field are en/decrypted using
 * the first 4 bytes of IDEA/CAST5(previous 8 bytes of ciphertext), but then
 * the last 4 bytes of that IDEA/CAST5 computation are thrown away, and the
 * first 8 bytes of the next field are en/decrypted using
 * IDEA/CAST5(last 8 bytes of ciphertext).  This is equivalent to using a
 * shorter feedback length (if you're familiar with the general CFB
 * technique) briefly, and doesn't weaken the cipher any (using shorter
 * CFB lengths makes it stronger, actually), it just makes it a bit unusual.
 *
 * Anyway, to accomodate this behaviour, every time we do an IDEA/CAST5
 * encrpytion of 8 bytes of ciphertext to get 8 bytes of XOR mask,
 * we remember the ciphertext.  Then if we have to resync things
 * after having processed, say, 2 bytes, we refill the iv buffer
 * with the last 6 bytes of the old ciphertext followed by the
 * 2 bytes of new ciphertext stored in the front of the iv buffer.
 */
void blockCfbSync(struct BlockCfbContext *context)
{
    int bufleft = context->bufleft;

    if (bufleft) {
	memmove(context->iv + bufleft, context->iv, 8 - bufleft);
	memcpy(context->iv, context->oldcipher + 8 - bufleft, bufleft);
	context->bufleft = 0;
    }
}

/*
 * Encrypt a buffer of data, using IDEA/CAST5 in CFB mode.
 * There are more compact ways of writing this, but this is
 * written for speed.
 */
void blockCfbEncrypt(struct BlockCfbContext *context, byte const *src,
		    byte * dest, int count)
{
    int bufleft = context->bufleft;
    byte *bufptr = context->iv + 8 - bufleft;

    /* If there are no more bytes to encrypt that there are bytes
     * in the buffer, XOR them in and return.
     */
    if (count <= bufleft) {
	context->bufleft = bufleft - count;
	while (count--) {
	    *dest++ = *bufptr++ ^= *src++;
	}
	return;
    }
    count -= bufleft;
    /* Encrypt the first bufleft (0 to 7) bytes of the input by XOR
     * with the last bufleft bytes in the iv buffer.
     */
    while (bufleft--) {
	*dest++ = (*bufptr++ ^= *src++);
    }
    /* Encrypt middle blocks of the input by cranking the cipher,
     * XORing 8-byte blocks, and repeating until the count
     * is 8 or less.
     */
    while (count > 8) {
	bufptr = context->iv;
	memcpy(context->oldcipher, bufptr, 8);
	if (context->Cipher == IDEA_ALGORITHM_BYTE)
		ideaCipher(bufptr, bufptr, context->idea_key);
	else if (context->Cipher == CAST5_ALGORITHM_BYTE)
		cast_encrypt_block(&context->C5ctx, bufptr, bufptr);
	bufleft = 8;
	count -= 8;
	do {
	    *dest++ = (*bufptr++ ^= *src++);
	} while (--bufleft);
    }
    /* Do the last 1 to 8 bytes */
    bufptr = context->iv;
    memcpy(context->oldcipher, bufptr, 8);
	if (context->Cipher == IDEA_ALGORITHM_BYTE)
        ideaCipher(bufptr, bufptr, context->idea_key);
	else if (context->Cipher == CAST5_ALGORITHM_BYTE)
		cast_encrypt_block(&context->C5ctx, bufptr, bufptr);
    context->bufleft = 8 - count;
    do {
	*dest++ = (*bufptr++ ^= *src++);
    } while (--count);
}


/*
 * Decrypt a buffer of data, using IDEA/CAST5 in CFB mode.
 * There are more compact ways of writing this, but this is
 * written for speed.
 */
void blockCfbDecrypt(struct BlockCfbContext *context, byte const *src,
		    byte * dest, int count)
{
    int bufleft = context->bufleft;
    static byte *bufptr;
    byte t;

    bufptr = context->iv + (8 - bufleft);
    if (count <= bufleft) {
	context->bufleft = bufleft - count;
	while (count--) {
	    t = *bufptr;
	    *dest++ = t ^ (*bufptr++ = *src++);
	}
	return;
    }
    count -= bufleft;
    while (bufleft--) {
	t = *bufptr;
	*dest++ = t ^ (*bufptr++ = *src++);
    }
    while (count > 8) {
	bufptr = context->iv;
	memcpy(context->oldcipher, bufptr, 8);
	if (context->Cipher == IDEA_ALGORITHM_BYTE)
		ideaCipher(bufptr, bufptr, context->idea_key);
	else if (context->Cipher == CAST5_ALGORITHM_BYTE)
		cast_encrypt_block(&context->C5ctx, bufptr, bufptr);
		//cast_decrypt_block(&context->C5ctx, bufptr, bufptr);
	bufleft = 8;
	count -= 8;
	do {
	    t = *bufptr;
	    *dest++ = t ^ (*bufptr++ = *src++);
	} while (--bufleft);
    }
    bufptr = context->iv;
    memcpy(context->oldcipher, bufptr, 8);
	if (context->Cipher == IDEA_ALGORITHM_BYTE)
        ideaCipher(bufptr, bufptr, context->idea_key);
	else if (context->Cipher == CAST5_ALGORITHM_BYTE)
		cast_encrypt_block(&context->C5ctx, bufptr, bufptr);
		//cast_decrypt_block(&context->C5ctx, bufptr, bufptr);
    context->bufleft = 8 - count;
    do {
	t = *bufptr;
	*dest++ = t ^ (*bufptr++ = *src++);
    } while (--count);
}

/********************************************************************/

/*
 * Cryptographically strong pseudo-random-number generator.
 * The design is from Appendix C of ANSI X9.17, "Financial
 * Institution Key Management (Wholesale)", with CAST5
 * substituted for the DES.
 */

/*
 * Initialize a cryptographic random-number generator.
 * key and seed should be arbitrary.
 */
void blockRandInit(struct BlockRandContext *context, byte const key[16],
		  byte const seed[8])
{
    int i;

    cast_setkey(&context->C5ctx, key, 16);
    context->bufleft = 0;
    memcpy(context->internalbuf, seed, 8);
}


/*
 * Read out the RNG's state.
 */
void blockRandState(struct BlockRandContext *context, byte key[16], byte seed[8])
{
    int i;

    memcpy(seed, context->internalbuf, 8);
    for (i = 0; i < 8; i++) {
	/*key[2 * i] = context->key[i] >> 8;*/
	/*key[2 * i + 1] = context->key[i];*/
	key[2 * i] = context->C5ctx.Km[i] >> 8;
	key[2 * i + 1] = context->C5ctx.Km[i];
    }

}

/*
 * Encrypt the RNG's state with the given CFB encryptor.
 */
void blockRandWash(struct BlockRandContext *context, struct BlockCfbContext *cfb)
{
    byte keyseed[16 + 8];
    int i;

    blockRandState(context, keyseed, keyseed + 16);
    blockCfbEncrypt(cfb, keyseed, keyseed, 16 + 8);
    blockRandInit(context, keyseed, keyseed + 16);

    memset(keyseed, 0, 16 + 8);
}

/*
 * Cryptographic pseudo-random-number generator, used for generating
 * session keys.
 */
byte
blockRandByte(struct BlockRandContext *c)
{
    int i;

    if (!c->bufleft) {
	byte timestamp[8];

	/* Get some true-random noise to help */
	randPoolGetBytes(timestamp, sizeof(timestamp));

	/* Compute next 8 bytes of output */
	for (i = 0; i < 8; i++)
	    c->outbuf[i] = c->internalbuf[i] ^ timestamp[i];
	cast_encrypt_block(&c->C5ctx, c->outbuf, c->outbuf);
	/* Compute new seed vector */
	for (i = 0; i < 8; i++)
	    c->internalbuf[i] = c->outbuf[i] ^ timestamp[i];
	cast_encrypt_block(&c->C5ctx, c->internalbuf, c->internalbuf);
	burn(timestamp);
	c->bufleft = 8;
    }
    return c->outbuf[--c->bufleft];
}
