
{
  UCDI_Def.pas
  definitions for the Universal Crypt Driver Interface (UCDI),
  can be compiled for the old version and the new Extended UCDI
  Copyright (C) 1997/98 Markus Hahn
  last update: 26 July 98
}

unit UCDI_Def;

interface
uses BasicTypes;

// The following switch (__EXTENDED_UCDI) decides if we either want to
// create a driver compatible the old UCDI definition or to the
// new Extended UCDI standard.
// The difference is mainly in the callback of the random generator
// function. In the old version there were no possibility for
// a complete multithreading environment to use the UCDI
// without any tricks (e.g. semaphores). In Extended UCDI we now pass
// an additional 32bit integer to the callback function which
// can be used e.g. to store an object reference to a random
// generator. Additionally the version number is delivered in
// the information block structure plus the driver type.
// Because other changes of the interface are not necessary
// Extended UCDI drivers should have the extension ".UCX" to let
// applications detect them immediately (e.g. "blowfish.ucx").
// It's strongly recommended to use the Extended UCDI format for
// new applications.

{$define __EXTENDED_UCDI}


// to be sure we define our own NULL pointer value
const
  UCDI_NULL = Pointer(0);


// Extended UCDI defines now a version number
{$ifdef __EXTENDED_UCDI}
const
  UCDI_VERSION_MAJOR = 2;
  UCDI_VERSION_MINOR = 0;
{$endif}



// "@" constants (where the function must be located in the DLL)
// Remember this when you create your .def file for the driver compilation
// to declare the fixed entry points in the DLL.
const
  UCDI_FUNCATDLL_GETDRIVERINFO      = 1;
  UCDI_FUNCATDLL_SELFTEST           = 2;
  UCDI_FUNCATDLL_CREATEWORKCONTEXT  = 3;
  UCDI_FUNCATDLL_RESETWORKCONTEXT   = 4;
  UCDI_FUNCATDLL_DESTROYWORKCONTEXT = 5;
  UCDI_FUNCATDLL_ENCRYPTBUFFER      = 6;
  UCDI_FUNCATDLL_DECRYPTBUFFER      = 7;


// the following constants describe a driver's characteristic
{$ifdef __EXTENDED_UCDI}
const
  // driver is a stream cipher xoring (stream is plaintext
  // independant) the plaintext
  UCDI_CIPHERIS_XORSTREAM = 0;
  // driver is blockcipher, enabling single read and write block access (ECB)
  UCDI_CIPHERIS_BLOCK     = 1;
  // driver is blockcipher, enabling single read-only block access (CBC)
  UCDI_CIPHERIS_BLOCKLINK = 2;
  // driver doesn't allow block access, but does also not simply
  // xor the key stream over the plaintext (multiple streams can
  // be used with the same key and some known plaintext also)
  UCDI_CIPHERIS_NOBLOCK   = 3;
{$endif}


// error codes
const
  UCDI_ERROR_NOERROR         = 0;
  UCDI_ERROR_INVALIDDRIVER   = 1;
  UCDI_ERROR_KEYSETUPERROR   = 2;
  UCDI_ERROR_WEAKKEY         = 3;   // (this must only be a warning)
  UCDI_ERROR_DESTRUCTERROR   = 4;

// work modes
const
  UCDI_MODE_ENCRYPT = 0;
  UCDI_MODE_DECRYPT = 1;


// max. length of a UCDI driver title
const
  UCDI_MAX_DRIVERTITLE_LEN = 100;


// the following structure is returned by UCDI_GetDriverInfo()
type
  PUCDIINFOBLOCK = ^TUCDIINFOBLOCK;
  TUCDIINFOBLOCK = packed record
    // size of this structure, must be set by the caller,
    // used for future extensions
    wSizeOf : WORD16;
    // name of the driver
    pDriverName : PChar;
    // size of one cipher block in bytes (max. 64kB),
    // a stream cipher must return 1
    wBlockSize : WORD16;
    // key size (max. 64kB). Can be 0, e.g. for individual
    // DLLs simulating a keycard.
    wKeySize : WORD16;
{$ifndef __EXTENDED_UCDI}
    // Does the driver link the single blocks (if any)?
    // BOOL_FALSE: blocks are not linked (ECB)
    // BOOL_TRUE : blocks are linked (CBC)
    blBlockLink : BYTEBOOL;
{$endif}
    // Does the driver hash the password by its own routines?
    // BOOL_FALSE: the application must deliver a key already hashed
    //             to nKeySize bytes
    // BOOL_TRUE : the driver gets the raw password
    blOwnHasher : BYTEBOOL;
    // size of the initialisation data in bytes (max. 64kB),
    // e.g. 8 for a 64bit block cipher in CBC mode (IV) or 0 for any
    // block cipher in ECB mode
    wInitDataSize : WORD16;
    // size of a cryptdriver context, necessary for a proper
    // multitasking. Every caller's actual state is stored
    // in a context, e.g. the actual IV or pbox/sbox contents.
    lContextSize : WORD32;
{$ifdef __EXTENDED_UCDI}
    // version number (high byte: major, low byte: minor number)
    wVersion : WORD16;
    // the type of this cipher (see UCDI_CIPHERIS_xxx)
    bCipherIs : WORD8;
{$endif}
  end;

// to avoid that the driver must create necessary random numbers itself
// we define a function type which delivers a number of randombytes

type

// address of a function which creates cryptographic secure random numbers
// @param pTargetBuffer pointer to buffer where to place the random values
// @param lNumOfRandomBytes number of required random bytes
// @param pData general pointer for multithreading support (Extended UCDI only)
TUCDI_RandomGenerator = procedure(pTargetBuffer : Pointer;
{$ifdef __EXTENDED_UCDI}
                                  lNumOfRandomBytes : WORD32;
                                  pData : Pointer); cdecl;
{$else}
                                  lNumOfRandomBytes : WORD32); cdecl;
{$endif}


// The function interface, usually not needed (because the DLL
// functions are called by index) and replaced by individual
// function names, but with the same return values and parameters.
// For a better ledigble sourcecode you might replace the "TUCDI_"
// prefix with the one of your DLL name (e.g. "Blowfish_").
// The order of the following prototypes must be the same in a
// driver DLL, otherwise the caller will crash!
// Please remember that the calling convention is "_cdecl", always!

type

// to get a driver's characteristics
// @param pInfo pointer where to store the informations,
//              it's the driver's job to check the TUCDIINFOBLOCK.wSizeOfMember
//		first to deliver only that much bytes which fit
//		in the given memory block!
// @return error code:
// UCDI_ERROR_NOERROR       - driver declares itself as valid
// UCDI_ERROR_INVALIDDRIVER - driver declares itself as invalid
PUCDI_GetDriverInfo = ^TUCDI_GetDriverInfo;
TUCDI_GetDriverInfo = function (pInfo : PUCDIINFOBLOCK) : WORD32;


// to start a self-test of the driver
// @param pTestContext pointer to a memory block of TUCDIINFOBLOCK.lContextSize
//                     to store a context for the tests. Memory block is
//                     allocated by the caller and should be cleared and freed
//                     after the function call. The self test was separated to
//                     give the caller a chance no to use it, e.g. if the test
//                     takes to much time.
// @return error code:
// UCDI_ERROR_NOERROR       - selftest succeeded
// UCDI_ERROR_INVALIDDRIVER - selftest failed
PUCDI_SelfTest = ^TUCDI_SelfTest;
TUCDI_SelfTest = function (pTestContext : Pointer) : WORD32; cdecl;


// to create a work context
// @param pContext pointer to a memory block of TUCDIINFOBLOCK.lContextSize to
//                 store the work context. Memory block is allocated by the
//                 caller and is as long valid as encryption/decryption is
//                 necessary (if the process is finished the memory block
//                 should be cleared and freed).
// @param pKey pointer to an array of bytes represeting the key/password. If
//             TUCDIINFOBLOCK.blOwnHasher is BOOL_TRUE the array contains the
//             raw password, if's BOOL_FALSE the array contains the already
//             hashed key with the size of TUCDIINFOBLOCK.wKeySize bytes.
// @param wKeyLen length of the key/password. This parameter never will be zero
//                and contains always the valid length, even if the password
//                was already hashed to a key with the right size.
// @param wMode work mode, see UCDI_MODE_xxx constants
// @param pInitData pointer to init. data storage
// @param UCDI_RandomGenerator address of the random generator
// @param pRndGenData general pointer passed to the random generator (Extended
//                    UCDI only)
// @return error code:
// UCDI_ERROR_NOERROR       - key setup succeeded
// UCDI_ERROR_KEYSETUPERROR - key setup failed
// UCDI_ERROR_WEAKKEY       - success, but weak key detected
PUCDI_CreateWorkContext = ^TUCDI_CreateWorkContext;
{$ifdef __EXTENDED_UCDI}
  TUCDI_CreateWorkContext = function (pContext : Pointer;
                                      pKey : Pointer;
                                      wKeyLen : WORD16;
                                      wMode : WORD16;
                                      pInitData : Pointer;
                                      GetRndBytes : TUCDI_RandomGenerator;
				      pRndGenData : Pointer) : WORD32; cdecl;
{$else}
  TUCDI_CreateWorkContext = function (pContext : Pointer;
                                      pKey : Pointer;
                                      wKeyLen : WORD16;
                                      wMode : WORD16;
                                      pInitData : Pointer;
                                      GetRndBytes : TUCDI_RandomGenerator)
                                       : WORD32; cdecl;
{$endif}


// resets a driver; to avoid time consuming key setups
// @param pContext pointer to a work context; see above for further details
// @param wMode mode; see UCDI_MODE_xxx constants
// @param pInitData pointer to init. data storage (for new en-/decryption)
// @param GetRndBytes address of the random generator
// @param pRndGenData general pointer passed to the random generator (Extended
//                    UCDI only)
PUCDI_ResetWorkContext = ^TUCDI_ResetWorkContext;
{$ifdef __EXTENDED_UCDI}
TUCDI_ResetWorkContext = procedure (pContext : Pointer;
                                    wMode : WORD16;
                                    pInitData : Pointer;
                                    GetRndBytes : TUCDI_RandomGenerator;
	          		    pRndGenData : Pointer); cdecl;
{$else}
TUCDI_ResetWorkContext = procedure (pContext : Pointer;
                                    wMode : WORD16;
                                    pInitData : Pointer;
                                    GetRndBytes : TUCDI_RandomGenerator); cdecl;
{$endif}


// destroys a work context; usually not necessary; but might be e.g. when the
// driver allocates memory for every work context
// @param pContext pointer to a work context; see above for further details
// @return error code:
// UCDI_ERROR_NOERROR       - destruction succeeded
// UCDI_ERROR_DESTRUCTERROR - destruction failed; this should force the caller
//                            not to use the driver again!
PUCDI_DestroyWorkContext = ^TUCDI_DestroyWorkContext;
TUCDI_DestroyWorkContext = function (pContext : Pointer) : WORD32; cdecl;


// encrypts a buffer content to a target buffer. The target buffer must be
// adjusted to the block size and can be the same as the source buffer. The
// caller must be adjusted to thethe number of bytes to the blocksize; so the
// driver hasn't to do any padding.
// @param pContext pointer to a work context
// @param pSource pointer to the source buffer
// @param pTarget pointer to the target buffer
// @param lNumOfBytes number of bytes to encrypt
PUCDI_EncryptBuffer = ^TUCDI_EncryptBuffer;
TUCDI_EncryptBuffer = procedure (pContext : Pointer;
                                 pSource : Pointer;
                                 pTarget : Pointer;
                                 lNumOfBytes : WORD32); cdecl;


// decrypts a buffer content to a target buffer. The target buffer must be
// adjusted to the block size and can be the same as the source buffer.
// @param pContext pointer to a work context
// @param pSource pointer to the source buffer
// @param pTarget pointer to the target buffer
// @param lNumOfBytes number of bytes to decrypt
// @param pPreviousBlock pointer to previous block, used to enable to decrypt
//                       even chained blocks, e.g. CBC (ignored if UCDI_NULL),
//                       if you want to access the very first block a call
//        	         to the UCDI_ResetWorkContext routine is necessary and
//  		         the original init. data has to be replaced
PUCDI_DecryptBuffer = ^TUCDI_DecryptBuffer;
TUCDI_DecryptBuffer = procedure (pContext : Pointer;
                                 pSource : Pointer;
                                 pTarget : Pointer;
                                 lNumOfBytes :WORD32;
                                 pPreviousBlock : Pointer);


implementation

// no implementation needed here

end.













