

/**
  * UCDI_Def.h
  * the UCDI interface definition, 2nd Release
  *	Copyright (C) 1998 Markus Hahn
  *	last update: 1 Oct 98
  */

#ifndef __UCDI_DEF_H
#define __UCDI_DEF_H


// for own basic data types, to avoid system dependencies
#include "BasicTypes.h"


// The following switch (__EXTENDED_UCDI) decides if we either want to
// create a driver compatible the old UCDI definition or to the
// new Extended UCDI standard.
// The difference is mainly in the callback of the random generator
// function. In the old version there were no possibility for
// a complete multithreading environment to use the UCDI
// without any tricks (e.g. semaphores). In Extended UCDI we now pass
// an additional 32bit integer to the callback function which
// can be used e.g. to store an object reference to a random
// generator. Additionally the version number is delivered in
// the information block structure plus the driver type.
// Because other changes of the interface are not necessary
// Extended UCDI drivers should have the extension ".UCX" to let
// applications detect them immediately (e.g. "blowfish.ucx").
// It's strongly recommended to use the Extended UCDI format for
// new applications.


define __EXTENDED_UCDI


// Extended UCDI defines now a version number
#ifdef __EXTENDED_UCDI
  #define UCDI_VERSION_MAJOR	2
  #define UCDI_VERSION_MINOR	0
#endif

// to be sure we define our own NULL pointer value
#define  UCDI_NULL		0


// "@" constants (where the function must be located in the DLL),
// Remember this when you create your .def file for the driver compilation
// to declare the fixed entry points in the DLL.
#define UCDI_FUNCATDLL_GETDRIVERINFO      1
#define UCDI_FUNCATDLL_SELFTEST           2
#define UCDI_FUNCATDLL_CREATEWORKCONTEXT  3
#define UCDI_FUNCATDLL_RESETWORKCONTEXT   4
#define UCDI_FUNCATDLL_DESTROYWORKCONTEXT 5
#define UCDI_FUNCATDLL_ENCRYPTBUFFER      6
#define UCDI_FUNCATDLL_DECRYPTBUFFER      7


// the following constants describe a driver's characteristic
#ifdef __EXTENDED_UCDI
  // driver is a stream cipher xoring (stream is plaintext
  // independant) the plaintext
  #define UCDI_CIPHERIS_XORSTREAM   0
  // driver is blockcipher, enabling single read and write block access (ECB)
  #define UCDI_CIPHERIS_BLOCK      	1
  // driver is blockcipher, enabling single read-only block access (CBC)
  #define UCDI_CIPHERIS_BLOCKLINK   2
  // driver doesn't allow block access, but does also not simply
  // xor the key stream over the plaintext (multiple streams can
  // be used with the same key and some known plaintext also)
  #define UCDI_CIPHERIS_NOBLOCK		3
#endif



// error codes
#define UCDI_ERROR_NOERROR			0
#define UCDI_ERROR_INVALIDDRIVER    1
#define UCDI_ERROR_KEYSETUPERROR    2
#define UCDI_ERROR_WEAKKEY          3  // (this must only be a warning)
#define UCDI_ERROR_DESTRUCTERROR    4


// work modes
#define UCDI_MODE_ENCRYPT	0
#define UCDI_MODE_DECRYPT	1


// max. length of a UCDI driver title
#define UCDI_MAX_DRIVERTITLE_LEN	100


// the following structure is returned by UCDI_GetDriverInfo()
#pragma pack(push, 1)
typedef struct {
   // size of this structure, must be set by the caller,
   // used for future extensions
   WORD16 wSizeOf;
   // name of the driver
   WORD8* pDriverName;
   // size of one cipher block in bytes (max. 64kB),
   // a stream cipher must return 1
   WORD16 wBlockSize;
   // key size (max. 64kB). Can be 0, e.g. for individual
   // DLLs simulating a keycard.
   WORD16 wKeySize;
#ifndef __EXTENDED_UCDI
   // Does the driver link the single blocks (if any)?
   // BOOL_FALSE: blocks are not linked (ECB)
   // BOOL_TRUE : blocks are linked (CBC)
   BYTEBOOL blBlockLink;
#endif
   // Does the driver hash the password by its own routines?
   // BOOL_FALSE: the application must deliver a key already hashed
   //             to nKeySize bytes
   // BOOL_TRUE : the driver gets the raw password
   BYTEBOOL blOwnHasher;
   // size of the initialisation data in bytes (max. 64kB),
   // e.g. 8 for a 64bit block cipher in CBC mode (IV) or 0 for any
   // block cipher in ECB mode
   WORD16 wInitDataSize;
   // size of a work context, necessary for a proper multitasking.
   // Every caller's actual state is stored in a context, e.g. the
   // actual IV or pbox/sbox contents.
   WORD32 lContextSize;
#ifdef __EXTENDED_UCDI
   // version number (high byte: major, low byte: minor number)
   WORD16 wVersion;
    // the type of this cipher (see UCDI_CIPHERIS_xxx)
   WORD8 bCipherIs;
#endif
} UCDIINFOBLOCK;
#pragma pack(pop)


// to avoid that the driver must create necessary random numbers itself
// we define a function type which delivers a number of random bytes


/**
  * address of a function which creates cryptographic secure random numbers
  * @param pTargetBuffer pointer to buffer where to place the random values
  * @param lNumOfRandomBytes number of required random bytes
  * @param pData general pointer for multithreading support (Extended UCDI only)
  */
#ifdef __EXTENDED_UCDI
  typedef void UCDI_RandomGenerator(WORD8* pTargetBuffer,
                                    WORD32 lNumOfRandomBytes,
									void* pData);
#else
  typedef void UCDI_RandomGenerator(WORD8* pTargetBuffer,
                                    WORD32 lNumOfRandomBytes);
#endif




// The function interface, usually not needed (because the DLL
// functions are called by index) and replaced by individual
// function names, but with the same return values and parameters.
// For a better ledigble sourcecode you might replace the "UCDI_"
// prefix with the one of your DLL name (e.g. "Blowfish_").
// The order of the following prototypes must be the same in a
// driver DLL, otherwise the caller will crash!
// Please remember that the calling convention is "_cdecl", always!


/**
  * to get a driver's characteristics
  * @param pInfo pointer where to store the informations,
  *              it's the driver's job to check the UCDIINFOBLOCK.wSizeOfMember
  *				 first to deliver only that much bytes which fit
  *				 in the given memory block!
  * @return error code:
  * UCDI_ERROR_NOERROR       - driver declares itself as valid
  * UCDI_ERROR_INVALIDDRIVER - driver declares itself as invalid
  */
typedef WORD32 UCDI_GetDriverInfo(UCDIINFOBLOCK *pInfo);


/**
  * to start a self-test of the driver
  * @param pTestContext pointer to a memory block of UCDIINFOBLOCK.lContextSize
  *                     to store a context for the tests. Memory block is allocated by the
  *                     caller and should be cleared and freed after the function call.
  *                     The self test was separated to give the caller a chance no to use it,
  *                     e.g. if the test takes to much time.
  * @return error code:
  * UCDI_ERROR_NOERROR       - selftest succeeded
  * UCDI_ERROR_INVALIDDRIVER - selftest failed
  */
typedef WORD32 UCDI_SelfTest (void* pTestContext);


/**
  * to create a work context
  * @param pContext pointer to a memory block of UCDIINFOBLOCK.lContextSize
  *                 to store the work context. Memory block is allocated by the caller
  *                 and is as long valid as encryption/decryption is necessary (if the
  *                 process is finished the memory block should be cleared and freed).
  * @param pKey pointer to an array of bytes represeting the key/password. If
  *             UCDIINFOBLOCK.blOwnHasher is BOOL_TRUE the array contains the
  *             raw password, if it's BOOL_FALSE the array contains the already hashed key
  *             with the size of UCDIINFOBLOCK.wKeySize bytes.
  * @param wKeyLen length of the key/password. This parameter never will be zero and
  *                contains always the valid length, even if the password was already
  *                hashed to a key with the right size.
  * @param wMode work mode, see UCDI_MODE_xx constants
  * @param pInitData pointer to init. data storage
  * @param UCDI_RandomGenerator address of the random generator
  * @param pRndGenData general pointer passed to the random generator (Extended UCDI only)
  * @return error code:
  * UCDI_ERROR_NOERROR       - key setup succeeded
  * UCDI_ERROR_KEYSETUPERROR - key setup failed
  * UCDI_ERROR_WEAKKEY       - success, but weak key detected
  */
#ifdef __EXTENDED_UCDI
  typedef WORD32 UCDI_CreateWorkContext(void* pContext,
                                        WORD8* pKey,
                                        WORD16 wKeyLen,
                                        WORD16 wMode,
                                        void* pInitData,
                                        UCDI_RandomGenerator GetRndBytes,
										void* pRndGenData);
#else
  typedef WORD32 UCDI_CreateWorkContext(void* pContext,
                                        WORD8* pKey,
                                        WORD16 wKeyLen,
                                        WORD16 wMode,
                                        void* pInitData,
                                        UCDI_RandomGenerator GetRndBytes);
#endif


/**
  * resets a driver, to avoid time consuming key setups
  * @param pContext pointer to a work context, see above for further details
  * @param wMode mode, see UCDI_MODE_xx constants
  * @param pInitData pointer to init. data storage (for new en-/decryption)
  * @param GetRndBytes address of the random generator
  * @param pRndGenData general pointer passed to the random generator (Extended UCDI only)
  */
#ifdef __EXTENDED_UCDI
  typedef void UCDI_ResetWorkContext(void* pContext,
                                     WORD16 wMode,
                                     void* pInitData,
                                     UCDI_RandomGenerator GetRndBytes,
	          					     void* pRndGenData);
#else
  typedef void UCDI_ResetWorkContext(void* pContext,
                                     WORD16 wMode,
                                     void* pInitData,
                                     UCDI_RandomGenerator GetRndBytes);
#endif


/**
  * destroys a work context, usually not necessary, but might be e.g. when the driver
  * allocates memory for every work context
  * @param pContext pointer to a work context, see above for further details
  * @return error code:
  * UCDI_ERROR_NOERROR       - destruction succeeded
  * UCDI_ERROR_DESTRUCTERROR - destruction failed, this should force the
  *                            caller not to use the driver again!
  */
typedef WORD32 UCDI_DestroyWorkContext (void* pContext);


/**
  * encrypts a buffer content to a target buffer. The target buffer must be adjusted 
  * to the block size and can be the same as the source buffer. The caller must be adjusted
  * to the number of bytes to the blocksize, so the driver hasn't to do any padding.
  * @param pContext pointer to a work context
  * @param pSource pointer to the source buffer
  * @param pTarget pointer to the target buffer
  * @param lNumOfBytes number of bytes to encrypt
  */
typedef void UCDI_EncryptBuffer(void* pContext,
                                void* pSource,
                                void* pTarget,
                                WORD32 lNumOfBytes);


/**
  * decrypts a buffer content to a target buffer. The target buffer must be adjusted to 
  * the block size and can be the same as the source buffer.
  * @param pContext pointer to a work context
  * @param pSource pointer to the source buffer
  * @param pTarget pointer to the target buffer
  * @param lNumOfBytes number of bytes to decrypt
  * @param pPreviousBlock pointer to previous block, used to enable to decrypt 
  *                       even chained blocks, e.g. CBC (ignored if UCDI_NULL),
  *                       if you want to access the very first block a call
  *						  to the UCDI_ResetWorkContext routine is necessary and
  *						  the original init. data has to be replaced	
  */
typedef void UCDI_DecryptBuffer(void* pContext,
                                void* pSource,
                                void* pTarget,
                                WORD32 lNumOfBytes,
                                void* pPreviousBlock);


#endif


