

/**
  *	UCDIServer.h
  * header for the basic UCDI server module
  *	Copyright (C) 1998 Markus Hahn
  *	last update: 7 Nov 98
  */


#ifndef __UCDISERVER_H
#define __UCDISERVER_H

#include "BasicTypes.h"

// we need this here for the DLL handle
#include <windows.h>

// load Extended UCDI specifications
#include "UCDI_Def.h"

// we only accept Extended UCDI builds
#ifndef __EXTENDED_UCDI
  #error Only Extended UCDI builds are possible.
#endif

// we use our own random generator
#include "RandomPool.h"


// the highest supported major UCDI version number
// (this server will work with all 2.x drives generally)
#define UCDISERVER_MAX_UCDI_VERSION_MAJOR	2
#define UCDISERVER_MAX_UCDI_VERSION_MINOR	255


// error codes
#define UCDISERVER_ERROR_NOERROR			0
#define UCDISERVER_ERROR_ERROR				1
#define UCDISERVER_ERROR_DRIVERNOTFOUND		2
#define UCDISERVER_ERROR_BADFUNCADDR    	3
#define UCDISERVER_ERROR_INVALIDDRIVER		4
#define UCDISERVER_ERROR_OUTOFMEMORY		5
#define UCDISERVER_ERROR_WRONGDRIVERVERSION	6
#define UCDISERVER_ERROR_WEAKKEY    		7



// crypt modes
#define	UCDISERVER_MODE_ENCRYPT	UCDI_MODE_ENCRYPT
#define	UCDISERVER_MODE_DECRYPT	UCDI_MODE_DECRYPT


// work context, stores everything for cryptdriver session

#pragma pack(push, 1)

typedef struct {
  // DLL handle
  HINSTANCE drvHandle;
  // UCDI function addresses
  UCDI_GetDriverInfo*		pGetDriverInfo;
  UCDI_SelfTest*			pSelfTest;
  UCDI_CreateWorkContext*	pCreateWorkContext;
  UCDI_ResetWorkContext*	pResetWorkContext;
  UCDI_DestroyWorkContext*	pDestroyWorkContext;
  UCDI_EncryptBuffer*		pEncryptBuffer;
  UCDI_DecryptBuffer*		pDecryptBuffer;
  // UCDI driver information
  UCDIINFOBLOCK infoblock;
  // function reference for random number generation
  UCDI_RandomGenerator* pRandomGenerator;
  // data reference for the random generator
  void* pRndGenData;
  // to check if the internal random generator is used
  BYTEBOOL blUseInternalRndGen;
} CRYPTDRIVERCONTEXT;



// the handle for an encryption session
typedef struct {
  // crypt mode, see UCDISERV_xxx constants
  WORD16 wCryptMode;
  // pointer to a cryptdriver context
  CRYPTDRIVERCONTEXT* pDriverContext;
  // pointer to an algorithm's runtime memory area
  void* pWorkContext;
} CRYPTSESSIONHANDLE;

#pragma pack(pop)



/**
  * returns a driver's information block
  * @param pDriverFilename driver file name
  * @param pInfoBlock pointer to block where to store the data
  * @param pDriverTitle where to copy the driver title
  * @return error code:
  *         UCDISERVER_ERROR_NOERROR		: success
  *         UCDISERVER_ERROR_ERROR			: unloading error
  *         UCDISERVER_ERROR_DRIVERNOTFOUND	: driver file could not be found
  *         UCDISERVER_ERROR_BADFUNCADDR	: function could not get retrieved
  *         UCDISERVER_ERROR_BADFUNCADDR	: function could not get retrieved
  *         UCDISERVER_ERROR_INVALIDDRIVER	: driver is not valid
  */
WORD32 __stdcall 
	UCDIServer_GetDriverInfo (char* pDriverFilename,
	                          UCDIINFOBLOCK* pInfoBlock,
							  char* pDriverTitle);




/**
  * loads a driver, gets the function addresses into a context
  * @param pDriverName driver name
  * @param pDrvCtxPtr pointer where to store the cryptdriver handle (pointer to 
  *                   allocated context)
  * @param pRandGenFunc pointer to random generator function (may be UCDI_NULL, if so
  *			 		    an internal random generator is going to be used)
  * @param pRandGenData general pointer for random generator function, e.g. to point
  *				        to an object instance (ignored if pRandGenFunc is UCDI_NULL)
  * @pRandSeed pointer to random seed data (only used for the internal random generator,
  *                                         ignored if pRandGenFunc differs from UCDI_NULL,
  *                                         may also be UCDI_NULL)
  * @pRandSeedLen number of random seed bytes (ignored if pRandSeed is UCDI_NULL)
  * @return error code:
  *         UCDISERVER_ERROR_NOERROR            : success
  *         UCDISERVER_ERROR_OUTOFMEMORY        : not enough memory
  *         UCDISERVER_ERROR_ERROR              : loading error
  *         UCDISERVER_ERROR_DRIVERNOTFOUND     : driver file could not be found
  *         UCDISERVER_ERROR_BADFUNCADDR        : a function could not be retrieved
  *         UCDISERVER_ERROR_INVALIDDRIVER      : driver is not valid
  *         UCDISERVER_ERROR_WRONGDRIVERVERSION : UCDI version is too high to handle
  */
WORD32 __stdcall 
	UCDIServer_CreateDriverContext (char* pDriverName,
	                                CRYPTDRIVERCONTEXT** pDrvCtxPtr,
									UCDI_RandomGenerator* pRandGenFunc,
									void* pRandGenData,
									void* pRandSeed,
									WORD32 lRandSeedLen);




/**
  * unloads a driver, clears and frees the context
  * @param pDrvCtx cryptdriver handle
  * @return error code:
  *         UCDISERVER_ERROR_NOERROR : success
  *         UCDISERVER_ERROR_ERROR   : unloading (fatal) error
  */
WORD32 __stdcall 
	UCDIServer_DestroyDriverContext (CRYPTDRIVERCONTEXT* pDrvCtx);



/**
  * executes the selftest function of an already
  * loaded driver, this guarantees that the driver
  * cannot be altered between selftest and usage
  * @param pDrvCtx cryptdriver handle
  * @param blExtendedTest flag for extended testing (testing with own data)
  * @return error code:
  *         UCDISERVER_ERROR_NOERROR       : success
  *         UCDISERVER_ERROR_INVALIDDRIVER : driver is not valid
  *         UCDISERVER_ERROR_OUTOFMEMORY   : not enough memory
  */
WORD32 __stdcall 
	UCDIServer_ExecuteSelfTest(CRYPTDRIVERCONTEXT* pDrvCtx,
	                           BYTEBOOL blExtendedTest);




/**
  * returns the information block of the driver
  * @param pDrvCtx cryptdriver handle
  * @param pInfoBlock pointer to block where to store the data
  * @return error code:
  * UCDI_ERROR_NOERROR       - driver declares itself as valid
  * UCDI_ERROR_ERROR         - unknown error code (invalid driver)
  * UCDI_ERROR_INVALIDDRIVER - driver declares itself as invalid
  */
WORD32 __stdcall 
	UCDIServer_GetInfoBlock(CRYPTDRIVERCONTEXT* pDrvCtx,
	                        UCDIINFOBLOCK* pInfoBlock);



/**
  * creates a new session for encrypting or decrypting data
  * @param wMode work mode, see UCDISERVER_MODE_xxx constants
  * @param pKey pointer to key stored a byte buffer
  * @param wKeyLen key size in bytes (must fit to the cryptdriver, if demanded)
  * @param pDrvCtx cryptdriver handle
  * @param pInitData pointer where to get/set the init. data (usually a CBC IV)
  * @param pSessionHandlePtr pointer to the storage of the created cryptsession handle
  * @return error code:
  *         UCDISERVER_ERROR_NOERROR     : success
  *         UCDISERVER_ERROR_ERROR       : key setup failed
  *         UCDISERVER_ERROR_OUTOFMEMORY : not enough memory
  *         UCDISERVER_ERROR_WEAKKEY     : weak key detected
  */
WORD32 __stdcall 
	UCDIServer_OpenSession (WORD16 wMode,
                            WORD8* pKey,
                            WORD16 wKeyLen,
                            CRYPTDRIVERCONTEXT* pDrvCtx,
                            void* pInitData,
                            CRYPTSESSIONHANDLE** pSessionHandlePtr);


/**
  * allows an interruption of the en-/decryption stream
  * without a timeconsuming new key setup (mostly used
  * to set a new CBC IV, if any, to allow chunking)
  * @param pSessionHandle cryptsession handle
  * @param pInitData pointer where to get/set the init. data (usually a CBC IV)
  */
void __stdcall 
	UCDIServer_ResetSession (CRYPTSESSIONHANDLE* pSessionHandle,
                             void* pInitData);


/**
  * destroys a session handle
  * @param pSessionHandle cryptsession handle
  * @return error code:
  *         UCDISERVER_ERROR_NOERROR : success
  *         UCDISERVER_ERROR_ERROR   : driver couldn't stop properly (fatal error)
  */
WORD32 __stdcall 
	UCDIServer_CloseSession (CRYPTSESSIONHANDLE* pSessionHandle);




/**
  * encrypts a number of blocks
  * @param pSessionHandle cryptsession handle
  * @param pSource pointer to the source buffer (may be equal to the source buffer)
  * @param pTarget pointer to the target buffer
  * @param lNumOfBlocks number of _blocks_ to encrypt
  */
void __stdcall 
	UCDIServer_EncryptBlocks (CRYPTSESSIONHANDLE* pSessionHandle,
							  void* pSource,
                              void* pTarget,
                              WORD32 lNumOfBlocks);


/**
  * decrypts a number of blocks
  * @param pSessionHandle cryptsession handle
  * @param pSource pointer to the source buffer
  * @param pTarget pointer to the target buffer (may be equal to the source buffer)
  * @param lNumOfBlocks number of _blocks_ to decrypt
  * @param pPreviousBlock pointer to previous block (ignored if UCDI_NULL)
  */
void __stdcall 
	UCDIServer_DecryptBlocks (CRYPTSESSIONHANDLE* pSessionHandle,
							  void* pSource,
                              void* pTarget,
                              WORD32 lNumOfBlocks,
                              void* pPreviousBlock);


/**
  * delivers random data from the current generator
  * @param pDrvCtx cryptdriver handle
  * @param pTarget pointer to the output buffer
  * @param lNumOfBytes number of random bytes to deliver
  */
void __stdcall 
	UCDIServer_GetRandomData (CRYPTDRIVERCONTEXT* pDrvCtx,
                              void* pTarget,
                              WORD32 lNumOfBytes);

#endif

