
{
  CryptPak.pas
  interface unit for the usage of CryptPak.dll
  Copyright (C) 1998 Markus Hahn
  last update: 7 Nov 98
}

unit CryptPak;

interface
uses Windows,
     BasicTypes, UCDI_Def;


// Support

// returns the version number of CryptPak.dll
// @return version (higher word: low byte contains minor, high byte the
//         major version number / lower word : built number)
function Support_GetVersion : WORD32; stdcall; external 'CryptPak.dll';


// MD5

// size of an MD5 digest
const
  MD5_DIGESTSIZE = 16;

// MD5 context
type
  PMD5CTX = ^TMD5CTX;
  TMD5CTX = packed record
    state  : array [0..3] of WORD32;
    count  : array [0..1] of WORD32;
    buffer : array [0..63] of WORD8;
  end;


// selftest
// @return BOOL-TRUE: selftest succeded / BOOL_FALSE: failed
function MD5_SelfTest : BYTEBOOL; stdcall; external 'CryptPak.dll';

// initialises an MD5 context
// @param pCtx pointer to MD5 context to init.
procedure MD5_Init(pCtx : PMD5CTX); stdcall; external 'CryptPak.dll';

// updates an MD5 context
// @param pCtx pointer to MD5 context
// @param pData pointer to data to hash
// @param lNumOfBytes number of bytes to hash
procedure MD5_Update(pCtx : PMD5CTX;
                     pData : Pointer;
                     lNumOfBytes : WORD32); stdcall; external 'CryptPak.dll';

// finalizes the context and extracts the digest
// @param pDigest pointer to digest buffer (MD5_DIGESTSIZE bytes large)
// @param pCtx pointer to MD5 context
procedure MD5_Final(pDigest : Pointer;
                    pCtx : PMD5CTX); stdcall; external 'CryptPak.dll';


// SHA1

// size of an MD5 digest (SHA1_DATASIZE is an internal constant)
const
  SHA1_DIGESTSIZE = 20;
  SHA1_DATASIZE	  = 64;


// SHA-1 context
type
  PSHA1CTX = ^TSHA1CTX;
  TSHA1CTX = packed record
    digest     : array [0..4] of WORD32;
    lCountLo   : WORD32 ;
    lCountHi   : WORD32 ;
    data       : array[0..15] of WORD32;
    dataBuffer : array [0..SHA1_DATASIZE - 1] of WORD8;
    blDone     : BYTEBOOL;
  end;


// selftest
// @return BOOL-TRUE: selftest succeded / BOOL_FALSE: failed
function SHA1_SelfTest : BYTEBOOL; stdcall; external 'CryptPak.dll';

// initialises an SHA-1 context
// @param pCtx pointer to SHA-1 context to init.
procedure SHA1_Init(pCtx : PSHA1CTX); stdcall; external 'CryptPak.dll';

// updates an SHA-1 context
// @param pCtx pointer to SHA-1 context
// @param pData pointer to data to hash
// @param lNumOfBytes number of bytes to hash
procedure SHA1_Update(pCtx : PSHA1CTX;
                      pData : Pointer;
                      lNumOfBytes : WORD32); stdcall; external 'CryptPak.dll';

// finalizes the context and extracts the digest
// @param pDigest pointer to digest buffer (SHA1_DIGESTSIZE bytes large)
// @param pCtx pointer to SHA-1 context
procedure SHA1_Final(pDigest : Pointer;
                     pCtx : PSHA1CTX); stdcall; external 'CryptPak.dll';



// CRC32

// init. and final-xor constants
const
  CRC32_INITVALUE = $ffffffff;
  CRC32_DONEVALUE = $ffffffff;


// updates a CRC32
// @param lOldCRC32 old CRC32
// @param pData pointer to data to checksum
// @param lNumOfBytes number of bytes to checksum
// @return new CRC32
function CRC32_Update(lOldCRC32 : WORD32;
                      pData : Pointer;
                      lNumOfBytes : WORD32) : WORD32; stdcall;
                                                      external 'CryptPak.dll';


// RandomPool

// random pool context (internals are not of interest)
type
  PRANDOMPOOLCTX = ^TRANDOMPOOLCTX;
  TRANDOMPOOLCTX = packed record
    hashCtx   : TSHA1CTX;
    pool      : array [0..SHA1_DIGESTSIZE - 1] of WORD8;
    lPoolSize : WORD32;
  end;

// creates a new random pool context
// @param pCtx pointer to random pool context
// @param pAddSeed pointer to additonal seed (ignored if NULL);
// @param pAddSeedLen number of additonal seed bytes
procedure RandomPool_Startup(pCtx : PRANDOMPOOLCTX;
	                     pAddSeed : Pointer;
                             lAddSeedLen : WORD32) stdcall;
                                                   external 'CryptPak.dll';

// adds seed to the random generator
// @param pCtx pointer to context
// @param pSeed pointer to the seed data
// @param lSeedLen number of seed bytes
procedure RandomPool_Reseed(pCtx : PRANDOMPOOLCTX;
	                    pSeed : pointer;
                            lSeedLen : WORD32); stdcall;
                                                external 'CryptPak.dll';

// gets some more random bytes
// @param pCtx pointer to random pool context
// @param pDataTarget pointer to the target buffer
// @param lNumOfBytes number of random bytes to create
procedure RandomPool_GetData(pCtx : PRANDOMPOOLCTX;
			     pDataTarget : Pointer;
			     lNumOfBytes : WORD32); stdcall;
                                                    external 'CryptPak.dll';


// LZSS compression


// the condition codes (bitflags)
const
     LZSS_START = 1;  // start the (de)compression
     LZSS_WORK  = 2;  // continue (de)compression
     LZSS_STOP  = 4;  // stop (de)compression, flush the buffers

// some (internal) compressor constants
const
     LZSS_N         = 4096;
     LZSS_F         =   18;
     LZSS_THRESHOLD =    2;
     LZSS_NIL	    = LZSS_N;

// LZSS context (content is not of interest, just use it)
type
  PLZSSCTX = ^TLZSSCTX;
  TLZSSCTX = packed record
    blSaveDone : BYTEBOOL;
    nSaveI, nSaveR, nSaveC, nSaveLen, nSaveS : Integer;
    nSaveLastMatchLength, nSaveCodeBufPtr : Integer;
    bSaveMask : WORD8;
    saveCode_buf : array[0..16] of WORD8;
    nSaveJ, nSaveK : Integer;
    wSaveFlags : WORD16;
    wInterruptPoint : WORD16;
    text_buf : array[0..LZSS_N + LZSS_F - 2] of WORD8;
    nMatchPosition, nMatchLength : Integer;
    lson : array [0..LZSS_N] of Integer;
    rson : array [0..LZSS_N + 256] of Integer;
    dad : array [0..LZSS_N] of Integer;
    lSourceSize : WORD32;
    lDrainSize : WORD32;
    lBytesRead : WORD32;
    lBytesWritten : WORD32;
    pDataSource : PWORD8;
    pDataDrain : PWORD8;
    blEOD :BYTEBOOL;
  end;

// compresses data
// @param pCtx pointer to LZSS context
// @param pSource pointer to the source buffer
// @param pTarget pointer to the target buffer (size must be at least 125% of
//                the source buffer size for the case of uncompressable data)
// @param lNumOfBytes number of bytes to compress
// @param bCondition bitmask, see constants LZSS_START, LZSS_WORK, LZSS_STOP
// @return number of compressed bytes (may be larger than lNumOfBytes)
function LZSS_Compress(pCtx : PLZSSCTX;
		       pSource : Pointer;
	               pTarget : Pointer;
	               lNumOfBytes : WORD32;
	               bCondition : WORD8) : WORD32; stdcall;
                                                     external 'CryptPak.dll';

// decompresses data
// @param pCtx pointer to LZSS context
// @param pSource pointer to the source buffer
// @param pTarget pointer to the target buffer
// @param lNumOfBytes number of bytes to compress
// @param lSizeOfOutputBuffer size of the target buffer
// @param bCondition bitmask, see constants LZSS_START, LZSS_WORK, LZSS_STOP
// @param pblRepeatMe pointer to a signal flag, the call of the function must
//                    be repeated as long as this flag is BOOL_TRUE (due to the
//                    fact than a few compressed bytes can create millions of
//                    uncompressed bytes)
// @return number of decompressed bytes (in this cycle)
function LZSS_Decompress(pCtx : PLZSSCTX;
		         pSource : Pointer;
	                 pTarget : Pointer;
	                 lNumOfBytes : WORD32;
	                 lSizeOfOutputBuffer : WORD32;
	                 bCondition : WORD8;
                         pblRepeatMe : PBYTEBOOL) : WORD32;
                           stdcall; external 'CryptPak.dll';

// UCDIServer

// the highest supported major UCDI version number
// (this server will work with all 2.x drives generally)
const
  UCDISERVER_MAX_UCDI_VERSION_MAJOR = 2;
  UCDISERVER_MAX_UCDI_VERSION_MINOR = 255;

// error codes
const
  UCDISERVER_ERROR_NOERROR            =	0;
  UCDISERVER_ERROR_ERROR	      =	1;
  UCDISERVER_ERROR_DRIVERNOTFOUND     =	2;
  UCDISERVER_ERROR_BADFUNCADDR        =	3;
  UCDISERVER_ERROR_INVALIDDRIVER      =	4;
  UCDISERVER_ERROR_OUTOFMEMORY	      =	5;
  UCDISERVER_ERROR_WRONGDRIVERVERSION =	6;
  UCDISERVER_ERROR_WEAKKEY    	      =	7;

// crypt modes
const
  UCDISERVER_MODE_ENCRYPT = UCDI_MODE_ENCRYPT;
  UCDISERVER_MODE_DECRYPT = UCDI_MODE_DECRYPT;

// work context, stores everything for cryptdriver session
type
  PCRYPTDRIVERCONTEXT = ^TCRYPTDRIVERCONTEXT;
  TCRYPTDRIVERCONTEXT = packed record
    drvHandle : HINST;
    pGetDriverInfo : PUCDI_GetDriverInfo;
    pSelfTest : PUCDI_SelfTest;
    pCreateWorkContext : PUCDI_CreateWorkContext;
    pResetWorkContext : PUCDI_ResetWorkContext;
    pDestroyWorkContext : PUCDI_DestroyWorkContext;
    pEncryptBuffer : PUCDI_EncryptBuffer;
    pDecryptBuffer : PUCDI_DecryptBuffer;
    infoblock : TUCDIINFOBLOCK;
    RandomGenerator : TUCDI_RandomGenerator;
    pRndGenData : Pointer;
    blUseInternalRndGen : BYTEBOOL;
  end;

// the handle for an encryption session
type
  PCRYPTSESSIONHANDLE = ^TCRYPTSESSIONHANDLE;
  TCRYPTSESSIONHANDLE = packed record
    wCryptMode : WORD16;
    pDriverContext : PCRYPTDRIVERCONTEXT;
    pWorkContext : Pointer;
  end;

// returns a driver's information block
// @param pDriverFilename driver file name
// @param pInfoBlock pointer to block where to store the data
// @param pDriverTitle where to copy the driver title
// @return error code:
//         UCDISERVER_ERROR_NOERROR	   : success
//         UCDISERVER_ERROR_ERROR	   : unloading error
//         UCDISERVER_ERROR_DRIVERNOTFOUND : driver file could not be found
//         UCDISERVER_ERROR_BADFUNCADDR	   : function could not get retrieved
//         UCDISERVER_ERROR_BADFUNCADDR	   : function could not get retrieved
//         UCDISERVER_ERROR_INVALIDDRIVER  : driver is not valid
function UCDIServer_GetDriverInfo(pDriverFilename : PChar;
	                          pInfoBlock : PUCDIINFOBLOCK;
				  pDriverTitle : PChar) : WORD32;
                                    stdcall; external 'CryptPak.dll';

// loads a driver, gets the function addresses into a context
// @param pDriverName driver name
// @param pDrvCtxPtr pointer where to store the cryptdriver handle (pointer to
//                   allocated context)
// @param pRandGenFunc pointer to random generator function (may be UCDI_NULL,
//                     if so we an internal random generator is going to be
//                     used)
// @param pRandGenData general pointer for random generator function, e.g. to
//                     point to an object instance (ignored if pRandGenFunc is
//                     UCDI_NULL)
// @pRandSeed pointer to random seed data (only used for the internal random
//            generator, ignored if pRandGenFunc differs from UCDI_NULL, may
//            also be UCDI_NULL)
// @pRandSeedLen number of random seed bytes (ignored if pRandSeed is UCDI_NULL)
// @return error code:
//         UCDISERVER_ERROR_NOERROR            : success
//         UCDISERVER_ERROR_OUTOFMEMORY        : not enough memory
//         UCDISERVER_ERROR_ERROR              : loading error
//         UCDISERVER_ERROR_DRIVERNOTFOUND     : driver file could not be found
//         UCDISERVER_ERROR_BADFUNCADDR        : a function could not be
//                                               retrieved
//         UCDISERVER_ERROR_INVALIDDRIVER      : driver is not valid
//         UCDISERVER_ERROR_WRONGDRIVERVERSION : UCDI version is too high to
//                                               handle
function UCDIServer_CreateDriverContext(pDriverName : PChar;
	                                var pDrvCtxPtr : PCRYPTDRIVERCONTEXT;
					RandGenFunc : TUCDI_RandomGenerator;
					pRandGenData : Pointer;
                                        pRandSeed : Pointer;
                                        lRandSeedLen : WORD32) : WORD32;
                                          stdcall; external 'CryptPak.dll';

// unloads a driver; clears and frees the context
// @param pDrvCtx cryptdriver handle
// @return error code:
//         UCDISERVER_ERROR_NOERROR : success
//         UCDISERVER_ERROR_ERROR   : unloading (fatal) error
function UCDIServer_DestroyDriverContext (pDrvCtx : PCRYPTDRIVERCONTEXT)
                                           : WORD32;
                                           stdcall; external 'CryptPak.dll';

// executes the selftest function of an already
// loaded driver; this guarantees that the driver
// cannot be altered between selftest and usage
// @param pDrvCtx cryptdriver handle
// @param blExtendedTest flag for extended testing (testing with own data)
// @return error code:
//         UCDISERVER_ERROR_NOERROR       : success
//         UCDISERVER_ERROR_INVALIDDRIVER : driver is not valid
//         UCDISERVER_ERROR_OUTOFMEMORY   : not enough memory
function UCDIServer_ExecuteSelfTest(pDrvCtx : PCRYPTDRIVERCONTEXT;
	                            blExtendedTest : BYTEBOOL) : WORD32;
                                      stdcall; external 'CryptPak.dll';

// returns the information block of the driver
// @param pDrvCtx cryptdriver handle
// @param pInfoBlock pointer to block where to store the data
// @return error code:
//         UCDI_ERROR_NOERROR       : driver declares itself as valid
//         UCDI_ERROR_ERROR         : unknown error code (invalid driver)
//         UCDI_ERROR_INVALIDDRIVER : driver declares itself as invalid
function UCDIServer_GetInfoBlock(pDrvCtx : PCRYPTDRIVERCONTEXT;
	                         pInfoBlock : PUCDIINFOBLOCK) : WORD32;
                                  stdcall; external 'CryptPak.dll';

// creates a new session for encrypting or decrypting data
// @param wMode work mode; see UCDISERVER_MODE_xxx constants
// @param pKey pointer to key stored a byte buffer
// @param wKeyLen key size in bytes (must fit to the cryptdriver; if demanded)
// @param pDrvCtx cryptdriver handle
// @param pInitData pointer where to get/set the init. data (usually a CBC IV)
// @param pSessionHandlePtr pointer to the storage of the created cryptsession
//                          handle
// @return error code:
//         UCDISERVER_ERROR_NOERROR     : success
//         UCDISERVER_ERROR_ERROR       : key setup failed
//         UCDISERVER_ERROR_OUTOFMEMORY : not enough memory
//         UCDISERVER_ERROR_WEAKKEY     : weak key detected
function UCDIServer_OpenSession(wMode : WORD16;
                                pKey : Pointer;
                                wKeyLen : WORD16;
                                pDrvCtx : PCRYPTDRIVERCONTEXT;
                                pInitData : Pointer;
                                var pSessionHandlePtr : PCRYPTSESSIONHANDLE)
                                  : WORD32;
                                  stdcall; external 'CryptPak.dll';

// allows an interruption of the en-/decryption stream
// without a timeconsuming new key setup (mostly used
// to set a new CBC IV; if any; to allow chunking)
// @param pSessionHandle cryptsession handle
// @param pInitData pointer where to get/set the init. data (usually a CBC IV)
procedure UCDIServer_ResetSession (pSessionHandle : PCRYPTSESSIONHANDLE;
                                   pInitData : Pointer); stdcall;
                                     external 'CryptPak.dll';

// destroys a session handle
// @param pSessionHandle cryptsession handle
// @return error code:
//         UCDISERVER_ERROR_NOERROR : success
//         UCDISERVER_ERROR_ERROR   : driver couldn't stop properly (fatal
//                                    error)
function UCDIServer_CloseSession (pSessionHandle : PCRYPTSESSIONHANDLE)
                                    : WORD32;
                                    stdcall; external 'CryptPak.dll';

// encrypts a number of blocks
// @param pSessionHandle cryptsession handle
// @param pSource pointer to the source buffer (may be equal to the source
//                buffer)
// @param pTarget pointer to the target buffer
// @param lNumOfBlocks number of _blocks_ to encrypt
procedure UCDIServer_EncryptBlocks(pSessionHandle : PCRYPTSESSIONHANDLE;
				   pSource : Pointer;
                                   pTarget : Pointer;
                                   lNumOfBlocks : WORD32);
                                     stdcall; external 'CryptPak.dll';

// decrypts a number of blocks
// @param pSessionHandle cryptsession handle
// @param pSource pointer to the source buffer
// @param pTarget pointer to the target buffer (may be equal to the source
//                buffer)
// @param lNumOfBlocks number of _blocks_ to decrypt
// @param pPreviousBlock pointer to previous block (ignored if UCDI_NULL)
procedure UCDIServer_DecryptBlocks(pSessionHandle : PCRYPTSESSIONHANDLE;
				   pSource : Pointer;
                                   pTarget : Pointer;
                                   lNumOfBlocks : WORD32;
                                   pPreviousBlock : Pointer);
                                     stdcall; external 'CryptPak.dll';


// delivers random data from the current generator
// @param pDrvCtx cryptdriver handle
// @param pTarget pointer to the output buffer
// @param lNumOfBytes number of random bytes to deliver
procedure UCDIServer_GetRandomData(pDrvCtx : PCRYPTDRIVERCONTEXT;
                                   pTarget : Pointer;
                                   lNumOfBytes : WORD32);
                                     stdcall; external 'CryptPak.dll';


implementation

// nothing to implement here

end.
